﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nns/gfxLog.h>
#include <nns/nns_Log.h>
#include <nv/nv_MemoryManagement.h>
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>

#include "NnsLogCustomDrawGraphicsSystem.h"
#include "NnsLogCustomDrawShowing.h"

namespace {

nn::mem::StandardAllocator& GetStandardAllocator() NN_NOEXCEPT;

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
/**
* @brief グラフィックスの確保関数です。
*
* @param[in]    size         確保サイズ
* @param[in]    alignment    アライメント
* @param[in]    pUserData    ユーザーデータ
* @return 確保したメモリを返します。
* @details 確保関数です。
*/
void* Allocate(size_t size, size_t alignment, void* pUserData) NN_NOEXCEPT
{
    return GetStandardAllocator().Allocate(size, alignment);
}

/**
* @brief グラフィックスの解放関数です。
*
* @param[in]    addr         解放するメモリ
* @param[in]    pUserData    ユーザーデータ
* @details 解放関数です。
*/
void Free(void* addr, void* pUserData) NN_NOEXCEPT
{
    GetStandardAllocator().Free(addr);
}

/**
* @brief グラフィックスの再確保関数です。
*
* @param[in]    addr       既存のメモリ
* @param[in]    newSize    新しい確保サイズ
* @param[in]    pUserData  ユーザーデータ
* @return  再確保したメモリを返します。
* @details 再確保関数です。
*/
void* Reallocate(void* addr, size_t newSize, void* pUserData) NN_NOEXCEPT
{
    return GetStandardAllocator().Reallocate(addr, newSize);
}
#endif

/**
* @brief アロケーターのインスタンスを返す関数です。
*
* @return  アロケーターのインスタンスを返します。
* @details アロケーターのインスタンスを返す関数です。
*/
nn::mem::StandardAllocator& GetStandardAllocator() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(nn::mem::StandardAllocator, s_StandardAllocator);
    static bool s_IsInitialized = false;

    if (!s_IsInitialized)
    {
        const size_t MemorySize = 128 * 1024 * 1024;
        NN_ALIGNAS(4096) static nn::Bit8 s_Memory[MemorySize];

        s_StandardAllocator.Initialize(
            s_Memory, MemorySize
        );

        {
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
            // グラフィックスシステムのためのメモリ周りの初期化を行います。
            {
                const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
                nv::SetGraphicsAllocator(Allocate, Free, Reallocate, NULL);
                nv::InitializeGraphics(s_StandardAllocator.Allocate(GraphicsSystemMemorySize, 1),
                    GraphicsSystemMemorySize);
            }
            // グラフィックス開発者向けツールおよびデバッグレイヤのためのメモリアロケータを設定します。
            nv::SetGraphicsDevtoolsAllocator(Allocate, Free, Reallocate, NULL);
#endif
        }

        s_IsInitialized = true;
    }
    return s_StandardAllocator;
}
}

/**
* @brief コンストラクタです。
*
* @details コンストラクタです。
*/
Showing::Showing() NN_NOEXCEPT
{
    nn::os::InitializeEvent(&m_EventType, false, nn::os::EventClearMode_ManualClear);

    m_pGraphicsSystem = new GraphicsSystem();
    m_pGraphicsSystem->SetApplicationHeap(&GetStandardAllocator());
    m_pGraphicsSystem->Initialize();
    m_pGraphicsSystem->InitializeDebugFont();

    nns::gfxLog::SetConfiguration(
        &m_pGraphicsSystem->GetDevice(),
        &m_pGraphicsSystem->GetCommandBuffer(),
        &m_pGraphicsSystem->GetDebugFont());

    nns::gfxLog::SetDrawRect(
        0,
        0,
        static_cast<float>(m_pGraphicsSystem->GetDisplayWidth() / 2),
        static_cast<float>(m_pGraphicsSystem->GetDisplayHeight() / 2));

    m_pThreadStack = reinterpret_cast<char*>(
    GetStandardAllocator().Allocate(
        ThreadStackSize,
        nn::os::ThreadStackAlignment
        )
    );

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &m_Thread, ThreadFunc, this, m_pThreadStack,
        ThreadStackSize, nn::os::DefaultThreadPriority));

    // スレッドをStart
    nn::os::StartThread(&m_Thread);
}

/**
* @brief デストラクタです。
*
* @details デストラクタです。
*/
Showing::~Showing() NN_NOEXCEPT
{
    nn::os::SignalEvent(&m_EventType);
    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);

    GetStandardAllocator().Free(m_pThreadStack);

    m_pGraphicsSystem->FinalizeDebugFont();
    m_pGraphicsSystem->Finalize();
    delete m_pGraphicsSystem;
    m_pGraphicsSystem = NULL;
}

/**
* @brief 描画スレッドの実装関数です。
*
* @details 描画スレッドの実装関数です。
*/
void Showing::ThreadFuncImpl() NN_NOEXCEPT
{
    while (!nn::os::TryWaitEvent(&m_EventType))
    {
        m_pGraphicsSystem->BeginDraw();

        nns::gfxLog::WriteCommand();
        m_pGraphicsSystem->EndDraw();
    }
}

/**
* @brief スレッドのエントリーポイントです。
*
* @param[in]    pArg    このクラスの this ポインタ
* @details スレッドのエントリーポイントです。
*/
void Showing::ThreadFunc(void* pArg) NN_NOEXCEPT
{
    Showing * pShowing = reinterpret_cast<Showing*>(pArg);
    pShowing->ThreadFuncImpl();
}
