﻿param(
    [string[]]$Targets,
    [int]$TryCount = 100,
    [int]$Frames = 420,
    [int]$LinesPerFrame = 1,
    [int]$TimeoutSec = 20,
    [switch]$AlternateCrash,
    [switch]$DisableUILog
)

$ScriptDir         = Split-Path -Parent $MyInvocation.MyCommand.Path
$ScriptName        = $MyInvocation.MyCommand.Name
$DateTimeString    = Get-Date -Format "yyyyMMdd.HHmm"
$LogFilePrefix     = Resolve-Path "$ScriptDir\.."
$LogFilePrefix     = "$LogFilePrefix\Log-Simple-$DateTimeString"
$NintendoSdkRoot   = $env:NINTENDO_SDK_ROOT
$ToolRoot          = Resolve-Path "$NintendoSdkRoot\Tools\CommandLineTools"
$RunOnTarget       = Resolve-Path "$ToolRoot\RunOnTarget.exe"
$ControlTarget     = Resolve-Path "$ToolRoot\ControlTarget.exe"
$AppMessage        = "hello, world"
$AppSuccessMessage = "Execution finished"

foreach($target in $Targets)
{
    New-Variable -Name "AppMessageCount_$target" -Value @() -Scope Script
}

function Get-AppPath
{
    $result = ""
    # sdk git ツリー内かどうかを、Samples/NintendoSdkSampleRootMark の有無で判断する
    if (Test-Path "$ScriptDir\..\..\..\..\NintendoSdkSampleRootMark")
    {
        $result = Join-Path (Resolve-Path "$ScriptDir\..\..\..\..\") "Outputs\NX-NXFP2-a64\Applications\testMultipleDevkits_Simple\Develop\testMultipleDevkits_Simple.nspd"
    }
    else
    {
        $result = Join-Path (Resolve-Path "$ScriptDir\..\TargetApp\Simple") "Binaries\NX64\NX_Develop\testMultipleDevkits_Simple.nspd"
    }

    if (!(Test-Path $result))
    {
        throw [System.IO.FileNotFoundException] "[ERROR] Cannot find $result. Please build this."
    }
    return $result
}
$App = Get-AppPath

function Get-TargetManager
{
    $result = ""
    if (Test-Path "$ScriptDir\..\..\..\..\NintendoSdkSampleRootMark")
    {
        $result = Resolve-Path "${NintendoSdkRoot}\Externals\Oasis\bin\NintendoTargetManager.exe"
    }
    else
    {
        $result = Resolve-Path "${env:programfiles}\Nintendo\Oasis\bin\NintendoTargetManager.exe"
    }

    if (!(Test-Path $result))
    {
        throw [System.IO.FileNotFoundException] "[ERROR] Cannot find $result. Please install Nintendo Target Manager."
    }
    return $result
}
$TargetManager = Get-TargetManager

function Summarize-Setting
{
    "`nTarget devkits are ..."
    foreach ($target in $Targets) {
        "  $target"
    }

    "`nLog files are ..."
    foreach ($target in $Targets) {
        "  $LogFilePrefix@$target.txt"
    }
    
    "`nTools and app are ..."
    "  (NINTENDO_SDK_ROOT) $env:NINTENDO_SDK_ROOT"
    "  $RunOnTarget"
    "  $ControlTarget"
    "  $App"
}

function Register-Target
{
    "`n### Register-Target"
    if($DisableUILog)
    {
        "  Starting Target Manager with UI logging disabled."
        Stop-Process -Name "NintendoTargetManager" -Force -ErrorAction SilentlyContinue
        Start-Process -FilePath $TargetManager -ArgumentList "-disablelogging"
    }

    "  ControlTarget unregister-all"
    Invoke-Expression "$ControlTarget unregister-all"
    if($LastExitCode -ne 0)
    {
        throw "ControlTarget failed to unregister targets."
    }

    foreach ($target in $Targets) {
        "  ControlTarget register -t $target"
        Invoke-Expression "$ControlTarget register -t $target"
        if($LastExitCode -ne 0)
        {
            throw "ControlTarget failed to register target."
        }
        "  ControlTarget connect -t $target"
        Invoke-Expression "$ControlTarget connect -t $target"
        if($LastExitCode -ne 0)
        {
            throw "ControlTarget failed to connect to target."
        }
    }
    "  Register finish (sleep 5 seconds)"
    Start-Sleep -s 5
}

function Terminate-AllTarget
{
    "`n### Terminate-AllTarget"
    
    # Terminate each target in serial
    $failed = $false
    foreach ($target in $Targets)
    {
        Start-Job -Name "$ScriptName-Terminate-$target" -ScriptBlock {
            param($executer, $name)
            Invoke-Expression "$executer terminate --target $name"
            $LastExitCode
        } -ArgumentList $ControlTarget, $target | Out-Null
        
        # Wait for job to finish
        $job = Get-Job -Name "$ScriptName-Terminate-$target"
        Wait-Job -Job $job -Timeout 10 | Out-Null
        Stop-Job -Job $job

        # Check exit code of ControlTarget terminate
        $jobOutput = Receive-Job -Job $job
        if($jobOutput -eq $null)
        {
            Write-Warning "${target}: ControlTarget terminate failed to complete within timeout."
            $failed = $true
        }
        else
        {
            $exitCode = $jobOutput[-1]
            if($exitCode -ne 0)
            {
                Write-Warning "${target}: ControlTarget terminate returned $exitCode."
                $failed = $true
            }
            else
            {
                "  ${target}: ControlTarget terminated application successfully."
            }
        }
        
        Remove-Job -Job $job
    }

    if($failed)
    {
        throw "Failed to terminate at least one SDEV. Stopping execution."
    }
    
    "  Terminate finish (sleep 5 seconds)"
    Start-Sleep -s 5
}

function Run-Application(
    [switch]$ForceCrash
)
{
    "`n### Run-Application"
    foreach ($target in $Targets)
    {
        Start-Job -Name "$ScriptName-Run-$target" -ScriptBlock {
            param($executer, $app, $name, $lines, $frames, $crash)
            if($crash)
            {
                & $executer $app --target $name -- --target $name --forcecrash 2>&1
            }
            else
            {
                & $executer $app --target $name -- --target $name --linesperframe $lines --frames $frames --pattern-success-exit "$AppSuccessMessage" 2>&1
            }
        } -ArgumentList $RunOnTarget, $App, $target, $LinesPerFrame, $Frames, $ForceCrash | Out-Null
        "  Execution started on $target"
    }

    Get-Job -Name "$ScriptName-Run-*" | Wait-Job -Timeout $TimeoutSec | Out-Null
    Get-Job -Name "$ScriptName-Run-*" | Stop-Job

    foreach($target in $Targets)
    {
        $job = Get-Job -Name "$ScriptName-Run-$target"
        $jobOutput = Receive-Job -Job $job
        
        # Append job output to log
        $logFileName = "$LogFilePrefix@$target.txt"
        $jobOutput | Out-File -filepath $logFileName -Encoding UTF8 -Append -Width 1000
        
        # Get job output and parse it for the app's looping message
        $matchCount = ($jobOutput | Select-String $AppMessage -AllMatches).Matches.Count
        $resultsArray = Get-Variable -Name "AppMessageCount_$target" -ValueOnly
        Set-Variable -Name "AppMessageCount_$target" -Value ($resultsArray += @($matchCount)) -Scope Script
        "  ${target}: `"$AppMessage`" count: $matchCount"
        
        Remove-Job -Job $job
    }
}

function Test-Results()
{
    "`n`n### Test-Results (Count of `"$AppMessage`" per run)"
    
    $expectedCount = $Frames * $LinesPerFrame
    $failed = $false
    
    foreach($target in $Targets)
    {
        "`n"
        $resultsArray = Get-Variable -Name "AppMessageCount_$target" -ValueOnly
        for($i = 0; $i -lt $TryCount; $i++)
        {
            "  ${target}[$i] = $($resultsArray[$i])"
            if($AlternateCrash -and ($i % 2 -eq 0))
            {
                if($resultsArray[$i] -ne 0)
                {
                    Write-Warning "Expected a forced crash, but application did not crash."
                    $failed = $true
                }
            }
            elseif($resultsArray[$i] -ne $expectedCount)
            {
                Write-Warning "Count $($resultsArray[$i]) did not match expected ($expectedCount)."
                $failed = $true
            }
        }
    }
    
    if($failed)
    {
        throw "Test failed. Please check the test results for more details."
    }
}

try
{
    Summarize-Setting
    Register-Target
    for ($i = 0; $i -lt $TryCount; $i++)
    {
        "`n`n------------------ $i ------------------"
        $crash = $AlternateCrash -and ($i % 2 -eq 0)
        Run-Application -ForceCrash:$crash
        Terminate-AllTarget
    }

    # We can only judge results if frames was greater than 0
    # Otherwise, the application continues to print endlessly
    if($Frames -gt 0)
    {
        Test-Results
    }
}
finally
{
    "`n`n### Cleanup: Removing all jobs associated with $ScriptName"
    Remove-Job -Name "$ScriptName-*" -Force
}
