﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/fs.h>

#include <nns/mm.h>

#include <nn/nn_Result.h>
#include <movie/Utils.h>
#include <movie/Extractor.h>
#include <movie/Decoder.h>

#include "ByteStreamReader.h"

void AudioInputReadThreadFunction(void *arg);
void VideoInputReadThreadFunction(void *arg);
void AudioVideoWorkerThreadFunction(void *arg);

class MovieDecoderPlayer
{
public:

    MovieDecoderPlayer();

    movie::Status Initialize(uint64_t coreMask);

    void Finalize();

    movie::Status CreateExtractor(const char* uri,
        movie::ContainerType container,
        movie::Extractor** extractor,
        movie::ClientStreamReader** clientReader);

    movie::Status CreateDecoders(movie::Extractor* extractor,
        movie::DecoderMode audioDecoderMode,
        movie::DecoderOutputFormat audioDecoderOutputFormat,
        movie::DecoderMode videoDecoderMode,
        movie::DecoderOutputFormat videoDecoderOutputFormat);

    movie::Status CreateDecoder(movie::Decoder** decoder,
        movie::DecoderType decoderType,
        movie::DecoderMode decoderMode,
        movie::DecoderOutputFormat decoderOutputFormat);

    movie::Status StartDecoder(movie::Decoder* decoder);

    movie::Status StopDecoder(movie::Decoder* decoder);

    movie::Status FlushDecoder(movie::Decoder* decoder);

    void DestroyDecoder(movie::Decoder* decoder);

    movie::Status CreateAudioVideoRenderer();

    void StopAudioVideoRenderer();

    void DestroypAudioVideoRenderer();

    void DestroyExtractor(movie::Extractor* extractor);

    void SetAudioDecodeComplete() { m_AudioDecodeDone = true; }

    void ResetAudioDecodeComplete() { m_AudioDecodeDone = false; }

    bool IsAudioDecodeComplete() { return m_AudioDecodeDone; }

    void SetVideoDecodeComplete() { m_VideoDecodeDone = true; }

    void ResetVideoDecodeComplete() { m_VideoDecodeDone = false; }

    bool IsVideoDecodeComplete() { return m_VideoDecodeDone; }

    bool IsVideoDecoderCreated() { return m_VideoDecoderCreated; }

    bool IsAudioDecoderCreated() { return m_AudioDecoderCreated; }

    void ResetPlaybackComplete();

    bool IsPlaybackComplete();

    void SetPlaybackComplete() { m_PlaybackComplete = true; }

    bool StartPlayback();

    movie::Status ReadInputDataFromExtractorSendTodecoder(int index, movie::Decoder* decoder);

    movie::Status ReadInputDataFromExtractorSendTodecoder(ByteStreamReader* byteStreamReader,
        int index,
        movie::Decoder* decoder);

    movie::Status ConfigureDecoder(movie::Decoder* decoder,
        movie::DecoderEvents* decoderEvents,
        movie::MediaData* configuration);

    movie::Status SendEosToDecoder();

    movie::Status CheckForInputBuffersAndReadVideoInputData();

    movie::Status CheckForInputBuffersAndReadAudioInputData();

    movie::Status AddBufferToAudioIndexList(int32_t bufferIndex);

    movie::Status GetAudioIndexListSize(int32_t* indexListSize);

    movie::Status GetVideoIndexListSize(int32_t* indexListSize);

    movie::Status RemoveBufferFromAudioIndexList(int32_t* bufferIndex);

    movie::Status AddBufferToVideoIndexList(int32_t bufferIndex);

    movie::Status RemoveBufferFromVideoIndexList(int32_t* bufferIndex);

    movie::Status VideoInputBufferAvailableEvent();

    movie::Status VideoOutputBufferAvailableEvent();

    movie::Status VideoFormatChangedEvent();

    movie::Status AudioInputBufferAvailableEvent();

    movie::Status AudioOutputBufferAvailableEvent();

    movie::Status AudioFormatChangedEvent();

    movie::Status SignalVideoInputBufferAvailable();

    movie::Status SignalAudioInputBufferAvailable();

    movie::Status SignalVideoInputReadThreadExit();

    movie::Status SignalAudioInputReadThreadExit();

    movie::Status SignalAudioVideoWorkerThreadExit();

    movie::Status CreateThreadForVideoInputReader();

    movie::Status CreateThreadForAudioVideoWorker();

    movie::Status CreateThreadForAudioInputReader();

    void StartAudioInputReaderThread();

    void StartVideoInputReaderThread();

    void StartAudioVideoWorkerThread();

    void StopAndDestroyAudioInputReaderThread();

    void StopAndDestroyVideoInputReaderThread();

    void StopAndDestroyAudioVideoWorkerThread();

    friend void VideoInputReadThreadFunction(void *arg);

    friend void AudioVideoWorkerThreadFunction(void *arg);

    friend void AudioInputReadThreadFunction(void *arg);

    void Reset();

    movie::Status HandleLoopBack();

    bool IsAudioDecoder(movie::DecoderType decoderType);

    bool IsVideoDecoder(movie::DecoderType decoderType);

    void WaitForFormatChange();

    void CheckAndSignalFormatChangeWaiter();

    void HandleErrorExit();

    ~MovieDecoderPlayer() {};

public:
    movie::Extractor* m_Extractor;
    movie::Decoder* m_AudioDecoder;
    movie::Decoder* m_VideoDecoder;

    struct DecoderEventTypes
    {
        nn::os::EventType inputBufferAvailableEvent;
        nn::os::EventType outputBufferAvailableEvent;
        nn::os::EventType formatChangedEvent;
        nn::os::EventType errorEvent;
    };

    struct DecoderMultiWaitHolderTypes
    {
        nn::os::MultiWaitHolderType inputBufferAvailableEventHolder;
        nn::os::MultiWaitHolderType outputBufferAvailableEventHolder;
        nn::os::MultiWaitHolderType formatChangedEventHolder;
        nn::os::MultiWaitHolderType errorEventHolder;
    };

    nn::os::EventType m_AudioVideoWorkerThreadExitEvent;
    nn::os::MultiWaitHolderType m_AudioVideoWorkerThreadExitEventHolder;

    nn::os::MultiWaitType m_AudioVideoWorkerMultiWait;

    movie::DecoderEvents m_AudioDecoderEvents;
    movie::DecoderEvents m_VideoDecoderEvents;

    DecoderEventTypes m_AudioDecoderEventTypes;
    DecoderEventTypes m_VideoDecoderEventTypes;

    DecoderMultiWaitHolderTypes m_AudioDecoderMultiWaitHolderTypes;
    DecoderMultiWaitHolderTypes m_VideoDecoderMultiWaitHolderTypes;

    MovieAudioOutputHandler *m_MovieAudioOutputHandler;
    MovieVideoOutputHandler *m_MovieVideoOutputHandler;

    int32_t m_Tracks;
    int32_t m_Width;
    int32_t m_Height;
    int32_t m_NumChannels;
    int32_t m_SampleRate;
    double m_VideoFrameRate;
    int64_t m_TrackDurationUs;

    int32_t m_AudioTrackIndex;
    int32_t m_VideoTrackIndex;

    size_t m_ThreadStackSize;

    void* m_VideoInputReadThreadStack;
    bool m_VideoInputReadThreadDone;
    bool m_VideoInputReadThreadCreated;
    bool m_VideoInputReadThreadStarted;
    nn::os::ThreadType m_VideoInputReadThreadType;

    void* m_AudioInputReadThreadStack;
    bool m_AudioInputReadThreadCreated;
    bool m_AudioInputReadThreadStarted;
    nn::os::ThreadType m_AudioInputReadThreadType;

    void* m_AudioVideoWorkerThreadStack;
    bool m_AudioVideoWorkerThreadCreated;
    bool m_AudioVideoWorkerThreadStarted;
    nn::os::ThreadType m_AudioVideoWorkerThreadType;

    std::vector<int> m_AudioBufferIndexList;
    std::vector<int> m_VideoBufferIndexList;

    nn::os::MutexType m_AudioListMutex;
    nn::os::MutexType m_VideoListMutex;

    size_t m_BufferSize;
    uintptr_t *m_VideoInputMessageBuffer;

    nn::os::MessageQueueType m_VideoInputMessageQueue;

    uintptr_t *m_AudioInputMessageBuffer;
    nn::os::MessageQueueType m_AudioInputMessageQueue;

    enum MoviePlayerMessage
    {
        MoviePlayerMessage_VideoInputAvailable  = 1,
        MoviePlayerMessage_VideoInputThreadExit = 2,
        MoviePlayerMessage_AudioInputAvailable  = 3,
        MoviePlayerMessage_AudioInputThreadExit = 4,
    };

    ByteStreamReader* m_ByteStreamReader;
    movie::DecoderType m_AudioDecoderType;
    movie::DecoderType m_VideoDecoderType;
    bool m_AudioDecoderCreated;
    bool m_VideoDecoderCreated;
    bool m_LoopPlayback;
    int32_t m_LoopCount;
    std::vector<movie::MediaData> m_TrackFormat;
    movie::ClientStreamReader* m_ClientReader;
    MediaClock* m_MediaClock;
    bool m_PlaybackComplete;
    int64_t m_PlaybackTime;
    int64_t m_VideoPresentationTimeUs;
    bool m_VariableSizeAudioFrames;
    nn::os::EventType m_FormatChangedPlayerEvent;
    bool m_AudioFormatChangeEventReceived;
    bool m_VideoFormatChangeEventReceived;
    bool m_SdcardMounted;

private:
     bool m_AudioDecodeDone;
     bool m_VideoDecodeDone;

     int32_t m_AudioOutputBufferReceived;
     int32_t m_VideoOutputBufferReceived;
     const int32_t m_AudioOutputBufferCountToStartPlayback = 6;
     const int32_t m_VideoOutputBufferCountToStartPlayback = 4;
     movie::DecoderMode m_VideoDecoderMode;
     movie::DecoderMode m_AudioDecoderMode;
     movie::DecoderOutputFormat m_VideoDecoderOutputFormat;
     movie::DecoderOutputFormat m_AudioDecoderOutputFormat;
};
