﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
/**
 * @file mm_NvnVideoRenderer.h
 * @brief Render video data using NVN
 */
#include <nn/vi.h>
#include <nvn/nvn.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <movie/Common.h>
#include <movie/Status.h>
#include <movie/Utils.h>
#include <movie/Extractor.h>
#include <movie/Decoder.h>
#include <map>
#include <nns/mm/mm_MoviePlayerUtils.h>
unsigned Align(unsigned value, unsigned alignment);

struct DataSection
{
    const void *data;
    size_t size;
};

/**
 * @brief
 * GlslCompiler class for generating shader data.
 *
 */
class GlslCompiler
{
public:
    GlslCompiler();
    ~GlslCompiler();
    bool Init();
    void Destroy();
    bool CompileShader(const char *src, NVNshaderStage stage);

    //the pointers returned in DataSection are invalid after new compilations
    DataSection GetOutput() const;

private:
    GLSLCcompileObject m_Compiler;
    DataSection m_OutputData;
    bool m_Init;
};

/**
 * @brief
 * NvnShaderSerialize helper class for NVN rendering
 *
 */
class NvnShaderSerialize
{
public:
    NvnShaderSerialize();
    ~NvnShaderSerialize();
    void LoadNvnShaderMemory(const uint8_t *shaderBinary);

    DataSection GetDataSection();
    DataSection GetControlSection();
    unsigned GetScratchSize();

private:
    void *m_ShaderData;
    DataSection m_DataSection;
    DataSection m_ControlSection;
    unsigned m_ScratchSize;
};

/**
 * @brief
 * NvnMemoryPool helper class for NVN rendering
 *
 */
class NvnMemoryPool
{
public:
    NvnMemoryPool();
    ~NvnMemoryPool();
    bool Init(NVNdevice &device, int32_t flags, unsigned size, void *pMemory);
    void Destroy();
    ptrdiff_t GetNewMemoryChunkOffset(unsigned size, unsigned alignment);
    NVNmemoryPool &GetMemoryPool();

private:
    NVNmemoryPool m_MemoryPool;
    unsigned m_Size;
    void *m_pMemory;
    unsigned m_Offset;
};

/**
 * @brief
 * NvnBuffer helper class for NVN rendering
 *
 */
class NvnBuffer
{
public:
    NvnBuffer();
    ~NvnBuffer();
    bool Init(NVNdevice &device, NVNmemoryPool &pPool, unsigned offset, unsigned size);
    void Destroy();
    NVNbufferAddress GetAddress() const;
    NVNbuffer &GetBuffer();
    void *Map();

private:
    NVNbuffer m_Buffer;
    NVNdevice *m_pDevice;
    void *m_pMappedData;
};

/**
 * @brief
 * NvnCommandBuffer helper class for NVN rendering
 *
 */
class NvnCommandBuffer
{
public:
    NvnCommandBuffer();
    ~NvnCommandBuffer();
    bool Init(class NvnDevice &device, nn::mem::StandardAllocator &allocator, NvnMemoryPool &memoryPool);
    void Destroy();
    void Reset();
    size_t GetCommandMemoryUsed() const;
    size_t GetControlMemoryUsed() const;
    NVNcommandBuffer &GetCommandBuffer();

private:
    NVNcommandBuffer m_CommandBuffer;
    NvnDevice *m_pDevice;
    nn::mem::StandardAllocator *m_pAllocator;
    NvnMemoryPool *m_pMemoryPool;
    ptrdiff_t m_CommandOffset;
    void *m_pControlPool;

    static const size_t m_CommandSize = 2048;
    static const size_t m_ControlSize = 1024;
};

/**
 * @brief
 * NvnDevice helper class for NVN rendering
 *
 */
class NvnDevice
{
public:
    NvnDevice();
    ~NvnDevice();
    bool Init(NVNnativeWindow nativeWindow, nn::mem::StandardAllocator &allocator);
    void Destroy();
    void ResizeRenderTargets(unsigned width, unsigned height);
    NVNdevice &GetDevice();
    NVNwindow &GetWindow();
    NVNtexture &GetRenderTarget(int index);
    int GetDeviceInfo(NVNdeviceInfo name);

private:
    static void DebugLayerCallback(
        NVNdebugCallbackSource source,
        NVNdebugCallbackType type,
        int id,
        NVNdebugCallbackSeverity severity,
        const char * message,
        void* pUser);

    NVNdevice m_Device;

    nn::mem::StandardAllocator *m_pAllocator;

    NVNtextureBuilder m_RenderTargetBuilder;
    NVNtexture m_RenderTargets[2];

    NvnMemoryPool m_RenderTargetMemoryPool;
    void *m_pRenderTargetMemory;

    NVNwindow m_pWindow;
    NVNwindowBuilder m_WindowBuilder;
    size_t m_ColorTargetSize;

    bool m_FirstResize;

    static const int m_NumColorBuffers = 2;
};

/**
 * @brief
 * NvnProgram helper class for NVN rendering
 *
 */
class NvnProgram
{
public:
    NvnProgram();
    ~NvnProgram();
    bool Init(NVNdevice &device, const NVNshaderData *stageData, unsigned stageCount);
    void Destroy();
    NVNprogram &GetProgram();

private:
    NVNprogram m_Program;
    NVNdevice *m_pDevice;
};

/**
 * @brief
 * NvnQueue helper class for NVN rendering
 *
 */
class NvnQueue
{
public:
    NvnQueue();
    ~NvnQueue();
    bool Init(NVNdevice &device);
    void Destroy();
    void WaitForAllCommands();
    NVNqueue &GetQueue();
    NVNsync &GetNvnSync();

private:
    NVNqueue m_Queue;
    NVNsync m_CommandBufferSync;
    NVNdevice *m_pDevice;
};

/**
 * @brief
 * NvnSampler helper class for NVN rendering
 *
 */
class NvnSampler
{
public:
    NvnSampler();
    ~NvnSampler();
    bool Init(NVNdevice &device);
    void Destroy();
    NVNsampler &GetSampler();

private:
    NVNsampler m_Sampler;
    NVNdevice *m_pDevice;
};

/**
 * @brief
 * NvnTexture helper class for NVN rendering
 *
 */
class NvnTexture
{
public:
    NvnTexture();
    ~NvnTexture();
    bool Init(NVNdevice &device,
        NvnMemoryPool &memoryPool,
        unsigned width,
        unsigned height,
        NVNformat format,
        int32_t flags,
        unsigned stride,
        bool nvnVideoInteropMode = false);
    void Destroy();
    NVNtexture &GetTexture();

private:
    NVNtexture m_Texture;
    NVNdevice *m_pDevice;
};

/**
 * @brief
 * NvnTextureSamplerPool helper class for NVN rendering
 *
 */
class NvnTextureSamplerPool
{
public:
    NvnTextureSamplerPool();
    ~NvnTextureSamplerPool();
    bool Init(NVNdevice &device, nn::mem::StandardAllocator &allocator);
    void Destroy();
    NVNtexturePool &GetTexturePool();
    NVNsamplerPool &GetSamplerPool();
    unsigned Register(NVNtexture &pTexture, NVNsampler &pSampler);

private:
    NVNtexturePool m_TexturePool;
    NVNsamplerPool m_SamplerPool;
    NvnMemoryPool m_MemoryPool;
    void *m_pMemory;
    nn::mem::StandardAllocator *m_pAllocator;
    NVNdevice *m_pDevice;
    unsigned m_Id;
};

/**
 * @brief
 * NvnWindow helper class for NVN rendering
 *
 */
class NvnWindow
{
public:
    NvnWindow();
    ~NvnWindow();
    bool Init(nn::vi::NativeWindowHandle nativeWindow);
    void Destroy();
    nn::vi::NativeWindowHandle GetNativeWindow();
    int GetWidth() const;
    int GetHeight() const;

private:
    void InitMemory();
    bool InitWindow();

    nn::vi::NativeWindowHandle  m_NativeWindow;
    nn::vi::Display*            m_pDisplay;
    nn::vi::Layer*              m_pLayer;
    int                         m_WindowWidth;
    int                         m_WindowHeight;
    bool m_NativeWindowCreated;
};

/**
 * @brief
 * NvnVideoRenderer class for renderig of video using NVN
 *
 */
class NvnVideoRenderer
{
public:

   /**
    * @brief
    * Enumeration of Output formats used by NVN renderer.
    *
    */
    enum NvnOutputFormat
    {
        NvnOutputFormat_YuvNv12NvnBuffer  = 0,  //!< Video output format - NV12.
        NvnOutputFormat_AbgrNvnTexture    = 1,  //!< Video output format - ABGR (8:8:8:8 bit).
                                                //!< NVN texture direct access by decoder.
    };

   /**
    * @brief NvnVideoRenderer constructor.
    *
    * @param[in] nativeWindow                   VI NativeWindowHandle to use.
    * @param[in] nvnOutputFormat                NvnOutputFormat, one of the following:
    *                                           NvnOutputFormat_YuvNv12NvnBuffer or NvnOutputFormat_AbgrNvnTexture.
    */
    NN_IMPLICIT NvnVideoRenderer(nn::vi::NativeWindowHandle nativeWindow,
        NvnOutputFormat nvnOutputFormat = NvnOutputFormat_YuvNv12NvnBuffer) NN_IMPLICIT;

   /**
    * @brief Initialize NvnVideoRenderer.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError             Generic error.
    * @retval ::Status_OutOfMemory              Memory allocation failed.
    * @retval ::Status_NotSupported             Unsupported NvnOutputFormat.
    *
    */
    movie::Status Initialize(int numberOfVideosToRender);

   /**
    * @brief CreateVideoBufferAndTextures.
    *
    * @param[in] decoder                        Video decoder.
    * @param[in] width                          Video width (Number of distinct vertical lines).
    * @param[in] height                         Video height (Number of distinct horizontal lines).
    *
    * @return None
    */
    void CreateVideoBufferAndTextures(movie::Decoder* decoder, int width, int height);

   /**
    * @brief RegisterDecoder.
    *
    * @param[in] decoder                        Video decoder.
    * @param[in] width                          Video width (Number of distinct vertical lines).
    * @param[in] height                         Video height (Number of distinct horizontal lines).
    * @param[in] nvnOutputFormat                NvnOutputFormat, one of the following:
    *                                           NvnOutputFormat_YuvNv12NvnBuffer or NvnOutputFormat_AbgrNvnTexture.
    * @return ::movie::Status
    * @retval ::Status_Success
    */
    movie::Status RegisterDecoder(movie::Decoder* decoder, int width, int height, NvnOutputFormat nvnOutputFormat);

   /**
    * @brief GetNvnVideoBuffer.
    *
    * @param[in] decoder                        Video decoder.
    * @param[in] allocIndex                     Video buffer index.
    * @param[out] buffer                        Video buffer data pointer.
    * @param[out] bufferSize                    Video buffer size in bytes.
    *
    * @return None
    */
    void GetNvnVideoBuffer(movie::Decoder* decoder, int *allocIndex, void **buffer, size_t *bufferSize);

   /**
    * @brief GetNvnVideoTexture.
    *
    * @param[in] allocIndex                     Texture index.
    * @param[out] textureHandle                 NVN texture handle.
    * @param[out] size                          NVN texture size.
    *
    * @return None
    *
    * @details
    * Returns NVN texture handle, which can be used by video decoder. This API can be used only if
    * output format is set to NvnOutputFormat_AbgrNvnTexture.
    *
    */
    void GetNvnVideoTexture(movie::Decoder* decoder, int *allocIndex, void **textureHandle, size_t *size);

    /**
    * @brief GetNvnSyncInfo.
    *
    * @param[out] device                        NVN device handle.
    * @param[out] sync                          NVN sync handle.
    *
    * @return None
    *
    * @details
    * Returns NVN device & sync handle, which can be used by video decoder for synchronization between modules.
    *
    */
    void GetNvnSyncInfo(void **device, void **sync);

   /**
    * @brief Draw.
    *
    * @param[out] timeToRenderUs                Time spent rendering the frame
    *
    * @return true if a frame is rendered, false otherwise
    *
    * @details
    * Draw current video frame using NVN.
    *
    */
    bool Draw(int64_t *timeToRenderUs);

   /**
    * @brief AddVideoFrameToRender.
    *
    * @param[in] decoder                        Video decoder.
    * @param[in] buffer                         Video buffer data pointer.
    * @param[in] index                          Buffer index, used to identify backing NVNBuffer.
    * @param[in] width                          Video width (Number of distinct vertical lines).
    * @param[in] height                         Video height (Number of distinct horizontal lines).
    * @param[in] yOffset                        Offset for Y-data from start of the output buffer address, when output format is NV12
    * @param[in] yStride                        Stride for Y-rows, when output format is NV12.
    * @param[in] uvOffset                       Offset for UV-data from start of the output buffer address, when output format is NV12.
    * @param[in] colorSpace                     Color space for UV-data, when output format is NV12.
    *
    * @return None
    *
    * @details
    * Add video buffer to render.
    *
    */
    void AddVideoFrameToRender(movie::Decoder* decoder, void *buffer, int index, int width, int height, int yOffset, int yStride, int uvOffset, int colorSpace, size_t bufferSize);

    /**
    * @brief AddVideoFrameToRender.
    *
    * @param[in] decoder                        Video decoder.
    * @param[in] bufferMemory                   Video buffer data pointer.
    * @param[in] index                          Buffer index, used to identify backing NVNBuffer.
    * @param[in] width                          Video width (Number of distinct vertical lines).
    * @param[in] height                         Video height (Number of distinct horizontal lines).
    *
    * @return None
    *
    * @details
    * Add texture to render.
    *
    */
    void AddTextureToRender(movie::Decoder* decoder, void *textureHandle, int allocIndex, int width, int height);

   /**
    * @brief Finalize NvnVideoRenderer.
    *
    * @return None
    *
    * @details
    * All resources are freed.
    *
    */
    void Finalize();

   /**
    * @brief CanStartRendering.
    *
    * @return true if sufficient output video buffers available to start rendering.
    *
    */
    bool CanStartRendering();

   /**
    * @brief ReturnBufferToRenderer. A decoder may not be able to get a decoded frame
    * every time. When decoder buffer not available return the buffer back to renderer
    *
    * @return none.
    */
    void ReturnBufferToRenderer(movie::Decoder* decoder, void *handle, int allocIndex);

private:
    nn::mem::StandardAllocator m_allocator;
    void *m_pAllocatorMemory;

    NvnWindow m_window;
    NvnDevice m_device;
    NvnQueue m_queue;

    void *m_pShaderPoolMemory;
    void *m_pVertexPoolMemory;
    void *m_pCommandMemory;
    void *m_pShaderScratchMemory;
    NvnMemoryPool m_shaderPool;
    NvnMemoryPool m_vertexPool;
    NvnMemoryPool m_commandPool;
    NvnMemoryPool m_shaderScratchPool;
    NvnMemoryPool m_uniformPool;

    NvnProgram m_program;
    NvnShaderSerialize m_vertexShaderFile;
    NvnShaderSerialize m_fragmentShaderFile;

    NvnCommandBuffer m_renderTargetCommand;

    NvnTextureSamplerPool m_texturePool;
    NvnSampler m_videoSampler;

    NvnBuffer m_vertexDataBuffer;
    NvnBuffer m_fragmentDataBuffer;
    NvnBuffer m_squareMesh;
    NvnBuffer* m_squareMeshes;

    NVNblendState m_blendState;
    NVNchannelMaskState m_channelMaskState;
    NVNcolorState m_colorState;
    NVNmultisampleState m_multisampleState;
    NVNpolygonState m_polygonState;
    NVNdepthStencilState m_depthStencilState;

    DataSection m_vertexDataSection;
    DataSection m_vertexControlSection;
    DataSection m_fragmentDataSection;
    DataSection m_fragmentControlSection;
    nn::vi::NativeWindowHandle m_NativeWindow;
    int m_SizeOfVertices;
    NvnOutputFormat m_NvnOutputFormat;

    struct NvnFrameContext
    {
        NvnFrameContext(movie::Decoder* _decoder, void *_handle, int _index, int _width, int _height, int _yOffset, int _yStride, int _uvOffset, int _colorSpace, size_t _bufferSize)
            : decoder(_decoder),
            handle(_handle),
              index(_index),
              width(_width),
              height(_height),
              yOffset(_yOffset),
              yStride(_yStride),
              uvOffset(_uvOffset),
              colorSpace(_colorSpace),
              yuvBufferSize (_bufferSize){}
        movie::Decoder* decoder;
        void *handle;
        int index;
        int width;
        int height;
        int yOffset;
        int yStride;
        int uvOffset;
        int colorSpace;
        size_t yuvBufferSize;
    };
    struct NvnVideoFrameContext
    {
        NvnVideoFrameContext() {}
        explicit NvnVideoFrameContext(movie::Decoder* videoDecoder)
            : decoder(videoDecoder) {}
        NvnTexture videoTextureY;
        NvnTexture videoTextureUV;
        NVNtextureHandle videoTextureHandleY;
        NVNtextureHandle videoTextureHandleUV;
        unsigned int videoTextureIdY;
        unsigned int videoTextureIdUV;
        NvnBuffer videoBufferYUV;
        void *pTextureMemory;
        NvnMemoryPool textureMemoryPool;
        void *pBufferMemory;
        NvnMemoryPool bufferMemory;
        movie::Decoder* decoder;
        void *handle;
        size_t yuvBufferSize;
        int index;
        NvnTexture rgbTexture;
        NVNtextureHandle rgbTextureHandle;
        unsigned rgbTextureId;
    };

    struct NvnRenderContext
    {
        NvnRenderContext(movie::Decoder* videoDecoder, int videoWidth, int videoHeight, int videoNumber)
            : decoder(videoDecoder),
            width(videoWidth),
            height(videoHeight),
            videoId(videoNumber)
        {
            pUniformMemory = nullptr;
        }
        movie::Decoder* decoder;
        void *pUniformMemory;
        NvnMemoryPool uniformPool;
        NvnBuffer uniformBuffer;
        unsigned uniformUsefulSize;
        int width;
        int height;
        int videoId;
        std::vector<NvnVideoFrameContext*> m_NvnVideoFrameList;
        std::vector<NvnFrameContext> m_AllocList;
        std::vector<NvnFrameContext> m_RenderList;
        std::vector<NvnFrameContext> m_DecoderList;
    };

    void GetNvnTextureInfo(movie::Decoder* decoder, int index, NvnTexture** rgbTexture, NVNtextureHandle** rgbTextureHandle);
    void GetNvnVideoFrameInfo(movie::Decoder* decoder, int index, NvnBuffer** nvnBufferYUV,
        NVNtextureHandle**  textureHandleY , NVNtextureHandle**  textureHandleUV, NvnTexture** textureY, NvnTexture** textureUV);
    void GetDecodeVideoFrameInfo(movie::Decoder* decoder, int *index, int *width, int *height, int *yOffset, int *yStride, int *uvOffset, int *colorSpace);
    void ReleaseVideoFrame();
    std::map <movie::Decoder*, NvnRenderContext* > m_NvnRenderContextMap;
    int m_NumberOfVideosToRender;
    int m_VideoId;
    int m_ShaderScratchGranularity;
    int m_MaxNvnFramesPerDecoder;
    nn::os::Mutex m_Lock;
};
