﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include "MovieVideoInputHandler.h"
#include "MovieVideoOutputHandler.h"
#include <map>

/**
* @brief
* MovieDecoderEventHandler class for handling video decoder events.
*
*/
class MediaClock;

void MovieDecoderEventHandlerThreadFunction(void *arg);

class MovieDecoderEventHandler
{
public:
    /**
    * @brief
    * Enumeration of decoder event types.
    */
    enum DecoderEventType
    {
        DecoderEventType_Unknown = 0,
        DecoderEventType_InputBuffer = 1,
        DecoderEventType_OutputBuffer = 2,
        DecoderEventType_FormatChanged = 3,
        DecoderEventType_Error = 4,
        DecoderEventType_ThreadExit = 5,
        DecoderEventType_VideoRenderComplete = 6,
    };

   /**
    * @brief
    * MovieDecoderEventHandler constructor.
    *
    * @details
    * @class MovieDecoderEventHandler instance will be created.
    *
    */
    MovieDecoderEventHandler();

   /**
    * @brief
    * Initialize @class MovieDecoderEventHandler instance.
    *
    * @param[in]  coreMask                      Usable CPU cores in core mask.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_OutOfMemory
    * @retval ::Status_FailedToCreateThread
    *
    * @details
    * Usable cores in core mask. If Nth bit is set to 1, the thread can run on CPU core number N. The core mask need
    * to be set before creation of thread. @class MovieDecoderEventHandler thread is created. The thread will wait
    * on nn::os::MultiWaitType. Any client events need to be linked to this multi wait.
    */
    movie::Status Initialize(uint64_t coreMask);

    /**
    * @brief
    * RegisterEvent. API to register events with @class MovieDecoderEventHandler multi wait.
    *
    * @param[in] extractor                      movie::Extractor.
    * @param[in] decoder                        movie::Decoder (video).
    * @param[in] event                          nn::os::EventType to register.
    * @param[in] holder                         nn::os::MultiWaitHolderType for the event.
    * @param[in] eventType                      Type of event.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API can be used to register DecoderEventType events.
    *
    */
    movie::Status RegisterEvent(movie::Extractor* extractor, movie::Decoder *decoder, nn::os::EventType *event, nn::os::MultiWaitHolderType *holder, DecoderEventType eventType);

    /**
    * @brief
    * RegisterExtractorAndDecoder. API to register extractor, decoder and decoder events with @class MovieDecoderEventHandler.
    *
    * @param[in] extractor                      movie::Extractor.
    * @param[in] decoder                        movie::Decoder (video).
    * @param[in] decoderEvents                  movie::DecoderEvents for the given decoder.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API can be used to register video decoder and corresponding events. Video decoder uses events extensively.
    * @class MovieDecoderEventHandler can receive all events from video decoder and do appropriate actions based on
    * the type of event. Any event from DecoderEventType enumeration can be registered.
    *
    */
    movie::Status RegisterExtractorAndDecoder(movie::Extractor* extractor, movie::Decoder *videoDecoder, movie::DecoderEvents *decoderEvents);

    /**
    * @brief
    * RegisterInputHandler. API to register MovieVideoInputHandler.
    *
    * @param[in] inputHandler                   MovieVideoInputHandler to process input events.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API can be used to register @class MovieVideoInputHandler. @class MovieDecoderEventHandler will receive
    * all events from video decoder and delegates the input event information to @class MovieVideoInputHandler.
    *
    */
    movie::Status RegisterInputHandler(MovieVideoInputHandler *inputHandler);

    /**
    * @brief
    * RegisterInputHandler. API to register MovieVideoOutputHandler.
    *
    * @param[in] outputHandler                   MovieVideoOutputHandler to process output events.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API can be used to register @class MovieVideoOutputHandler. @class MovieDecoderEventHandler will receive
    * all events from video decoder and delegates output event information to @class MovieVideoOutputHandler.
    *
    */
    movie::Status RegisterOutputHandler(MovieVideoOutputHandler *outputHandler);

    /**
    * @brief
    * HandleInputBufferAvailableEvent. API to handle video decoder input buffer events.
    *
    * @param[in] extractor                      movie::Extractor.
    * @param[in] decoder                        movie::Decoder (video).
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError
    *
    * @details
    * When @class MovieDecoderEventHandler receive DecoderEventType_VideoInputBuffer event, this API is called.
    * The handler will acquire all available input buffer indices and add them into @class MovieVideoInputHandler
    * input list.
    *
    */
    movie::Status HandleInputBufferAvailableEvent(movie::Extractor* extractor, movie::Decoder *decoder);

    /**
    * @brief
    * HandleOutputBufferAvailableEvent. API to handle video decoder output buffer events.
    *
    * @param[in] decoder                        movie::Decoder (video).
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError
    *
    * @details
    * When @class MovieDecoderEventHandler receive DecoderEventType_VideoOutputBuffer event, this API is called.
    * The handler will acquire all available output buffer indices and adds them into @class MovieVideoOutputHandler list.
    *
    */
    movie::Status HandleOutputBufferAvailableEvent(movie::Decoder *decoder);

    /**
    * @brief
    * HandleFormatChangedEvent. API to handle video decoder format changed events.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * When @class MovieDecoderEventHandler receive DecoderEventType_VideoFormatChanged event, this API is called.
    * The handler will call CheckAndSignalFormatChangeWaiter() API. If there are any waiters they will be signaled.
    *
    */
    movie::Status HandleFormatChangedEvent();

    /**
    * @brief
    * HandleErrorEvent. API to handle error events from video decoder.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * When @class MovieDecoderEventHandler receive DecoderEventType_VideoError event, HandleErrorEvent() API is called.
    *
    */
    movie::Status HandleErrorEvent();

    /**
    * @brief
    * SignalInputBufferAvailableEvent. API to signal input buffer available event.
    *
    * @return none
    *
    * @details
    * This API can be used to signal input buffer available event of @class MovieVideoInputHandler. This API can also
    * be used to wake the thread and start checking for input buffers. If input buffers are not available, thread will
    * sleep.
    *
    */
    void SignalInputBufferAvailableEvent();

    /**
    * @brief
    * SignalOutputBufferAvailableEvent. API to signal output buffer available event.
    *
    * @return none
    *
    * @details
    * This API can be used to signal output buffer available event of @class MovieVideoOutputHandler. This API can also
    * be used to wake the thread and start checking for output buffers. If output buffers are not available, thread will
    * sleep.
    *
    */
    void SignalOutputBufferAvailableEvent();

    /**
    * @brief
    * WaitForFormatChange. API to wait for format changed events from video decoders.
    *
    * @return none
    *
    * @details
    * This API can be used to wait for format changed events from video decoders. This API will return when
    * format changed events are received from all video decoders or 5 seconds wait time is expired.
    *
    */
    void WaitForFormatChange();

    /**
    * @brief
    * Start. API to start MovieDecoderEventHandler.
    *
    * @return none
    *
    */
    void Start();

    /**
    * @brief
    * CheckAndSignalFormatChangeWaiter. API to check for format changed event and signal any waiter.
    *
    * @return none
    *
    * @details
    * This API can be used to check format changed event and signal if any waiters available.
    *
    */
    void CheckAndSignalFormatChangeWaiter();

    /**
    * @brief
    * MovieDecoderEventHandler destructor.
    *
    * @return None
    *
    * @details
    * All the resources allocated by @class MovieDecoderEventHandler are released.
    *
    */
    ~MovieDecoderEventHandler();

    /**
    * @brief
    * SetVideoDecoderFlushState, API to set video decoder flush state.
    *
    * @param[in] isflushing                     Decoder flush state.
    *
    * @return boolean
    *
    * @retval true                              Terminate thread.
    * @retval false                             Do not terminate thread..
    *
    * @details
    * This API can be used set video decoder flush state. Once flush is complete, decoder needs to be started again.
    * This will result in arrival of input buffer available events.
    *
    */
    void SetVideoDecoderFlushState(bool isflushing) { m_VideoDecoderInFlush = isflushing; }

    /**
    * @brief
    * GetVideoDecoderFlushState, API to get video decoder flush state.
    *
    * @return None
    *
    * @details
    * This API can be used get video decoder flush state.
    *
    */
    bool GetVideoDecoderFlushState() { return m_VideoDecoderInFlush.load(); }

    nn::os::MultiWaitType m_DecoderMultiWait;

    friend void MovieDecoderEventHandlerThreadFunction(void *arg);

    struct EventUserData
    {
        EventUserData(nn::os::EventType *event, DecoderEventType eventType, movie::Extractor* extractor, movie::Decoder* decoder)
            : m_Event(event),
              m_EventType(eventType),
              m_Decoder(decoder),
              m_Extractor(extractor){ }
        nn::os::EventType *m_Event;
        DecoderEventType m_EventType;
        movie::Decoder* m_Decoder;
        movie::Extractor* m_Extractor;
    };

private:
    void DeleteUserData(EventUserData* userData);
    movie::Decoder* m_VideoDecoder;
    MovieVideoInputHandler *m_MovieVideoInputHandler;
    MovieVideoOutputHandler *m_MovieVideoOutputHandler;
    size_t m_ThreadStackSize;
    void* m_ThreadStack;
    nn::os::ThreadType m_ThreadType;
    nn::os::EventType m_ThreadExitEvent;
    nn::os::MultiWaitHolderType m_ThreadExitEventHolder;
    nn::os::EventType m_FormatChangedPlayerEvent;
    bool m_VideoFormatChangeEventReceived;
    std::atomic<bool> m_VideoDecoderInFlush;

    struct DecoderEventInfo
    {
        movie::Decoder* decoder;
        movie::Status lastError;

        nn::os::EventType errorEvent;
        nn::os::EventType formatChangedEvent;
        nn::os::EventType inputBufferAvailableEvent;
        nn::os::EventType outputBufferAvailableEvent;

        nn::os::MultiWaitHolderType errorEventHolder;
        nn::os::MultiWaitHolderType formatChangedEventHolder;
        nn::os::MultiWaitHolderType inputBufferAvailableEventHolder;
        nn::os::MultiWaitHolderType outputBufferAvailableEventHolder;
    };
    std::map<DecoderEventInfo*, movie::Decoder*> m_DecoderEventMap;
};
