﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @brief
 * Memory allocator functions used by player sample.
 *
 */
#pragma once

#include <memory>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nv/nv_MemoryManagement.h>
#include <nns/mm/mm_MoviePlayerUtils.h>

static bool g_TrackMemoryAllocation = true;

/**
 * @brief
 * MovieAllocate, API to provide memory to movie library allocation requests
 *
 * @param[in]  size                             Requested memory size.
 * @param[in]  alignment                        Requested memory alignment.
 * @param[in]  userPtr                          User pointer, can be used for tracking memory allocations.
 *
 * @return void* 				                Allocated memory pointer.
 *
 * @details
 * Allocates memory using nn::mem::StandardAllocator. If memory tracking is enabled allocation are logged and are
 * checked for leaks when the program exits.
 *
 */
void* MovieAllocate(size_t size, size_t alignment, void *userPtr);

/**
 * @brief
 * MovieDeallocate, API to free memory allocated through MovieAllocate() API
 *
 * @param[in]  address                          Memory pointer.
 * @param[in]  userPtr                          User pointer, can be used for tracking memory allocations.
 *
 * @return none.
 *
 * @details
 * Frees memory using nn::mem::StandardAllocator. If memory tracking is enabled each free call removes allocation
 * entry from the tracker.
 *
 */
void MovieDeallocate(void *address, void *userPtr);

/**
 * @brief
 * MovieReallocate, API to reallocate memory.
 *
 * @param[in]  address                          Old memory address.
 * @param[in]  newSize                          New memory size
 * @param[in]  userPtr                          User pointer, can be used for tracking memory allocations.
 *
 * @return void* 				                Reallocated memory pointer.
 *
 * @details
 * Reallocates memory using nn::mem::StandardAllocator. If memory tracking is enabled previous memory allocation entry
 * is cleared and new allocation are logged and are checked for leaks when the program exits.
 *
 */
void* MovieReallocate(void *address, size_t newSize, void *userPtr);

/**
 * @brief
 * FsInitHeap, API create CreateExpHeap() used by file system
 *
 * @return none
 *
 */
void FsInitHeap();

/**
 * @brief
 * FsAllocate, API to allocate memory from ExHeap.
 *
 * @param[in]  size                             Memory size
 *
 * @return void* 	                            Allocated memory pointer.
 *
 * @details
 * Allocated memory using nn::lmem::AllocateFromExpHeap()
 *
 */
void* FsAllocate(size_t size);

/**
 * @brief
 * MovieDeallocate, API to free memory allocated through FsAllocate()
 *
 * @param[in]  address                           Memory pointer.
 * @param[in]  size                              Allocated memory size.
 *
 * @return none.
 *
 * @details
 * Frees memory using nn::lmem::FreeToExpHeap()
 *
 */
void FsDeallocate(void *address, size_t size);

/**
 * @brief
 * GlobalAllocator, API to get global nn::mem::StandardAllocator reference
 *
 *
 * @return nn::mem::StandardAllocator.
 *
 * @details
 * Returns reference to global nn::mem::StandardAllocator.
 *
 */
nn::mem::StandardAllocator &GlobalAllocator();

/**
 * @brief
 * GetGraphicsHeap, API to get graphics heap info.
 *
 * @param[out]  graphicsHeap                           Graphics heap memory pointer.
 * @param[out]  graphicsHeapSize                       Graphics heap memory size.
 *
 * @return none.
 *
 * @details
 * This API can be used to get the global memory allocated for graphics heap.
 *
 */
void GetGraphicsHeap(uint8_t **graphicsHeap, size_t *graphicsHeapSize);

MemoryTracker &MovieMemoryTracker();

MemoryTracker &CoreMemoryTracker();

MemoryTracker &MallocMemoryTracker();

MemoryTracker &NewMemoryTracker();
