﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @examplesource{MiiG3dHeadwear.cpp,PageSampleMiiG3dHeadwear}
 *
 * @brief
 * G3d で Mii に帽子をかぶせるサンプルプログラム
 */

/**
 * @page PageSampleMiiG3dHeadwear MiiG3dHeadwear
 * @tableofcontents
 *
 * @brief
 * G3d で Mii に帽子をかぶせるサンプルプログラムの解説です。
 *
 * @section PageSampleMiiG3dHeadwear_SectionBrief 概要
 * Mii モデルがアサインされた nn::g3d::ResModel に帽子をかぶせるサンプルです。
 * Mii の各パーツの nn::mii::CreateModelType を G3d を使用して切り替えています。
 *
 * @section PageSampleMiiG3dHeadwear_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/MiiG3dHeadwear
 * Samples/Sources/Applications/MiiG3dHeadwear @endlink 以下にあります。
 *
 * @section PageSampleMiiG3dHeadwear_SectionNecessaryEnvironment 必要な環境
 * 特にありません。
 *
 * @section PageSampleMiiG3dHeadwear_SectionHowToOperate 操作方法
 * 特にありません。
 *
 * @section PageSampleMiiG3dHeadwear_SectionPrecaution 注意事項
 * 動作には Mii ライブラリが必要です。
 *
 * @section PageSampleMiiG3dHeadwear_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleMiiG3dHeadwear_SectionDetail 解説
 * このサンプルプログラムは、nn::g3d::ResFile へ Mii モデルをアサインし、アニメーションを適用し、
 * Mii モデルを表示します。サンプルプログラムの全体像を説明します。
 *
 * @subsection PageSampleMiiG3dHeadwear_SectionDetail_ModelConvert モデルコンバート処理
 * 下記のリソースをコンバートし 1 つの bfres に纏めます。
 * - Samples/Sources/Applications/MiiG3dHeadwear/Resources/Model/MiiSeparateBone.fmdb
 *
 *   頂点やインデクスを含まないデータ構造のみが記述された空のモデルデータです。
 *   ランタイムで Mii モデルがアサインされます。
 *   ボーンがパーツ毎に独立して存在するのでパーツ毎にビジビリティを切り替える事ができます。
 *   しかし、ボーンに関する CPU 処理コストがボーンを結合しない時に比べて増加します。
 *
 * - Samples/Sources/Applications/MiiG3dHeadwear/Resources/Model/CapSample00.fmdb
 *
 *   Mii にかぶせる帽子のモデルです。
 *
 * - Samples/Sources/Applications/MiiG3dHeadwear/Resources/Model/MiiSeparateBoneFaceExpressionChange.fmab
 *
 *   Mii の表情を切り替えるためのマテリアルアニメーションです。
 *   ランタイムで Mii モデルがアサインされた nn::g3d::ResModel に適用します。
 *
 * - Samples/Sources/Applications/MiiG3dHeadwear/Resources/Model/MiiSeparateBoneModelTypeChange.fvbb
 *
 *   Mii の各パーツについて nn:mii::CreateModelType を切り替えるアニメーションです。
 *
 * - Samples/Sources/Applications/MiiG3dHeadwear/Resources/Model/CapSample00WearOnOff.fvbb
 *
 *   帽子をかぶったり脱いだりするアニメーションです。
 *
 * - Samples/Sources/Applications/MiiG3dHeadwear/Resources/Shader/SampleShader.glsl
 *
 *   Mii を表示するためのシェーダです。
 *
 * @subsection PageSampleMiiG3dHeadwear_SectionDetail_RuntimeFlow サンプルプログラム実行時の処理
 * サンプルプログラム実行時の処理の流れを説明します。
 *
 * - Mii モデルの構築
 * - Mii モデルの nn::g3d::ResModel へのアサイン
 * - nn::g3d モデルインスタンスの作成
 * - nn::g3d マテリアル、ボーンビジビリティアニメーションインスタンスの作成
 * - レンダーループ開始
 * - アニメーション更新
 * - モデル更新
 * - コマンドリスト実行
 * - スキャンバッファをスワップ
 * - レンダーループ開始に戻る
 *
 */

#include <cstdlib>
#include <cstring>
#include <vector>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/mii.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_BitArray.h>
#include <nn/g3d/g3d_ResFile.h>
#include <nn/g3d/g3d_ResShader.h>
#include <nn/g3d/g3d_ModelObj.h>
#include <nn/g3d/g3d_MaterialAnimObj.h>
#include <nn/g3d/g3d_BoneVisibilityAnimObj.h>
#include <nn/g3d/mii/g3d_MiiConverter.h>
#include <nns/gfx/gfx_GraphicsFramework.h>

namespace {

nns::gfx::GraphicsFramework g_GfxFramework;

const size_t MemoryHeapSize = 16 * 1024 * 1024;

/// CharInfoのパス
const char* CharInfoPath = "Contents:/Mii/charinfo/SampleMiiG3d.dat";
/// TextureShaderのパス
const char* TextureShaderPath = "Contents:/Mii/shader/TextureShader.bnsh";
#if NN_BUILD_CONFIG_SPEC_NX
#if NN_BUILD_CONFIG_OS_HORIZON
/// NX用テクスチャリソースのパス
const char* TextureResourcePath = "Contents:/Mii/resource/NXTextureMidSRGB.dat";
#else
/// WinNvn用テクスチャリソースのパス
const char* TextureResourcePath = "Contents:/Mii/resource/WinNvnTextureMidSRGB.dat";
#endif    // #if NN_BUILD_CONFIG_SPEC_NX
# else
/// OpenGl用リソースのパス
const char* TextureResourcePath = "Contents:/Mii/resource/WinGenericTextureMidSRGB.dat";
#endif/// シェイプリソースのパス
const char* ShapeResourcePath = "Contents:/Mii/resource/ShapeMid.dat";
/// Mii描画用簡易シェーダー
const char* SampleShaderPath = "Contents:/SampleShader.bnsh";

/// マスクテクスチャとFacelineテクスチャの解像度
const size_t TextureResolution = 512;

/// マスクテクスチャとFacelineテクスチャのミップマップ数
const int TextureMipCount = 1;

/// マスクテクスチャとFacelineテクスチャのフォーマット
const nn::gfx::ImageFormat TextureFormat = nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;

/// 生成する表情のフラグ
const int ExpressionFlags = nn::mii::ExpressionFlag_Normal | nn::mii::ExpressionFlag_Smile | nn::mii::ExpressionFlag_Surprise;

/// 生成するModelの設定
const int CreateFlag = nn::mii::CreateFlag_Normal | nn::mii::CreateFlag_Hat |  nn::mii::CreateFlag_NoseNormal;

/// SampleShaderの行列コンスタントバッファ
struct ConstantBufferMatrix
{
    nn::util::FloatColumnMajor4x3 modelView;
    nn::util::FloatColumnMajor4x4 projection;
};

/// SampleShaderのModulateコンスタントバッファ
struct ConstantBufferModulate
{
    int32_t modulateType; ///< @ref ModulateType
    int32_t pad[3];
    nn::util::Float4 constantColor[3]; ///< z値は利用されていない。
};

enum Model
{
    Model_MiiSeparateBone,
    Model_Cap,
    Model_Count,
};

enum MaterialAnim
{
    MaterialAnim_FaceExpressionChange,
    MaterialAnim_Count,
};
const char* g_MaterialAnimNameArray[] =
{
    "MiiSeparateBoneFaceExpressionChange",
};
NN_STATIC_ASSERT( ( sizeof( g_MaterialAnimNameArray ) / sizeof( char* ) ) == MaterialAnim_Count );

enum BoneVisibilityAnim
{
    BoneVisibilityAnim_CharModelTypeChange,
    BoneVisibilityAnim_CapWearOnOff,
    BoneVisibilityAnim_Count,
};
const char* g_BoneAnimNameArray[] =
{
    "MiiSeparateBoneModelTypeChange",
    "CapSample00WearOnOff",
};
NN_STATIC_ASSERT( ( sizeof( g_BoneAnimNameArray ) / sizeof( char* ) ) == BoneVisibilityAnim_Count );

nn::g3d::ModelObj* g_ModelObjPtrArray[Model_Count];
bool g_IsVerticalFlip = false;

//-----------------------------------------------------------------------------
// メモリ
nn::util::BytePtr g_pMemoryHeap( NULL );
nn::util::BytePtr g_pMemory( NULL );

char* g_MountRomCacheBuffer = NULL;

// ビューポートシザーを初期化
nn::gfx::ViewportScissorState g_ViewportScissor;
void InitializeViewport()
{
    auto pDevice = g_GfxFramework.GetDevice();

    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled( true );
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetWidth( static_cast<float>( g_GfxFramework.GetDisplayWidth() ) );
        viewportInfo.SetHeight( static_cast<float>( g_GfxFramework.GetDisplayHeight() ) );
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetWidth( g_GfxFramework.GetDisplayWidth() );
        scissorInfo.SetHeight( g_GfxFramework.GetDisplayHeight() );
    }
    info.SetViewportStateInfoArray( &viewportInfo, 1 );
    info.SetScissorStateInfoArray( &scissorInfo, 1 );
    g_ViewportScissor.Initialize( pDevice, info );
}

void InitializeResources()
{
    // メモリ
    g_pMemoryHeap.Reset( malloc( MemoryHeapSize ) );
    g_pMemory = g_pMemoryHeap;

    InitializeViewport();
}

void FinalizeResources()
{
    g_ViewportScissor.Finalize( g_GfxFramework.GetDevice() );
    g_GfxFramework.Finalize();
    free( g_pMemoryHeap.Get() );
}

void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t )
{
    free( ptr );
}

//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
void InitializeFs()
{
    nn::Result result;

    nn::fs::SetAllocator( Allocate, Deallocate );

    size_t cacheSize = 0;
    result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT( result.IsSuccess() );

    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(g_MountRomCacheBuffer);

    result = nn::fs::MountRom("Contents", g_MountRomCacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );
}

void FinalizeFs()
{
    nn::fs::Unmount("Contents");
    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

nn::gfx::Sampler g_Sampler;
void InitializeSampler()
{
    nn::gfx::Sampler::InfoType info;

    /// Miiで利用できるようにサンプラを設定する
    nn::mii::GetSamplerInfo(&info);

    /// サンプラー初期化
    g_Sampler.Initialize( g_GfxFramework.GetDevice(), info );
}

nn::mii::CharInfo g_CharInfo;
void LoadCharInfo()
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle,CharInfoPath,nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    result = nn::fs::ReadFile(handle, 0, &g_CharInfo, sizeof(g_CharInfo));
    NN_ASSERT(result.IsSuccess());

    nn::fs::CloseFile(handle);

    /// 中身をチェックする
    NN_ASSERT(nn::mii::CharInfoAccessor(g_CharInfo).IsValid());
}
nn::gfx::ShaderCodeType GetShaderCodeType(const nn::gfx::ResShaderContainer* pContainer)
{
    for( int idxShaderCodeType = 0; idxShaderCodeType < nn::gfx::ShaderCodeType_End; ++idxShaderCodeType )
    {
        auto codeType = nn::gfx::ShaderCodeType( idxShaderCodeType );
        if( pContainer->GetResShaderVariation(0)->GetResShaderProgram( codeType ) != nullptr )
        {
            return codeType;
        }
    }
    NN_ASSERT( 0 );
    return nn::gfx::ShaderCodeType_Source;
}

nn::gfx::ResShaderFile* InitializeResShaderFile(nn::gfx::ShaderCodeType* pCodeType,const char* pPath)
{
    void* pFile;
    {
        /// シェーダーファイルを読み込む
        nn::fs::FileHandle handle;
        nn::Result result = nn::fs::OpenFile(&handle,pPath,nn::fs::OpenMode_Read);
        NN_ASSERT(result.IsSuccess());

        /// ファイルサイズ取得
        int64_t size;
        result = nn::fs::GetFileSize(&size, handle);
        NN_ASSERT(result.IsSuccess());

        /// アライメントを取得する
        size_t alignment;
        nn::util::BinaryFileHeader header;
        result = nn::fs::ReadFile(handle, 0, &header,sizeof(header));
        NN_ASSERT(result.IsSuccess());
        alignment = header.GetAlignment();

        /// リソース用メモリ確保
        g_pMemory.AlignUp(alignment);
        pFile = g_pMemory.Get();
        g_pMemory.Advance(ptrdiff_t(size));

        /// リソースを読み込む
        result = nn::fs::ReadFile(handle, 0, pFile,size_t(size));
        NN_ASSERT(result.IsSuccess());
        NN_UNUSED( result );
        nn::fs::CloseFile(handle);
    }

    /// シェーダーの初期化
    nn::gfx::ResShaderFile* pShaderFile = nn::gfx::ResShaderFile::ResCast(pFile);
    nn::gfx::ResShaderContainer* pContainer = pShaderFile->GetShaderContainer();
    *pCodeType = GetShaderCodeType(pContainer);
    pContainer->Initialize( g_GfxFramework.GetDevice() );
    for( int idxShaderVariation = 0; idxShaderVariation < pContainer->GetShaderVariationCount(); ++idxShaderVariation )
    {
        nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation( idxShaderVariation );
        nn::gfx::ShaderInitializeResult result = pVariation->GetResShaderProgram(*pCodeType)->Initialize( g_GfxFramework.GetDevice() );
        NN_ASSERT( result == nn::gfx::ShaderInitializeResult_Success );
        NN_UNUSED( result );
    }

    return pShaderFile;
}

nn::gfx::ResShaderFile* g_pTextureShaderFile;
nn::gfx::ShaderCodeType g_TextureShaderCodeType;
nn::mii::TextureShader g_TextureShader;
nn::mii::TextureShaderInfo g_TextureShaderInfo;
void InitializeTextureShader()
{
    auto pDevice = g_GfxFramework.GetDevice();
    g_pTextureShaderFile = InitializeResShaderFile(&g_TextureShaderCodeType, TextureShaderPath);

    /// フォーマット設定
    g_TextureShaderInfo.SetFacelineImageFormat(TextureFormat);
    g_TextureShaderInfo.SetMaskImageFormat(TextureFormat);

    /// アライメントとサイズを取得
    size_t alignment = nn::mii::TextureShader::CalculateMemoryAlignment(*g_pTextureShaderFile, g_TextureShaderCodeType, g_TextureShaderInfo, pDevice);
    size_t size = nn::mii::TextureShader::CalculateMemorySize(*g_pTextureShaderFile, g_TextureShaderCodeType, g_TextureShaderInfo, pDevice);

    g_pMemory.AlignUp(alignment);
    void* pMemory = g_pMemory.Get();
    g_pMemory.Advance(size);

    nn::Result result = g_TextureShader.Initialize( pMemory,size, pDevice,*g_pTextureShaderFile, g_TextureShaderCodeType, g_TextureShaderInfo );
    NN_ASSERT(result.IsSuccess());
}

void* ReadFile(size_t* pSize, const char* filePath, size_t alignement)
{
    int64_t size;

    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filePath, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    /// ファイルサイズ取得
    result = nn::fs::GetFileSize(&size, handle);
    NN_ASSERT(result.IsSuccess());

    /// メモリ確保
    g_pMemory.AlignUp(alignement);
    void* ptr = g_pMemory.Get();
    g_pMemory.Advance(ptrdiff_t(size));

    /// ファイルを読み込む
    result = nn::fs::ReadFile(handle, 0, ptr, size_t(size));
    NN_ASSERT(result.IsSuccess());

    /// 閉じる
    nn::fs::CloseFile(handle);

    *pSize = size_t(size);
    return ptr;
}

nn::mii::Resource g_Resource;
void* g_pShapeResourceFile;
void* g_pTextureResourceFile;
void InitializeMiiResource()
{
    /// テクスチャリソースファイルの読み込み
    size_t textureFileSize;
    g_pTextureResourceFile = ReadFile( &textureFileSize, TextureResourcePath, nn::mii::ResourceMemoryAlignment );
    /// シェイプリソースファイルの読み込み
    size_t shapeFileSize;
    g_pShapeResourceFile = ReadFile( &shapeFileSize, ShapeResourcePath, nn::mii::ResourceMemoryAlignment );

    nn::mii::ResourceInfo info;
    info.SetDefault();
    info.SetGammaType(nn::mii::GammaType_Srgb);

    /// サイズとアライメント取得
    size_t alignment =
        nn::mii::Resource::CalculateMemoryAlignment( g_pTextureResourceFile, textureFileSize,
                                                    g_pShapeResourceFile, shapeFileSize, info );

    size_t size =
        nn::mii::Resource::CalculateMemorySize( g_pTextureResourceFile, textureFileSize,
                                                g_pShapeResourceFile, shapeFileSize, info );

    /// メモリ確保
    g_pMemory.AlignUp(alignment);
    void* pMemory = g_pMemory.Get();
    g_pMemory.Advance(size);

    /// リソースを初期化する
    nn::Result result = g_Resource.Initialize(pMemory,size, g_pTextureResourceFile, textureFileSize,
                                            g_pShapeResourceFile, shapeFileSize, info, g_GfxFramework.GetDevice() );
    NN_ASSERT(result.IsSuccess());
}

/// 指定フラグのindex番目の表情を取得する
nn::mii::Expression GetExpression(int flag,int index)
{
    int count = 0;
    for( int idxExpression = 0; idxExpression < nn::mii::Expression_End; ++idxExpression )
    {
        if( ( flag & ( 1 << idxExpression ) ) != 0 )
        {
            if( count == index )
            {
                return nn::mii::Expression( idxExpression );
            }
            ++count;
        }
    }
    /// ここには到達しない。
    return nn::mii::Expression_Normal;
}

nn::mii::CharModel g_CharModel;
void InitializeCharModel()
{
    auto pDevice = g_GfxFramework.GetDevice();
    nn::mii::CharModelInfo info;
    info.SetDefault();
    info.SetDynamicTextureResolution( TextureResolution,TextureResolution );
    info.SetCreateFlag( CreateFlag );
    info.SetDynamicTextureFormat( TextureFormat,TextureFormat );
    info.SetMaskCount( nn::mii::GetExpressionCount(ExpressionFlags ) );
    info.SetDynamicTextureMipCount( TextureMipCount, TextureMipCount );

    /// メモリサイズとアライメント取得
    size_t alignment = nn::mii::CharModel::CalculateMemoryAlignment( info );
    size_t size = nn::mii::CharModel::CalculateMemorySize( info );

    /// 一時メモリのサイズ取得
    size_t tempSize = nn::mii::CharModel::CalculateMemorySize( info );

    /// メモリと一時メモリを確保する
    g_pMemory.AlignUp( alignment );
    void* pMemory = g_pMemory.Get();
    g_pMemory.Advance( size );

    /// 一時メモリはCharModel::Initialize()後破棄できます。
    nn::mii::TemporaryBuffer* pTempMemory = g_pMemory.Get<nn::mii::TemporaryBuffer>();
    g_pMemory.Advance(tempSize);

    /// メモリプールとサイズを取得
    size_t dynamicMemoeryPoolSize = nn::mii::CharModel::CalculateDynamicMemoryPoolSize( pDevice, info );
    size_t dynamicMemoryPoolAlignment = nn::mii::CharModel::CalculateDynamicMemoryPoolAlignment( pDevice, info );
    size_t staticMemoeryPoolSize = nn::mii::CharModel::CalculateStaticMemoryPoolSize( g_Resource );
    size_t staticMemoryPoolAlignment = nn::mii::CharModel::CalculateStaticMemoryPoolAlignment( g_Resource );
    ptrdiff_t dynamicMemoryPoolOffset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,
                                                                            dynamicMemoeryPoolSize, dynamicMemoryPoolAlignment );
    ptrdiff_t staticMemoryPoolOffset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_Data,
                                                                            staticMemoeryPoolSize, staticMemoryPoolAlignment );


    /// CharModelを初期化する
    nn::Result result = g_CharModel.Initialize( pMemory, size
                                                , pDevice
                                                , g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget )
                                                , dynamicMemoryPoolOffset, dynamicMemoeryPoolSize
                                                , g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_Data )
                                                , staticMemoryPoolOffset, staticMemoeryPoolSize
                                                , g_Resource
                                                , pTempMemory
                                                , info, g_CharInfo );
    NN_ASSERT(result.IsSuccess());
}

nn::mii::Faceline g_Faceline;
void InitializeFaceline()
{
    /// メモリサイズとアライメント取得
    size_t alignment = nn::mii::Faceline::CalculateMemoryAlignment();
    size_t size = nn::mii::Faceline::CalculateMemorySize();

    /// 一時メモリのサイズ取得
    size_t tempSize = nn::mii::TemporaryBufferSize;

    /// メモリと一時メモリを確保する
    g_pMemory.AlignUp(alignment);
    void* pMemory = g_pMemory.Get();
    g_pMemory.Advance(size);

    /// 一時メモリはFaceline::Initialize()後破棄できます。
    nn::mii::TemporaryBuffer* pTempMemory = g_pMemory.Get<nn::mii::TemporaryBuffer>();
    g_pMemory.Advance(tempSize);


    /// メモリプールとサイズを取得
    auto pDevice = g_GfxFramework.GetDevice();
    size_t memoryPoolAlignment = nn::mii::Faceline::CalculateMemoryPoolAlignment( pDevice, g_Resource );
    size_t memoryPoolSize = nn::mii::Faceline::CalculateMemoryPoolSize( pDevice, g_Resource );

    /// メモリプールアライメント調整
    ptrdiff_t offset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_Data, memoryPoolSize, memoryPoolAlignment );

    g_Faceline.Initialize( pMemory, size, pDevice
                        , g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_Data )
                        , offset, memoryPoolSize, pTempMemory
                        , g_Resource, g_CharInfo, g_IsVerticalFlip );
}

nn::mii::Mask g_Mask;
void InitializeMask()
{
    /// メモリサイズとアライメント取得
    size_t alignment = nn::mii::Mask::CalculateMemoryAlignment();
    size_t size = nn::mii::Mask::CalculateMemorySize();

    /// 一時メモリのサイズ取得
    size_t tempSize = nn::mii::TemporaryBufferSize;

    /// メモリと一時メモリを確保する
    g_pMemory.AlignUp(alignment);
    void* pMemory = g_pMemory.Get();
    g_pMemory.Advance(size);

    /// 一時メモリはMask::Initialize()後破棄できます。
    nn::mii::TemporaryBuffer* pTempMemory = g_pMemory.Get<nn::mii::TemporaryBuffer>();
    g_pMemory.Advance(tempSize);

    /// メモリプールとサイズを取得
    auto pDevice = g_GfxFramework.GetDevice();
    size_t memoryPoolAlignment = nn::mii::Mask::CalculateMemoryPoolAlignment( pDevice, g_Resource, ExpressionFlags);
    size_t memoryPoolSize = nn::mii::Mask::CalculateMemoryPoolSize( pDevice, g_Resource, ExpressionFlags);

    /// メモリプールアライメント調整
    ptrdiff_t offset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_Data, memoryPoolSize, memoryPoolAlignment );

    g_Mask.Initialize( pMemory, size, pDevice,
                        g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_Data ), offset,
                        memoryPoolSize,
                        pTempMemory, g_Resource, g_CharInfo, ExpressionFlags, g_IsVerticalFlip );
}

nn::gfx::DescriptorSlot g_SamplerSlot;
void BindDescriptorSlot()
{
    /// サンプラのバインド
    {
        auto pSamplerDescriptorPool = g_GfxFramework.GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler );
        pSamplerDescriptorPool->BeginUpdate();
        const int slotIndex = g_GfxFramework.AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, 1 );
        pSamplerDescriptorPool->SetSampler( slotIndex, &g_Sampler);
        pSamplerDescriptorPool->GetDescriptorSlot( &g_SamplerSlot, slotIndex );
        pSamplerDescriptorPool->EndUpdate();
    }

    /// テクスチャービューのバインド
    auto pTextureDescriptorPool = g_GfxFramework.GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView );
    pTextureDescriptorPool->BeginUpdate();

    /// CharModelのMask以外のテクスチャをバインドする
    for( int idxTextureType = 0; idxTextureType < nn::mii::CharModel::TextureType_End; ++idxTextureType )
    {
        int mask = 0;
        nn::mii::CharModel::TextureType type = nn::mii::CharModel::TextureType( idxTextureType );
        const nn::gfx::TextureView* pView = g_CharModel.GetTextureView( type, mask );

        if( pView == nullptr )
        {
            continue;
        }

        if( type == nn::mii::CharModel::TextureType_Mask )
        {
            continue;
        }

        nn::gfx::DescriptorSlot slot;
        const int slotIndex = g_GfxFramework.AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1 );
        pTextureDescriptorPool->SetTextureView( slotIndex, pView );
        pTextureDescriptorPool->GetDescriptorSlot( &slot, slotIndex );

        /// 取得したスロットをCharModelに設定する
        g_CharModel.SetTextureDescriptorSlot( type, slot, mask );
    }

    /// CharModelのMaskのテクスチャをバインドする
    for( int idxExperssion = 0; idxExperssion < nn::mii::GetExpressionCount( ExpressionFlags ); ++idxExperssion )
    {
        nn::mii::CharModel::TextureType type = nn::mii::CharModel::TextureType_Mask;
        const nn::gfx::TextureView* pView = g_CharModel.GetTextureView( type, idxExperssion );

        nn::gfx::DescriptorSlot slot;
        const int slotIndex = g_GfxFramework.AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1 );
        pTextureDescriptorPool->SetTextureView( slotIndex, pView );
        pTextureDescriptorPool->GetDescriptorSlot( &slot, slotIndex );

        /// 取得したスロットをCharModelに設定する
        g_CharModel.SetTextureDescriptorSlot( type, slot, idxExperssion );
    }

    /// Facelineのテクスチャをバインドする
    for( int idxTextureType = 0; idxTextureType < nn::mii::Faceline::TextureType_End; ++idxTextureType )
    {
        nn::mii::Faceline::TextureType type = nn::mii::Faceline::TextureType( idxTextureType );
        const nn::gfx::TextureView* pView = g_Faceline.GetTextureView( type );

        if( pView == nullptr )
        {
            continue;
        }

        nn::gfx::DescriptorSlot slot;
        const int slotIndex = g_GfxFramework.AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1 );
        pTextureDescriptorPool->SetTextureView( slotIndex, pView );
        pTextureDescriptorPool->GetDescriptorSlot( &slot, slotIndex );

        /// 取得したスロットをCharModelに設定する
        g_Faceline.SetTextureDescriptorSlot( type, slot );
    }

    /// Maskのテクスチャをバインドする
    for( int idxMask = 0; idxMask < nn::mii::Mask::TextureType_End; ++idxMask )
    {
        nn::mii::Mask::TextureType type = nn::mii::Mask::TextureType( idxMask );
        const nn::gfx::TextureView* pView = g_Mask.GetTextureView( type );

        if( pView == nullptr )
        {
            continue;
        }

        nn::gfx::DescriptorSlot slot;
        const int slotIndex = g_GfxFramework.AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1 );
        pTextureDescriptorPool->SetTextureView( slotIndex, pView );
        pTextureDescriptorPool->GetDescriptorSlot( &slot, slotIndex );

        /// 取得したスロットをCharModelに設定する
        g_Mask.SetTextureDescriptorSlot( type, slot );
    }

    pTextureDescriptorPool->EndUpdate();
}

nn::mii::FacelineGpuBuffer g_FacelineGpuBuffer;
void InitializeFacelineGpuBuffer()
{
    size_t alignment = nn::mii::FacelineGpuBuffer::CalculateMemoryAlignment();
    size_t size = nn::mii::FacelineGpuBuffer::CalculateMemorySize();

    /// メモリを確保する
    g_pMemory.AlignUp(alignment);
    void* pMemory = g_pMemory.Get();
    g_pMemory.Advance( size );

    /// メモリプールとサイズを取得
    auto pDevice = g_GfxFramework.GetDevice();
    size_t memoryPoolAlignment = nn::mii::FacelineGpuBuffer::CalculateMemoryPoolAlignment( pDevice );
    size_t memoryPoolSize = nn::mii::FacelineGpuBuffer::CalculateMemoryPoolSize( pDevice );

    /// メモリプールアライメント調整
    ptrdiff_t offset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, memoryPoolSize, memoryPoolAlignment );
    g_FacelineGpuBuffer.Initialize( pMemory, size, pDevice, g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer ), offset, memoryPoolSize );
}

nn::mii::MaskGpuBuffer g_MaskGpuBuffer;
void InitializeMaskGpuBuffer()
{
    size_t alignment = nn::mii::MaskGpuBuffer::CalculateMemoryAlignment();
    size_t size = nn::mii::MaskGpuBuffer::CalculateMemorySize();

    /// メモリを確保する
    g_pMemory.AlignUp(alignment);
    void* pMemory = g_pMemory.Get();
    g_pMemory.Advance(size);

    /// メモリプールとサイズを取得
    auto pDevice = g_GfxFramework.GetDevice();
    size_t memoryPoolAlignment = nn::mii::MaskGpuBuffer::CalculateMemoryPoolAlignment( pDevice );
    size_t memoryPoolSize = nn::mii::MaskGpuBuffer::CalculateMemoryPoolSize( pDevice );

    /// メモリプールアライメント調整
    ptrdiff_t offset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, memoryPoolSize, memoryPoolAlignment );
    g_MaskGpuBuffer.Initialize( pMemory, size, pDevice, g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer ), offset, memoryPoolSize );
}

void RenderFacelineTexture()
{
    /// GPUバッファを設定
    g_FacelineGpuBuffer.SetColorTarget( g_GfxFramework.GetDevice(), g_CharModel.GetFacelineTexture()
                                       ,g_TextureShaderInfo
                                       ,TextureResolution / 2,TextureResolution, TextureMipCount );
    g_FacelineGpuBuffer.SetFaceline( g_Faceline );

    /// GPUを使って描画する
    const int cmdBufferIndex = 0;
    auto pCommandBuffer = g_GfxFramework.GetRootCommandBuffer( cmdBufferIndex );
    g_GfxFramework.BeginFrame( cmdBufferIndex );
    pCommandBuffer->SetDescriptorPool( g_GfxFramework.GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView ) );
    pCommandBuffer->SetDescriptorPool( g_GfxFramework.GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler ) );
    pCommandBuffer->InvalidateMemory( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
                                    | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer );
    g_TextureShader.DrawFaceline( pCommandBuffer, &g_FacelineGpuBuffer, g_SamplerSlot );
    g_GfxFramework.EndFrame( cmdBufferIndex );

    /// コマンドの実行と完了待ち
    auto pQueue = g_GfxFramework.GetQueue();
    pQueue->ExecuteCommand( pCommandBuffer, 0 );
    pQueue->Sync();
}

void RenderMaskTexture()
{
    for( int idxExpression = 0; idxExpression < nn::mii::GetExpressionCount( ExpressionFlags ); ++idxExpression )
    {
        /// 表情を取得
        nn::mii::Expression expression = GetExpression( ExpressionFlags, idxExpression );

        /// GPUバッファを設定
        g_MaskGpuBuffer.SetColorTarget( g_GfxFramework.GetDevice(), g_CharModel.GetMaskTexture( idxExpression )
                                        ,g_TextureShaderInfo,TextureResolution, TextureMipCount );
        g_MaskGpuBuffer.SetMask( g_Mask, expression );

        /// GPUを使って描画する
        const int cmdBufferIdx = 0;
        auto pCommandBuffer = g_GfxFramework.GetRootCommandBuffer( cmdBufferIdx );
        g_GfxFramework.BeginFrame( cmdBufferIdx );
        pCommandBuffer->SetDescriptorPool( g_GfxFramework.GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView ) );
        pCommandBuffer->SetDescriptorPool( g_GfxFramework.GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler ) );
        pCommandBuffer->InvalidateMemory( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
                                        | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer );
        g_TextureShader.DrawMask( pCommandBuffer, &g_MaskGpuBuffer, g_SamplerSlot );
        g_GfxFramework.EndFrame( cmdBufferIdx );

        /// コマンドの実行と完了待ち
        auto pQueue = g_GfxFramework.GetQueue();
        pQueue->ExecuteCommand( pCommandBuffer, 0 );
        pQueue->Sync();
    }
}

nn::gfx::ResShaderFile* g_SampleShaderFile;
nn::gfx::ShaderCodeType g_SampleShaderCodeType;
int g_SampleShaderMatrixIndex;
int g_SampleShaderModulateIndex;
int g_SampleShaderSamplerIndex;
void InitializeSampleShader()
{
    g_SampleShaderFile = InitializeResShaderFile(&g_SampleShaderCodeType, SampleShaderPath);

    const nn::gfx::ResShaderVariation* pVariation
        = g_SampleShaderFile->GetShaderContainer()->GetResShaderVariation(0);
    const nn::gfx::ResShaderProgram* pProgram
        = pVariation->GetResShaderProgram(g_SampleShaderCodeType);
    const nn::gfx::Shader* pShader = pProgram->GetShader();

    g_SampleShaderMatrixIndex = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_Matrix" );
    g_SampleShaderModulateIndex = pShader->GetInterfaceSlot (nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_Modulate" );
    g_SampleShaderSamplerIndex = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_Tex" );

    NN_ASSERT( g_SampleShaderMatrixIndex >= 0 );
    NN_ASSERT( g_SampleShaderModulateIndex >= 0 );
    NN_ASSERT( g_SampleShaderSamplerIndex >= 0 );
}

nn::gfx::RasterizerState g_RasterizerState[nn::gfx::CullMode_End];
void InitializeRasterizerState()
{
    nn::gfx::RasterizerStateInfo info;
    info.SetDefault();
    info.SetPrimitiveTopologyType( nn::gfx::PrimitiveTopologyType_Triangle );
    info.SetScissorEnabled( true );
    info.SetDepthClipEnabled( false );
    for( int idxCullMode = 0; idxCullMode < nn::gfx::CullMode_End; ++idxCullMode )
    {
        nn::gfx::CullMode cullMode = nn::gfx::CullMode( idxCullMode );
        info.SetCullMode( cullMode );
        g_RasterizerState[idxCullMode].Initialize( g_GfxFramework.GetDevice(), info );
    }
}

enum DrawMode
{
    DrawMode_Opa,
    DrawMode_Xlu,
    DrawMode_End
};

nn::gfx::BlendState g_BlendState[DrawMode_End];
void InitializeBlendState()
{
    for( int idxDrawMode = 0; idxDrawMode < DrawMode_End; ++idxDrawMode )
    {
        DrawMode drawMode = DrawMode( idxDrawMode );
        nn::gfx::BlendTargetStateInfo targetInfo;
        {
            targetInfo.SetDefault();
        }

        if( drawMode == DrawMode_Xlu )
        {
            targetInfo.SetBlendEnabled( true );
            targetInfo.SetColorBlendFunction( nn::gfx::BlendFunction_Add );
            targetInfo.SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
            targetInfo.SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
        }

        nn::gfx::BlendState::InfoType info;
        info.SetDefault();
        info.SetBlendTargetStateInfoArray( &targetInfo, 1 );

        size_t size = nn::gfx::BlendState::GetRequiredMemorySize( info );
        g_pMemory.AlignUp( nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
        g_BlendState[idxDrawMode].SetMemory( g_pMemory.Get(), size );
        g_pMemory.Advance( size );
        g_BlendState[idxDrawMode].Initialize( g_GfxFramework.GetDevice(), info );
    }
}

nn::gfx::DepthStencilState g_DepthStencilState[DrawMode_End];
void InitializeDepthStencilState()
{
    for( int idxDrawMode = 0; idxDrawMode < DrawMode_End; ++idxDrawMode )
    {
        nn::gfx::DepthStencilStateInfo info;
        info.SetDefault();
        info.SetDepthTestEnabled( true );
        info.SetDepthWriteEnabled( idxDrawMode == DrawMode_Opa );
        g_DepthStencilState[idxDrawMode].Initialize( g_GfxFramework.GetDevice(), info );
    }
}

nn::gfx::VertexState g_VertexStateArray[Model_Count];
void InitializeVertexState( nn::gfx::VertexState* pVertexState, const nn::g3d::ModelObj* pModelObj )
{
    // 有効な mii のシェイプで頂点ステートを初期化します。
    int validMiiShapeIdx = -1;
    for( int idxShape = 0; idxShape < pModelObj->GetShapeCount(); ++idxShape )
    {
        if( pModelObj->IsShapeVisible( idxShape ) )
        {
            validMiiShapeIdx = idxShape;
            break;
        }
    }
    NN_ASSERT( validMiiShapeIdx != -1 );
    auto pResVertex = pModelObj->GetShape( validMiiShapeIdx )->GetResVertex();

    nn::gfx::VertexAttributeStateInfo attribs[nn::mii::DrawParam::AttributeType_End];
    nn::gfx::VertexBufferStateInfo buffers[nn::mii::DrawParam::AttributeType_End];
    for( int idxAttrib = 0; idxAttrib < pResVertex->GetVertexAttrCount(); ++idxAttrib )
    {
        auto pResVertexAttr = pResVertex->GetVertexAttr( idxAttrib );
        attribs[idxAttrib].SetDefault();
        attribs[idxAttrib].SetBufferIndex( idxAttrib );                    // mii のリソースは 1 attribute 1 buffer です。
        attribs[idxAttrib].SetFormat( pResVertexAttr->GetFormat() );
        attribs[idxAttrib].SetOffset( pResVertexAttr->GetOffset() );
        attribs[idxAttrib].SetShaderSlot( idxAttrib );                    // スロットはシェーダでロケーションを指定しています。
    }

    for( int idxBuff = 0; idxBuff < pResVertex->GetVertexBufferCount(); ++idxBuff )
    {
        buffers[idxBuff].SetDefault();
        buffers[idxBuff].SetStride( pResVertex->GetVertexBufferStride( idxBuff ) );
    }

    nn::gfx::VertexState::InfoType info;
    info.SetDefault();
    info.SetVertexAttributeStateInfoArray( attribs, pResVertex->GetVertexAttrCount() );
    info.SetVertexBufferStateInfoArray( buffers, pResVertex->GetVertexBufferCount() );

    size_t size = nn::gfx::VertexState::GetRequiredMemorySize( info );
    g_pMemory.AlignUp( nn::gfx::VertexState::RequiredMemoryInfo_Alignment );
    pVertexState->SetMemory( g_pMemory.Get(), size );
    g_pMemory.Advance( size );
    pVertexState->Initialize( g_GfxFramework.GetDevice(), info, NULL );
}

nn::gfx::Buffer g_ConstantBufferMatrix;
void InitializeConstantBuffers()
{
    auto pDevice = g_GfxFramework.GetDevice();
    nn::gfx::BufferInfo info;
    info.SetDefault();
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_ConstantBuffer );
    info.SetSize( sizeof( ConstantBufferMatrix ) );

    size_t alignment = nn::gfx::Buffer::GetBufferAlignment( pDevice, info );
    ptrdiff_t offset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, info.GetSize(), alignment );

    g_ConstantBufferMatrix.Initialize( pDevice, info
                                        , g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer )
                                        , offset
                                        , info.GetSize() );
}

void SetupConstantBuffers()
{
    ConstantBufferMatrix* pBuffer = g_ConstantBufferMatrix.Map<ConstantBufferMatrix>();

    /// 投影行列を設定
    float fovy = 3.14f / 4.0f;
    float aspect = static_cast<float>( g_GfxFramework.GetDisplayWidth() ) / static_cast<float>( g_GfxFramework.GetDisplayHeight() );
    nn::util::Matrix4x4fType projection = {};
    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projection, fovy, aspect, 0.01f, 10000.0f);
    nn::util::MatrixStore(&pBuffer->projection, projection);

    /// LookAt設定
    nn::util::Vector3fType camPos = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f,30.0f,115.0f);
    nn::util::Vector3fType camTarget = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f,30.0f,0.0f);
    nn::util::Matrix4x3fType modelView;
    nn::util::MatrixLookAtRightHanded(&modelView, camPos, camTarget, 0.0f);
    nn::util::MatrixStore(&pBuffer->modelView, modelView);

    g_ConstantBufferMatrix.Unmap();
}

void FinalizeMii()
{
    auto pDevice = g_GfxFramework.GetDevice();
    g_ConstantBufferMatrix.Finalize( pDevice );
    for( int idxModel = 0; idxModel < Model_Count; ++idxModel )
    {
        g_VertexStateArray[idxModel].Finalize( pDevice );
    }
    for( int idxDrawMode = 0; idxDrawMode < DrawMode_End; ++idxDrawMode )
    {
        g_BlendState[idxDrawMode].Finalize( pDevice );
        g_DepthStencilState[idxDrawMode].Finalize( pDevice );
    }
    for( int idxCullMode = 0; idxCullMode < nn::gfx::CullMode_End; ++idxCullMode )
    {
        g_RasterizerState[idxCullMode].Finalize( pDevice );
    }

    nn::gfx::ResShaderContainer* pContainer = g_SampleShaderFile->GetShaderContainer();
    for( int idxShaderVariation = 0; idxShaderVariation < pContainer->GetShaderVariationCount(); ++idxShaderVariation )
    {
        nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation( idxShaderVariation );
        pVariation->GetResShaderProgram(g_SampleShaderCodeType)->Finalize( pDevice );
    }
    pContainer->Finalize( pDevice );

    g_MaskGpuBuffer.Finalize( pDevice);
    g_FacelineGpuBuffer.Finalize( pDevice );
    g_Mask.Finalize( pDevice );
    g_Faceline.Finalize( pDevice );
    g_CharModel.Finalize (pDevice );
    g_Resource.Finalize( pDevice );
    g_TextureShader.Finalize( pDevice );

    pContainer = g_pTextureShaderFile->GetShaderContainer();
    for(int idxShaderVariation = 0; idxShaderVariation < pContainer->GetShaderVariationCount(); ++idxShaderVariation )
    {
        nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation( idxShaderVariation );
        pVariation->GetResShaderProgram(g_TextureShaderCodeType)->Finalize( pDevice );
    }
    g_pTextureShaderFile->GetShaderContainer()->Finalize( pDevice );
    g_Sampler.Finalize( pDevice );

    for(int idxModel = 0; idxModel < Model_Count; ++idxModel )
    {
        if( g_ModelObjPtrArray[idxModel] != nullptr )
        {
            g_ModelObjPtrArray[idxModel]->CleanupBlockBuffer( pDevice );
        }
    }
}

}

template <typename TType1, typename TType2> struct MiiBindTable
{
    nn::util::string_view    name;
    TType1    type1;
    TType2    type2;
};

typedef MiiBindTable<nn::mii::CharModel::DrawType, int> MaterialBindTable;
typedef MiiBindTable<nn::mii::CreateModelType, nn::mii::CreateNoseType> BoneBindTable;

template<typename TType1, typename TType2, size_t TTableSize>
int FindMiiBindTableIndex( const char* pTargetName, const MiiBindTable<TType1, TType2>( &miiBindTableArray )[TTableSize] )
{
    for( int idxTbl = 0; idxTbl  < static_cast<int>( TTableSize ); ++idxTbl  )
    {
        auto& miiBindTable = miiBindTableArray[idxTbl];
        if( miiBindTable.name == pTargetName )
        {
            return idxTbl;
        }
    }
    return -1;
};
template<typename TType1, typename TType2>
int FindMiiBindTableIndex( const char* pTargetName, const MiiBindTable<TType1, TType2>* miiBindTableArray, size_t miiBindTableArraySize )
{
    for( int idxTbl = 0; idxTbl  < static_cast<int>( miiBindTableArraySize ); ++idxTbl  )
    {
        auto& miiBindTable = miiBindTableArray[idxTbl];
        if( miiBindTable.name == pTargetName )
        {
            return idxTbl;
        }
    }
    return -1;
};

// render_info, shader_param のデフォルト値登録用コールバックパラメータ
class MaterialCallbackParam
{
public:
    MaterialCallbackParam() :
        renderInfoCullingName( "culling" ),
        shaderParamModulateTypeId( "mii_modulate_type" ),
        shaderParamConstColorId( "mii_const_color" )
    {
    }

    const char* renderInfoCullingName;
    const char* shaderParamModulateTypeId;
    const char* shaderParamConstColorId;

};

class DrawParamCallbckparam
{
public:
    DrawParamCallbckparam( const MaterialBindTable* materialBindTableArray, size_t materialBindTableArraySize,
                            const BoneBindTable* boneBindTableArray, size_t boneBindTableArraySize )
        : charModelMaterialBindTableArray( materialBindTableArray )
        , charModelMaterialBindTableArraySize( materialBindTableArraySize )
        , charModelBoneBindTableArray( boneBindTableArray )
        , charModelBoneBindTableArraySize( boneBindTableArraySize )
    {
    }
    const MaterialBindTable* charModelMaterialBindTableArray;
    size_t charModelMaterialBindTableArraySize;
    const BoneBindTable* charModelBoneBindTableArray;
    size_t charModelBoneBindTableArraySize;
};

// マテリアル名と mii のパーツを紐づけるテーブルです。
const MaterialBindTable g_CharModelMaterialBindTableArray[] =
{
    { "beard",                nn::mii::CharModel::DrawType_Beard, 0 },
    { "faceline",            nn::mii::CharModel::DrawType_Faceline, 0 },
    { "forehead",    nn::mii::CharModel::DrawType_Forehead, 0 },
    { "hair",        nn::mii::CharModel::DrawType_Hair, 0 },
    { "hat",            nn::mii::CharModel::DrawType_Hat, 0 },
    { "nose",        nn::mii::CharModel::DrawType_Nose, 0 },
    { "noseline",    nn::mii::CharModel::DrawType_Noseline, 0 },
    { "glass",    nn::mii::CharModel::DrawType_Glass, 0 },
    { "mask",    nn::mii::CharModel::DrawType_Mask, 0 },
};
const size_t g_CharModelMaterialBindTableArraySize = sizeof( g_CharModelMaterialBindTableArray ) / sizeof( MaterialBindTable );

// ボーン名と CreateModel/NoseType を紐づけるテーブルです。
const BoneBindTable g_CharModelBoneBindTableArray[] =
{
    { "mii_common",        nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "beard",            nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "glass",            nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "forehead_normal",    nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "forehead_hat",        nn::mii::CreateModelType_Hat, nn::mii::CreateNoseType_Normal },
    { "hair_normal",        nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "hair_hat",            nn::mii::CreateModelType_Hat, nn::mii::CreateNoseType_Normal },
    { "hat_normal",            nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "hat_hat",            nn::mii::CreateModelType_Hat, nn::mii::CreateNoseType_Normal },
    { "nose_normal",        nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "nose_flatten",        nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Flatten },
    { "noseline_normal",    nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "noseline_flatten",    nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Flatten },
};
const size_t g_CharModelBoneBindTableArraySize = sizeof( g_CharModelBoneBindTableArray ) / sizeof( BoneBindTable );

// 初期状態で描画するボーンのビジビリティです。
const char* g_VisibleMiiBoneNameArray[] =
{
    "mii_common",
    "beard",
    "forehead_hat",
    "hat_hat",
    "hair_hat",
    "nose_normal",
    "noseline_normal",
    "glass",
};
const size_t g_VisibleMiiBoneNameArraySize = sizeof( g_VisibleMiiBoneNameArray ) / sizeof( char* );

// DrawParam 取得コールバック
void DrawParamCallback( nn::g3d::mii::DrawParamCallbackOutput* pDrawParamCallbackOutput, const nn::g3d::ResModel* pModel,
                        const int shapeIndex, const nn::mii::CharModel* pCharModel, void* pCallbackParam )
{
    auto materialBindTableArray = reinterpret_cast<DrawParamCallbckparam*>( pCallbackParam )->charModelMaterialBindTableArray;
    size_t materialBindTableArraySize = reinterpret_cast<DrawParamCallbckparam*>( pCallbackParam )->charModelMaterialBindTableArraySize;
    auto boneBindTableArray = reinterpret_cast<DrawParamCallbckparam*>( pCallbackParam )->charModelBoneBindTableArray;
    size_t boneBindTableArraySize = reinterpret_cast<DrawParamCallbckparam*>( pCallbackParam )->charModelBoneBindTableArraySize;
    auto pShape = pModel->GetShape( shapeIndex );
    const char* pMatName = pModel->GetMaterial( pShape->GetMaterialIndex() )->GetName();
    int idxCharModelMaterialBindTable = FindMiiBindTableIndex( pMatName, materialBindTableArray, materialBindTableArraySize );

    // Mii のパーツではない場合
    if( idxCharModelMaterialBindTable == -1 )
    {
        pDrawParamCallbackOutput->isMiiFace = false;
        pDrawParamCallbackOutput->pDrawParam = nullptr;
        return;
    }

    // Mii のパーツの場合、DrawParam を取得します。
    const char* pBoneName = pModel->GetSkeleton()->GetBoneName( pShape->GetBoneIndex() );
    int idxCharModelBoneBindTable = FindMiiBindTableIndex( pBoneName, boneBindTableArray, boneBindTableArraySize );
    NN_ASSERT( idxCharModelBoneBindTable != -1 );
    auto& boneBindTbl = boneBindTableArray[idxCharModelBoneBindTable];
    auto& matBindTbl = materialBindTableArray[idxCharModelMaterialBindTable];
    pDrawParamCallbackOutput->isMiiFace = true;
    pDrawParamCallbackOutput->pDrawParam = pCharModel->GetDrawParam( matBindTbl.type1, boneBindTbl.type1, boneBindTbl.type2, matBindTbl.type2 );
    pDrawParamCallbackOutput->pBoundingBox = &pCharModel->GetBoundingBox( boneBindTbl.type1, boneBindTbl.type2 );
}

// render_info, shader_param のデフォルト値登録用のコールバック関数
void MaterialCallback( nn::g3d::ResMaterial* pMaterial, const nn::mii::DrawParam* pDrawParam, void* pUserParam )
{
    auto pCbParam = static_cast<MaterialCallbackParam*>( pUserParam );

    const int ShaderParamCount = pMaterial->GetShaderParamCount();
    for( int idxShaderParam = 0; idxShaderParam < ShaderParamCount; ++idxShaderParam )
    {
        auto pShaderParam = pMaterial->GetShaderParam( idxShaderParam );
        const char* pShaderParamId = pShaderParam->GetId();

        if( pCbParam->shaderParamModulateTypeId == std::string( pShaderParamId ) )
        {
            NN_SDK_ASSERT( pShaderParam->GetSrcSize() == sizeof( int32_t ), "Modulate type param size is not consistent." );
            void* pSrcParamBuffer = pMaterial->GetSrcParam( idxShaderParam );
            *static_cast<int32_t*>(pSrcParamBuffer) = static_cast<int32_t>(pDrawParam->GetModulateType());
        }
        else if( pCbParam->shaderParamConstColorId == std::string( pShaderParamId ) )
        {
            const size_t ConstColorSize = sizeof(float) * 9;    // vec3 * 3
            NN_SDK_ASSERT( pShaderParam->GetSrcSize() == ConstColorSize, "Modulate type param size is not consistent." );
            NN_UNUSED( ConstColorSize );
            char* pSrcParamBuffer = reinterpret_cast<char*>( pMaterial->GetSrcParam(idxShaderParam) );
            for( int idxConstColor = 0; idxConstColor < nn::mii::DrawParam::ConstantColorNum; ++idxConstColor, pSrcParamBuffer += sizeof( float ) * 4 )
            {
                const nn::mii::Color3* pConstColor = pDrawParam->GetConstantColor( idxConstColor );
                if( pConstColor )
                {
                    memcpy( pSrcParamBuffer, pConstColor, sizeof( nn::mii::Color3 ) );
                }
            }
        }
    }
    const int RenderInfoCount = pMaterial->GetRenderInfoCount();
    for( int idxRenderInfo = 0; idxRenderInfo < RenderInfoCount; ++idxRenderInfo )
    {
        auto pRenderInfo = pMaterial->GetRenderInfo( idxRenderInfo );
        const char* pRenderInfoName = pRenderInfo->GetName();
        if( pRenderInfoName == std::string( pCbParam->renderInfoCullingName ) )
        {
            NN_SDK_ASSERT( nn::g3d::ResRenderInfo::Type_Int == pRenderInfo->GetType(), "RenderInfo culling type mismatch." );
            nn::gfx::CullMode cullMode = pDrawParam->GetCullMode();    // none, front, back
            *pRenderInfo->ToData().intValueArray.Get() = static_cast<int>(cullMode);
        }
    }
}

nn::g3d::TextureRef TextureBindCallback( const char* name, void* pUserData )
{
    auto pCharModel = static_cast<nn::mii::CharModel*>( pUserData );

    const char* texNameArray[] =
    {
        "faceline",
        "hat",
        "mask",
        "noseline",
        "glass",
    };
    const int TexCount = sizeof( texNameArray ) / sizeof( char* );

    nn::g3d::TextureRef textureRef;
    for( int idxTex = 0; idxTex < TexCount; ++idxTex )
    {
        if( name == std::string(texNameArray[idxTex]) )
        {
            int maskSlot = nn::mii::Expression_Min;    // "mask" 以外では MaskSlot は無効なので適当な値を入れます

            if( name == std::string( texNameArray[2] ) )
            {
                maskSlot = 1;    // "mask" を切り替えるアニメーションの適用で上書きされます
            }
            nn::mii::CharModel::TextureType miiTexType = static_cast<nn::mii::CharModel::TextureType>( idxTex );
            auto* pTextureView = pCharModel->GetTextureView( miiTexType, maskSlot );
            nn::gfx::DescriptorSlot descSlot = pCharModel->GetTextureDescriptorSlot( miiTexType, maskSlot );
            textureRef.SetTextureView( pTextureView );
            textureRef.SetDescriptorSlot( descSlot );
            return textureRef;
        }
    }
    return textureRef;
}

nn::g3d::ResFile* g_pResFile = nullptr;
nn::util::BitArray    g_VisibleBoneBit;    // visible boneのリストこの中からvisibleにするボーンを選択しないといけない。
void* g_pVisibleBoneBitBuff = nullptr;
size_t g_VisibleBoneBitBuffSize = 0;
void InitializeResFile()
{
    // Mii 用のテンプレート ResFile をロード
    {
        nn::fs::FileHandle handle;
        nn::Result result = nn::fs::OpenFile( &handle, "Contents:/MiiSeparateBone.bfres", nn::fs::OpenMode_Read );
        NN_ASSERT( result.IsSuccess() );

        int64_t fileSize = 0;
        nn::fs::GetFileSize( &fileSize, handle );
        g_pMemory.AlignUp( 4 * 1024 );    // ResFile の最大アライメントにアラインアップ
        void* pData = g_pMemory.Get();
        g_pMemory.Advance( static_cast<ptrdiff_t>( fileSize ) );
        result = nn::fs::ReadFile( handle, 0, pData, static_cast<size_t>( fileSize ) );
        NN_ASSERT( result.IsSuccess() );

        g_pResFile = nn::g3d::ResFile::ResCast( pData );
        NN_ASSERT( g_pResFile->IsValid( g_pResFile ) );
        g_pResFile->Setup( g_GfxFramework.GetDevice() );
        g_pResFile->BindTexture(TextureBindCallback, &g_CharModel );

        // gfx::SamplerDescriptorSlot, gfx::Sampler の登録
        for( int idxModel = 0; idxModel < g_pResFile->GetModelCount(); ++idxModel )
        {
            auto pModel = g_pResFile->GetModel( idxModel );
            for( int idxMaterial = 0; idxMaterial < pModel->GetMaterialCount(); ++idxMaterial )
            {
                auto pMaterial = pModel->GetMaterial( idxMaterial );
                if( pMaterial->GetSamplerCount() )
                {
                    const int IdxSampler = 0;    // SimpleMii の shader では uniform sampler は 1 つ
                    pMaterial->SetSamplerDescriptorSlot( IdxSampler, g_SamplerSlot );
                }
                pMaterial->SetRawParamSize( sizeof( ConstantBufferModulate ) );
            }
        }
        nn::fs::CloseFile(handle);
    }

    // CharModel を ResFile にアサインします。
    {
        auto pModel = g_pResFile->GetModel( Model_MiiSeparateBone );
        MaterialCallbackParam renderInfoShaderParamCbFuncParam;
        DrawParamCallbckparam drawParamCallbackParam( g_CharModelMaterialBindTableArray, g_CharModelMaterialBindTableArraySize,
                                                    g_CharModelBoneBindTableArray, g_CharModelBoneBindTableArraySize );
        nn::g3d::mii::AssignCharModel( pModel, &g_CharModel, MaterialCallback, &renderInfoShaderParamCbFuncParam, DrawParamCallback, &drawParamCallbackParam );
    }

    // 利用可能なパーツの種類を保存します。
    {
        auto pSkeleton = g_pResFile->GetModel( Model_MiiSeparateBone )->GetSkeleton();
        const int boneCount = pSkeleton->GetBoneCount();
        g_VisibleBoneBitBuffSize = nn::util::BitArray::CalculateWorkMemorySize( boneCount );
        g_pVisibleBoneBitBuff = malloc( g_VisibleBoneBitBuffSize );
        NN_ASSERT_NOT_NULL( g_pVisibleBoneBitBuff );

        g_VisibleBoneBit.ResetWorkMemory( g_pVisibleBoneBitBuff, g_VisibleBoneBitBuffSize, boneCount );
        g_VisibleBoneBit.reset();
        for( int idxBone = 0; idxBone < pSkeleton->GetBoneCount(); ++idxBone )
        {
            auto pBone = pSkeleton->GetBone( idxBone );
            auto& boneData = pBone->ToData();
            g_VisibleBoneBit.set( idxBone, boneData.flag & nn::g3d::ResBone::Flag_Visibility );
        }
    }
}

void FinalizeResFile()
{
    // nn::g3d::mii::AssignCharModel() により頂点、インデックスバッファが一部差し替えられているので ResFile が元から持つものに戻す。
    for( int idxModel = 0; idxModel < g_pResFile->GetModelCount(); ++idxModel )
    {
        auto pResModel = g_pResFile->GetModel( idxModel );
        pResModel->Reset();
    }
    g_pResFile->Cleanup( g_GfxFramework.GetDevice() );
    g_pResFile = nullptr;

    if( g_pVisibleBoneBitBuff != nullptr )
    {
        free( g_pVisibleBoneBitBuff );
        g_pVisibleBoneBitBuff = nullptr;
        g_VisibleBoneBitBuffSize = 0;
    }
}

void InitializeModelObj( nn::g3d::ResFile* pFile )
{
    for( int idxModel = 0; idxModel < pFile->GetModelCount(); ++idxModel )
    {
        auto pModel = pFile->GetModel( idxModel );
        nn::g3d::ModelObj::Builder builder( pModel );

        builder.CalculateMemorySize();
        size_t alignment = nn::g3d::ModelObj::Alignment_Buffer;
        size_t bufferSize = builder.GetWorkMemorySize();
        size_t objectSize = nn::util::align_up(sizeof(nn::g3d::ModelObj), alignment);
        g_pMemory.AlignUp( alignment );
        g_ModelObjPtrArray[idxModel] = reinterpret_cast<nn::g3d::ModelObj*>( g_pMemory.Get() );
        g_pMemory.Advance( objectSize + bufferSize );
        NN_ASSERT_NOT_NULL( g_ModelObjPtrArray[idxModel] );
        new( g_ModelObjPtrArray[idxModel] ) nn::g3d::ModelObj();

        bool success = false;
        success = builder.Build(g_ModelObjPtrArray[idxModel], nn::g3d::AddOffset( g_ModelObjPtrArray[idxModel], objectSize ), bufferSize);
        ( void ) success;
        NN_ASSERT( success == true );

        auto pDevice = g_GfxFramework.GetDevice();
        size_t size = g_ModelObjPtrArray[idxModel]->CalculateBlockBufferSize( pDevice );
        size_t memoryPoolAlignment = g_ModelObjPtrArray[idxModel]->GetBlockBufferAlignment( g_GfxFramework.GetDevice() );
        ptrdiff_t offset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, size, memoryPoolAlignment );
        success = g_ModelObjPtrArray[idxModel]->SetupBlockBuffer(pDevice, g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer ), offset, size );
        ( void ) success;
        NN_ASSERT( success == true );

        // GPU が参照する Material UniformBlock の値を更新します。
        for(int idxMat = 0, materialCount = g_ModelObjPtrArray[idxModel]->GetMaterialCount(); idxMat < materialCount; ++idxMat)
        {
            nn::g3d::MaterialObj* pMaterialObj = g_ModelObjPtrArray[idxModel]->GetMaterial(idxMat);
            if( pMaterialObj->GetShaderParamCount() )
            {
                ConstantBufferModulate* pMaterialBlock = pMaterialObj->GetMaterialBlock(0)->Map<ConstantBufferModulate>();
                memcpy( &pMaterialBlock->modulateType, pMaterialObj->GetShaderParam( 0 ), sizeof( int ) );
                memcpy( &pMaterialBlock->constantColor, pMaterialObj->GetShaderParam( 1 ), sizeof( float ) * 9 );
                pMaterialObj->GetMaterialBlock(0)->Unmap();
            }
        }
    }

    // 描画するパーツのボーンビジビリティを設定します。
    for( int idxBone = 0; idxBone < g_ModelObjPtrArray[Model_MiiSeparateBone]->GetSkeleton()->GetBoneCount(); ++idxBone )
    {
        g_ModelObjPtrArray[Model_MiiSeparateBone]->SetBoneVisible( idxBone, false );
    }
    for( int idxVisibleBone = 0; idxVisibleBone < g_VisibleMiiBoneNameArraySize; ++idxVisibleBone )
    {
        int idxBone = g_ModelObjPtrArray[Model_MiiSeparateBone]->GetSkeleton()->FindBoneIndex( g_VisibleMiiBoneNameArray[idxVisibleBone] );
        NN_ASSERT( idxBone != -1 );
        if( g_VisibleBoneBit.test( idxBone ) )
        {
            g_ModelObjPtrArray[Model_MiiSeparateBone]->SetBoneVisible( idxBone, true );
        }
    }

    // 帽子のビジビリティの設定
    for( int idxBone = 0; idxBone < g_ModelObjPtrArray[Model_Cap]->GetSkeleton()->GetBoneCount(); ++idxBone )
    {
        g_ModelObjPtrArray[Model_Cap]->SetBoneVisible( idxBone, true );
    }
}

void InitializeMaterialAnimObj( nn::g3d::MaterialAnimObj* pMaterialAnimObj, const nn::g3d::ResFile* pFile, MaterialAnim animIdx )
{
    NN_SDK_ASSERT( pMaterialAnimObj );
    nn::g3d::MaterialAnimObj::Builder builder;
    builder.Reserve( g_ModelObjPtrArray[Model_MiiSeparateBone]->GetResource() );
    builder.Reserve( pFile->GetMaterialAnim( 0 ) );
    builder.CalculateMemorySize();
    size_t bufferSize = builder.GetWorkMemorySize();

    g_pMemory.AlignUp( nn::g3d::MaterialAnimObj::Alignment_Buffer );
    void* pBuffer = g_pMemory.Get();
    NN_ASSERT_NOT_NULL( pBuffer );
    g_pMemory.Advance( bufferSize );
    bool success = builder.Build( pMaterialAnimObj, pBuffer, bufferSize );
    NN_UNUSED( success );
    NN_ASSERT( success );
    const nn::g3d::ResMaterialAnim* pMaterialAnim = pFile->FindMaterialAnim( g_MaterialAnimNameArray[animIdx] );
    NN_ASSERT_NOT_NULL( pMaterialAnim );
    pMaterialAnimObj->SetResource( pMaterialAnim );
    pMaterialAnimObj->Bind( g_ModelObjPtrArray[Model_MiiSeparateBone] );

    // MaterialanimObj へテクスチャの登録をします
    NN_ASSERT( pMaterialAnimObj->GetTextureCount() <= nn::mii::GetExpressionCount( ExpressionFlags ) );
    for( int idxTex = 0; idxTex < pMaterialAnimObj->GetTextureCount(); ++idxTex )
    {
        nn::g3d::TextureRef textureRef;
        auto pTextureView = g_CharModel.GetTextureView( nn::mii::CharModel::TextureType::TextureType_Mask, idxTex );
        NN_ASSERT_NOT_NULL( pTextureView );
        textureRef.SetTextureView( pTextureView );
        textureRef.SetDescriptorSlot( const_cast<nn::gfx::DescriptorSlot&>( g_CharModel.GetTextureDescriptorSlot( nn::mii::CharModel::TextureType::TextureType_Mask, idxTex ) ) );
        pMaterialAnimObj->SetTexture( idxTex, textureRef );
    }
}

void InitializeBoneVisibilityAnimObj( nn::g3d::BoneVisibilityAnimObj* pMiiBoneVisAnimObj, const nn::g3d::ModelObj* pModelObj, const nn::g3d::ResFile* pFile, BoneVisibilityAnim animIdx )
{
    NN_ASSERT_NOT_NULL( pMiiBoneVisAnimObj );
    NN_ASSERT_NOT_NULL( pModelObj );

    nn::g3d::BoneVisibilityAnimObj::Builder builder;
    builder.Reserve( pModelObj->GetResource() );
    auto pBoneVisAnim = pFile->FindBoneVisibilityAnim( g_BoneAnimNameArray[animIdx] );
    NN_ASSERT_NOT_NULL( pBoneVisAnim );
    builder.Reserve( pBoneVisAnim );
    builder.CalculateMemorySize();
    size_t bufferSize = builder.GetWorkMemorySize();

    g_pMemory.AlignUp( nn::g3d::MaterialAnimObj::Alignment_Buffer );
    void* pBuffer = g_pMemory.Get();
    NN_ASSERT_NOT_NULL( pBuffer );
    g_pMemory.Advance( bufferSize );
    bool success = builder.Build( pMiiBoneVisAnimObj, pBuffer, bufferSize );
    NN_ASSERT( success );
    NN_UNUSED( success );

    NN_ASSERT_NOT_NULL( pBoneVisAnim );
    pMiiBoneVisAnimObj->SetResource( pBoneVisAnim );
    pMiiBoneVisAnimObj->Bind( pModelObj );
}

// モデル内のシェイプインデクスを不透明から順に並べます。
void SortMaterial( std::vector<int>* pShapeIndexArray, const nn::g3d::ModelObj* pModelObj )
{
    NN_ASSERT_NOT_NULL( pShapeIndexArray );
    NN_ASSERT_NOT_NULL( pModelObj );

    auto& shapeIdxArray = *pShapeIndexArray;
    shapeIdxArray.reserve( pModelObj->GetShapeCount() );
    for( int idxShape = 0; idxShape < pModelObj->GetShapeCount(); ++idxShape )
    {
        auto pShapeObj = pModelObj->GetShape( idxShape );
        auto pMaterialObj = pModelObj->GetMaterial( pShapeObj->GetMaterialIndex() );
        auto pDrawMode = pMaterialObj->GetResource()->FindRenderInfo( "opacity" )->GetInt();
        if( ( pDrawMode == nullptr ) || ( ( pDrawMode != nullptr ) && ( *pDrawMode == DrawMode_Opa ) ) ) // "opacity"を持っていない場合は不透明とします。
        {
            shapeIdxArray.insert( shapeIdxArray.begin(), idxShape );
        }
        else
        {
            shapeIdxArray.emplace_back( idxShape );
        }
    }
}

//  計算処理コールバック
struct CalculateCallbackUserData
{
    nn::g3d::MaterialAnimObj* pMaterialAnimObj;
    nn::g3d::BoneVisibilityAnimObj* pMiiBoneVisAnimObj;
    nn::g3d::BoneVisibilityAnimObj* pCapBoneVisAnimObj;
};

void CalculateCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    CalculateCallbackUserData* pCallbackUserData = reinterpret_cast<CalculateCallbackUserData*>(pUserData);

    // CPU 処理
    if (pCallbackUserData->pMaterialAnimObj->GetResource())
    {
        pCallbackUserData->pMaterialAnimObj->Calculate();
        pCallbackUserData->pMaterialAnimObj->ApplyTo(g_ModelObjPtrArray[Model_MiiSeparateBone]);
        pCallbackUserData->pMaterialAnimObj->GetFrameCtrl().UpdateFrame();
    }
    if (pCallbackUserData->pMiiBoneVisAnimObj->GetResource())
    {
        pCallbackUserData->pMiiBoneVisAnimObj->Calculate();
        pCallbackUserData->pMiiBoneVisAnimObj->ApplyTo(g_ModelObjPtrArray[Model_MiiSeparateBone]);
        pCallbackUserData->pMiiBoneVisAnimObj->GetFrameCtrl().UpdateFrame();
    }
    if (pCallbackUserData->pCapBoneVisAnimObj->GetResource())
    {
        pCallbackUserData->pCapBoneVisAnimObj->Calculate();
        pCallbackUserData->pCapBoneVisAnimObj->ApplyTo(g_ModelObjPtrArray[Model_Cap]);
        pCallbackUserData->pCapBoneVisAnimObj->GetFrameCtrl().UpdateFrame();
    }
}

//  コマンド生成コールバック
void MakeCommandCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    std::vector<int>* shapeIdxArrayArray = reinterpret_cast<std::vector<int>*>(pUserData);

    g_GfxFramework.BeginFrame(bufferIndex);
    auto pCommandBuffer = g_GfxFramework.GetRootCommandBuffer(bufferIndex);

    /// クリア
    nn::gfx::ColorTargetView* pColorTarget = g_GfxFramework.GetColorTargetView();
    pCommandBuffer->ClearColor(pColorTarget, 0.3f, 0.1f, 0.1f, 0.1f, NULL);
    auto pDepthStencilView = g_GfxFramework.GetDepthStencilView();
    pCommandBuffer->ClearDepthStencil(pDepthStencilView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);

    pCommandBuffer->SetRenderTargets(1, &pColorTarget, pDepthStencilView);
    pCommandBuffer->SetViewportScissorState(g_GfxFramework.GetViewportScissorState());

    for (int idxModel = 0; idxModel < Model_Count; ++idxModel)
    {
        for (auto idxShape : shapeIdxArrayArray[idxModel])
        {
            if (!g_ModelObjPtrArray[idxModel]->IsShapeVisible(idxShape))
            {
                continue;
            }
            auto pShapeObj = g_ModelObjPtrArray[idxModel]->GetShape(idxShape);
            auto pMaterialObj = g_ModelObjPtrArray[idxModel]->GetMaterial(pShapeObj->GetMaterialIndex());

            /// 描画ステート設定
            nn::gfx::CullMode cullMode = nn::gfx::CullMode_Back;
            auto pRenderInfo = pMaterialObj->GetResource()->FindRenderInfo("culling");
            if (pRenderInfo)
            {
                cullMode = static_cast<nn::gfx::CullMode>(*pRenderInfo->GetInt());
            }
            pCommandBuffer->SetRasterizerState(&g_RasterizerState[cullMode]);

            DrawMode drawMode = DrawMode_Opa;
            pRenderInfo = pMaterialObj->GetResource()->FindRenderInfo("opacity");
            if (pRenderInfo)
            {
                drawMode = static_cast<DrawMode>(*pRenderInfo->GetInt());
            }

            pCommandBuffer->SetBlendState(&g_BlendState[drawMode]);
            pCommandBuffer->SetDepthStencilState(&g_DepthStencilState[drawMode]);
            pCommandBuffer->SetVertexState(&g_VertexStateArray[idxModel]);
            pCommandBuffer->SetShader(g_SampleShaderFile->GetShaderContainer()->GetResShaderVariation(0)->GetResShaderProgram(g_SampleShaderCodeType)->GetShader(), nn::gfx::ShaderStageBit_All);

            /// MVP 行列のセット
            nn::gfx::GpuAddress gpuAddress;
            g_ConstantBufferMatrix.GetGpuAddress(&gpuAddress);
            pCommandBuffer->SetConstantBuffer(g_SampleShaderMatrixIndex, nn::gfx::ShaderStage_Vertex, gpuAddress, sizeof(ConstantBufferMatrix));

            /// 描画する
            // Material Uniform Block の設定
            if (pMaterialObj->GetMaterialBlockSize() != 0)
            {
                pMaterialObj->GetMaterialBlock(0)->GetGpuAddress(&gpuAddress);
                pCommandBuffer->SetConstantBuffer(g_SampleShaderModulateIndex, nn::gfx::ShaderStage_Pixel, gpuAddress, pMaterialObj->GetMaterialBlockSize());
            }

            for (int idxSampler = 0; idxSampler < pMaterialObj->GetSamplerCount(); ++idxSampler)
            {
                pCommandBuffer->SetTextureAndSampler(g_SampleShaderSamplerIndex, nn::gfx::ShaderStage_Pixel
                    , pMaterialObj->GetTexture(idxSampler).GetDescriptorSlot()
                    , pMaterialObj->GetSampler(idxSampler).GetDescriptorSlot());
            }

            for (int idxBuff = 0; idxBuff < pShapeObj->GetVertexBufferCount(); ++idxBuff)
            {
                auto pResVtx = pShapeObj->GetResVertex();
                auto pBuffer = pShapeObj->GetVertexBuffer(idxBuff);
                NN_ASSERT_NOT_NULL(pResVtx);
                NN_ASSERT_NOT_NULL(pBuffer);
                pBuffer->GetGpuAddress(&gpuAddress);
                pCommandBuffer->SetVertexBuffer(idxBuff, gpuAddress, pResVtx->GetVertexBufferStride(idxBuff), pResVtx->GetVertexBufferInfo(idxBuff)->GetSize());
            }
            pShapeObj->GetResMesh()->Draw(pCommandBuffer, 1);
        }
    }

    g_GfxFramework.EndFrame(bufferIndex);
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
//-----------------------------------------------------------------------------
// nninitStartup() is invoked before calling nnMain().
//
extern "C" void nninitStartup()
{
    const size_t ApplicationMemorySize = 512 * 1024 * 1024;
    const size_t MallocMemorySize = 256 * 1024 * 1024;
    nn::Result result = nn::os::SetMemoryHeapSize(ApplicationMemorySize);
    NN_ASSERT(result.IsSuccess());
    uintptr_t address;
    result = nn::os::AllocateMemoryBlock(&address, MallocMemorySize);
    NN_ASSERT(result.IsSuccess());
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), MallocMemorySize);
}
#endif

//
//  Main Function
//  メイン関数です。
//

extern "C" void nnMain()
{
    InitializeFs();

    // フレームワーク初期化
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem( 1024 * 1024 * 32 );
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetBufferCount(2);
    fwInfo.SetSwapChainBufferCount(2);
    fwInfo.SetMemoryPoolSize( nns::gfx::GraphicsFramework::MemoryPoolType_Data, 1024 * 1024 * 32 );
    fwInfo.SetMemoryPoolSize( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 1024 * 1024 * 32 );
    g_GfxFramework.Initialize( fwInfo );

    InitializeResources();

    /// Mii用のサンプラを準備する
    InitializeSampler();

    /// CharInfoをファイルから読み込む
    LoadCharInfo();

    /// Faceline,Maskテクスチャ作成用シェーダーを初期化
    InitializeTextureShader();

    //// Miiリソースを初期化
    InitializeMiiResource();

    /// CharModelを初期化
    InitializeCharModel();

    /// Facelineを初期化
    InitializeFaceline();

    /// Maskを初期化
    InitializeMask();

    /// デスクリプタスロットの割り当て
    BindDescriptorSlot();

    /// Faceline,Maskを描画する
    InitializeFacelineGpuBuffer();
    InitializeMaskGpuBuffer();
    /// テクスチャへ描画後はg_Faceline,g_Maskを破棄できます。
    RenderFacelineTexture();
    RenderMaskTexture();

    // ResFile、ModelObj の構築
    InitializeResFile();
    InitializeModelObj( g_pResFile );

    // MaterialAnimObj の構築
    nn::g3d::MaterialAnimObj    materialAnimObj;
    InitializeMaterialAnimObj( &materialAnimObj, g_pResFile, MaterialAnim_FaceExpressionChange );
    nn::g3d::BoneVisibilityAnimObj miiBoneVisAnimObj;
    InitializeBoneVisibilityAnimObj( &miiBoneVisAnimObj, g_ModelObjPtrArray[Model_MiiSeparateBone], g_pResFile, BoneVisibilityAnim_CharModelTypeChange );
    nn::g3d::BoneVisibilityAnimObj capBoneVisAnimObj;
    InitializeBoneVisibilityAnimObj( &capBoneVisAnimObj, g_ModelObjPtrArray[Model_Cap], g_pResFile, BoneVisibilityAnim_CapWearOnOff );

    /// Mii描画用のシェーダーを初期化
    InitializeSampleShader();

    /// Mii描画用の処理を初期化
    InitializeRasterizerState();
    InitializeBlendState();
    InitializeDepthStencilState();
    for( int idxModel = 0; idxModel < Model_Count; ++idxModel )
    {
        InitializeVertexState( &g_VertexStateArray[idxModel], g_ModelObjPtrArray[idxModel] );
    }
    InitializeConstantBuffers();

    /// 描画のためにコンスタントバッファを設定する
    SetupConstantBuffers();

    /// バッファーオーバーフローチェック
    NN_ASSERT( g_pMemoryHeap.Distance( g_pMemory.Get() ) < MemoryHeapSize );

    // マテリアルソート、簡易的にモデル内でソートします。
    std::vector<int> shapeIdxArrayArray[Model_Count];
    SortMaterial( &shapeIdxArrayArray[Model_Cap], g_ModelObjPtrArray[Model_Cap] );
    SortMaterial( &shapeIdxArrayArray[Model_MiiSeparateBone], g_ModelObjPtrArray[Model_MiiSeparateBone] );

    // フレームワーク設定
    CalculateCallbackUserData calculateUserData;
    calculateUserData.pMaterialAnimObj = &materialAnimObj;
    calculateUserData.pMiiBoneVisAnimObj = &miiBoneVisAnimObj;
    calculateUserData.pCapBoneVisAnimObj = &capBoneVisAnimObj;
    g_GfxFramework.SetCalculateCallback(CalculateCallback, &calculateUserData);
    g_GfxFramework.SetMakeCommandCallback(MakeCommandCallback, shapeIdxArrayArray);

    // 毎フレームのレンダリング
    for( int frame = 0; frame < 400; ++frame )
    {
        g_GfxFramework.ProcessFrame();
    }
    g_GfxFramework.QueueFinish();

    FinalizeMii();
    FinalizeResFile();
    FinalizeResources();
    FinalizeFs();
} //NOLINT(impl/function_size)
