﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/nfc.h>

namespace nns { namespace nfc { namespace mifare {

    /*!
        @brief      NFC ライブラリの処理に非同期に行うクラスです。

        @details
                    NFC ライブラリで提供される API のほとんどは長時間処理をブロックしてしまいます。
                    このクラスは NFC ライブラリの API を別スレッドで実行する実装のサンプルです。

                    実行する処理に対応するコマンド識別子を @ref Enqueue() で登録し、
                    @ref Run() を呼びだすことで登録した処理をバックグラウンドで実行します。
                    登録できるコマンド識別子の最大数は @ref CommandCountMax です。

                    登録した処理が全て完了した場合、あるいは何らかのエラーが発生して中断された場合には
                    @ref Run() の引数に指定したイベントにシグナルが発行されます。
                    いくつの処理が実行されたかと最後の処理結果は、
                    @ref GetExecutedCount() と @ref GetLastResult() で取得することができます。

                    本クラスのメソッドはスレッドセーフではありませんので、一つのスレッドから呼び出すようにしてください。
     */
    class TagAccessorBase
    {
        NN_DISALLOW_COPY( TagAccessorBase );
        NN_DISALLOW_MOVE( TagAccessorBase );

    public:

        //! @ref Run() の前に登録できるコマンド識別子の最大数です。
        static const int CommandCountMax = 16;

        //! 無効なコマンド識別子です。
        static const uint32_t Command_None = 0;

        //! 実行する処理に渡されるパラメータに格納できるポインタ変数の最大数です。
        static const int ParameterElementCountMax = 8;

        //! 実行する処理に渡されるパラメータのバッファです。
        struct ParameterBuffer
        {
            uintptr_t   _reserved[ ParameterElementCountMax ];
        };

    public:

         /*!
            @brief      登録された処理の数を取得します。
            @return     登録された処理の数です。
         */
        int GetEnqueuedCount() const NN_NOEXCEPT;

         /*!
            @brief      実行された処理の数を取得します。
            @return     実行された処理の数です。
         */
        int GetExecutedCount() const NN_NOEXCEPT;

         /*!
            @brief      最後に実行された処理の結果を取得します。
            @return     最後に実行された処理の結果です。
         */
        nn::Result GetLastResult() const NN_NOEXCEPT;

        /*!
            @brief      指定番目の登録された処理の結果を取得します。
            @param[out]     pOutResult  結果を格納するバッファへのポインタを指定します。
            @param[in]      index       登録した処理の位置です。
            @return     指定番目の登録された処理の有無です。
                        指定番目の処理が実行されなかった場合には false を返します。
         */
        bool GetNthResult(nn::Result* pOutResult, int index) const NN_NOEXCEPT;

        /*!
            @brief      指定されたコマンド識別子の実行結果を取得します。
            @param[out]     pOutResult  結果を格納するバッファへのポインタを指定します。
            @param[in]      command     実行する処理を表すコマンド識別子です。
            @return     指定されたコマンド識別子の実行結果の有無です。
                        指定されたコマンド識別子が実行されなかった場合には false を返します。

            @details    複数回実行されたコマンド識別子を指定した場合には、先の実行結果を取得します。
         */
        bool GetCommandResult(nn::Result* pOutResult, uint32_t command) const NN_NOEXCEPT;

        /*!
            @brief      登録した処理の実行を開始します。
            @param[in]      pEvent      処理の終了通知を受け取るイベントです。
         */
        void Run(nn::os::EventType* pEvent) NN_NOEXCEPT;

        /*!
            @brief      実行中の処理をキャンセルします。
         */
        void Cancel() NN_NOEXCEPT;

        /*!
            @brief      タグに処理を行っているかを判定します。
            @return     タグへの処理中なら真を返します。
         */
        virtual bool IsBusy() NN_NOEXCEPT { return m_IsRunning != false; }

    protected:

        /*!
            @brief      NFC ライブラリの処理をバックグラウンドで実行するスレッドを開始します。
         */
        TagAccessorBase() NN_NOEXCEPT;

        /*!
            @brief      スレッドが稼働中であれば終了します。
         */
        virtual ~TagAccessorBase() NN_NOEXCEPT;

        /*!
            @brief      実行する処理を登録します。
            @param[in]      command     実行する処理を表すコマンド識別子です。
            @param[in]      param       実行する処理に渡すパラメータです。
            @return     登録された処理の位置です。
         */
        template <typename TParam>
        int Enqueue(uint32_t command, const TParam& param) NN_NOEXCEPT
        {
            return Enqueue(command, &param, sizeof(TParam));
        }

        /*!
            @brief      実行する処理を登録します。
            @param[in]      command     実行する処理を表すコマンド識別子です。
            @param[in]      pParam      実行する処理に渡すパラメータです。
            @param[in]      paramSize   実行する処理に渡すパラメータのサイズです。
            @return     登録された処理の位置です。
         */
        int Enqueue(uint32_t command, const void* pParam, size_t paramSize) NN_NOEXCEPT;

        /*!
            @brief      コマンド識別子で表される処理を実行します。
            @param[in]      command     実行する処理を表すコマンド識別子です。
            @param[in]      param       実行する処理に渡すパラメータです。
            @return     実行された処理の結果です。
         */
        virtual nn::Result Execute(uint32_t command, const ParameterBuffer& param) NN_NOEXCEPT = 0;

        /*!
            @brief      NFC ライブラリの処理をバックグラウンドで行うスレッド関数です。
         */
        static void MifareThreadFunc(void* arg) NN_NOEXCEPT;

    private:

        volatile bool       m_IsThreadRunnable;
        volatile bool       m_IsRunning;

        int                 m_CommandCount;
        uint32_t            m_CommandList[ CommandCountMax ];
        ParameterBuffer     m_ParameterList[ CommandCountMax ];

        int                 m_ExecutedCommandCount;
        uint32_t            m_LastCommand;
        nn::Result          m_LastResult;

        nn::os::EventType*  m_pNotifyEvent;
        nn::os::ThreadType  m_Thread;
        nn::os::EventType   m_RunEvent;
    };

}}} // end of namespace nns::nfc::mifare
