﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include "UpnpDeviceGameUI.h"

#include <cstring>

namespace nns {
namespace libupnp {

//***************************************************************
//*
//*                    G L O B A L s
//*
//***************************************************************

const size_t              UpnpDeviceGameUI::ApplicationHeapSize = 64 * 1024 * 1024;
const int                 UpnpDeviceGameUI::FrameBufferWidth = 1280;
const int                 UpnpDeviceGameUI::FrameBufferHeight = 720;
const int                 UpnpDeviceGameUI::FrameRate = 60;
const size_t              UpnpDeviceGameUI::GraphicsMemorySize = 8 * 1024 * 1024;

const float               UpnpDeviceGameUI::AValueX = 320;
const float               UpnpDeviceGameUI::AValueY = UpnpDeviceGameUI::FrameBufferHeight / 2;
const float               UpnpDeviceGameUI::BValueX = 640;
const float               UpnpDeviceGameUI::BValueY = UpnpDeviceGameUI::FrameBufferHeight / 2;
const float               UpnpDeviceGameUI::CValueX = 960;
const float               UpnpDeviceGameUI::CValueY = UpnpDeviceGameUI::FrameBufferHeight / 2;

const float               UpnpDeviceGameUI::ExitValueX = 1180;
const float               UpnpDeviceGameUI::ExitValueY = 665;
const nn::util::Unorm8x4  UpnpDeviceGameUI::ExitValueColor = Color::Yellow;

const float               UpnpDeviceGameUI::TouchBoxInitialX = 320;
const float               UpnpDeviceGameUI::TouchBoxInitialY = UpnpDeviceGameUI::FrameBufferHeight / 4;
const nn::util::Unorm8x4  UpnpDeviceGameUI::TouchBoxColor = Color::Indigo;

const float               UpnpDeviceGameUI::UpnpInfoX = 0;
const float               UpnpDeviceGameUI::UpnpInfoY = 0;
const nn::util::Unorm8x4  UpnpDeviceGameUI::UpnpInfoColor = Color::White;

const float               UpnpDeviceGameUI::UpnpInstructX = 0;
const float               UpnpDeviceGameUI::UpnpInstructY = 100;
const nn::util::Unorm8x4  UpnpDeviceGameUI::UpnpInstructColor = Color::Yellow;

const float               UpnpDeviceGameUI::UIMessagesX = 0;
const float               UpnpDeviceGameUI::UIMessagesY = 600;
const nn::util::Unorm8x4  UpnpDeviceGameUI::UIMessagesColor = Color::White;

//***************************************************************
//*
//* UpnpDeviceGamePlayer
//*
//***************************************************************

UpnpDeviceGameUI::UpnpDeviceGameUI() NN_NOEXCEPT : m_pControllerManager(nullptr),
                                                   m_pApplicationHeap(nullptr),
                                                   m_pGraphicsSystem(nullptr),
                                                   m_pFontSystem(nullptr),
                                                   m_GameUIShouldRun(false),
                                                   m_DisplayTouchBox(false),
                                                   m_LastTouchedValue(GameUIValueUnset)
{
    m_TouchBoxX = TouchBoxInitialX;
    m_TouchBoxY = TouchBoxInitialY;

    m_AGameValue  = UpnpDeviceGameColor_Green;
    m_AValueColor = Color::White;
    m_BGameValue  = UpnpDeviceGameColor_Orange;
    m_BValueColor = Color::White;
    m_BGameValue  = UpnpDeviceGameColor_Red;
    m_CValueColor = Color::White;

    m_Messages.clear();
}

UpnpDeviceGameUI::~UpnpDeviceGameUI() NN_NOEXCEPT
{
    // Clear Messages
    m_Messages.clear();
}

void UpnpDeviceGameUI::SetTouchBoxDisplayed(const bool isDisplayed) NN_NOEXCEPT
{
    m_DisplayTouchBox = isDisplayed;
}

int UpnpDeviceGameUI::ValueToColor(const int inValue, nn::util::Unorm8x4& outColor) NN_NOEXCEPT
{
    int   ret = 0;

    switch(inValue)
    {
    case UpnpDeviceGameColor_Grey:
        outColor = Color::Gray;
        break;
    case UpnpDeviceGameColor_Green:
        outColor = Color::Green;
        break;
    case UpnpDeviceGameColor_Orange:
        outColor = Color::Orange;
        break;
    case UpnpDeviceGameColor_Red:
        outColor = Color::Red;
        break;
    case UpnpDeviceGameColor_Blue:
        outColor = Color::Blue;
        break;
    case UpnpDeviceGameColor_Yellow:
        outColor = Color::Yellow;
        break;
    case UpnpDeviceGameColor_Indigo:
        outColor = Color::Indigo;
        break;
    case UpnpDeviceGameColor_Violet:
        outColor = Color::Violet;
        break;
    default:
        outColor = Color::White;
        break;
    }

    return(ret);
}

int UpnpDeviceGameUI::NextValue(const int inValue) NN_NOEXCEPT
{
    int   result = UpnpDeviceGameColor_Min;

    if (inValue + 1 <= UpnpDeviceGameColor_Max)
    {
        result = inValue + 1;
    }

    return(result);
}

void * UpnpDeviceGameUI::NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void UpnpDeviceGameUI::NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void * UpnpDeviceGameUI::NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}

void UpnpDeviceGameUI::PrintBox( nn::gfx::util::DebugFontTextWriter* pTextWriter,
                                 const char* pGameValue,
                                 const float  offsetX,
                                 const float  offsetY) NN_NOEXCEPT
{
    if (pTextWriter == nullptr || pGameValue == nullptr)
    {
        return;
    }

    const float barOffsetX   = pTextWriter->CalculateStringWidth("|") / 2;
    const float barOffsetY   = pTextWriter->CalculateStringHeight("|") / 2;
    const float minusOffsetX = pTextWriter->CalculateStringWidth("-") / 2;
    const float minusOffsetY = pTextWriter->CalculateStringHeight("-") / 2;
    const float plusOffsetX  = pTextWriter->CalculateStringWidth("+") / 2;
    const float plusOffsetY  = pTextWriter->CalculateStringHeight("+") / 2;

    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print(pGameValue);

    pTextWriter->SetCursor(offsetX - barOffsetX - 45,
                           offsetY - barOffsetY);
    pTextWriter->Print("|");
    pTextWriter->SetCursor(offsetX - barOffsetX + 45,
                           offsetY - barOffsetY);
    pTextWriter->Print("|");
    pTextWriter->SetCursor(offsetX - minusOffsetX,
                           offsetY - minusOffsetY - 45);
    pTextWriter->Print("-");
    pTextWriter->SetCursor(offsetX - minusOffsetX,
                           offsetY - minusOffsetY + 45);
    pTextWriter->Print("-");
    pTextWriter->SetCursor(offsetX - plusOffsetX - 45,
                           offsetY - plusOffsetY - 45);
    pTextWriter->Print("+");
    pTextWriter->SetCursor(offsetX - plusOffsetX - 45,
                           offsetY - plusOffsetY + 45);
    pTextWriter->Print("+");
    pTextWriter->SetCursor(offsetX - plusOffsetX + 45,
                           offsetY - plusOffsetY - 45);
    pTextWriter->Print("+");
    pTextWriter->SetCursor(offsetX - plusOffsetX + 45,
                           offsetY - plusOffsetY + 45);
    pTextWriter->Print("+");
}

bool UpnpDeviceGameUI::BoxIsTouched(const float srcX, const float srcY, const float touchX, const float touchY) NN_NOEXCEPT
{
    bool    isTouched = false;

    if ( (touchX > srcX - 45 && touchX < srcX + 45)  &&
         (touchY > srcY - 45 && touchY < srcY + 45)  )
    {
        isTouched = true;
    }

    return(isTouched);
}

int UpnpDeviceGameUI::DrawBoxes(nn::gfx::util::DebugFontTextWriter* pTextWriter, UpnpDeviceGamePlayer* pGamePlayer) NN_NOEXCEPT
{
    UpnpDeviceGameUIValueTouched valueTouched = GameUIValueUnset;
    int ret = 0;

    if (pTextWriter == nullptr)
    {
        return(0);
    }

    // See if Nintendo Game Values have Changes
    if (pGamePlayer->DidNintendoGameValueChange() == true)
    {
        // Get the Nintendo Game Values
        ret = pGamePlayer->GetGameValues(m_AGameValue, m_BGameValue, m_CGameValue);
        if (ret != 0)
        {
            NN_LOG("Failing calling 'GetGameValues()' - Getting: ValueA, ValueB, ValueC!\n");
            ret = -1;
            return(ret);
        }

        // Set Nintendo Game Value Color
        ValueToColor(m_AGameValue, m_AValueColor);
        ValueToColor(m_BGameValue, m_BValueColor);
        ValueToColor(m_CGameValue, m_CValueColor);
    }

    // If Gameplayer Object Messages have changed
    if (pGamePlayer->DidMessagesChange() == true)
    {
        ret = pGamePlayer->GetMessages(m_Messages);
        if (ret != 0)
        {
            NN_LOG("Failing calling 'GetMessages()' - Getting: new Message Buffers!\n");
            ret = -1;
            return(ret);
        }
    }

    // Begin Drawing
    pTextWriter->SetScale(1, 1);

    if (BoxIsTouched(AValueX, AValueY, m_TouchBoxX, m_TouchBoxY) == true)
    {
        valueTouched = GameUIValueA;
        pTextWriter->SetTextColor(Color::White);
        if (m_LastTouchedValue != GameUIValueA)
        {
            pGamePlayer->SetA(NextValue(m_AGameValue));
        }
    }
    else
    {
        pTextWriter->SetTextColor(m_AValueColor);
    }
    PrintBox(pTextWriter, "A", AValueX, AValueY);

    if (BoxIsTouched(BValueX, BValueY, m_TouchBoxX, m_TouchBoxY) == true)
    {
        valueTouched = GameUIValueB;
        pTextWriter->SetTextColor(Color::White);
        if (m_LastTouchedValue != GameUIValueB)
        {
            pGamePlayer->SetB(NextValue(m_BGameValue));
        }
    }
    else
    {
        pTextWriter->SetTextColor(m_BValueColor);
    }
    PrintBox(pTextWriter, "B", BValueX, BValueY);

    if (BoxIsTouched(CValueX, CValueY, m_TouchBoxX, m_TouchBoxY) == true)
    {
        valueTouched = GameUIValueC;
        pTextWriter->SetTextColor(Color::White);
        if (m_LastTouchedValue != GameUIValueC)
        {
            pGamePlayer->SetC(NextValue(m_CGameValue));
        }
    }
    else
    {
        pTextWriter->SetTextColor(m_CValueColor);
    }
    PrintBox(pTextWriter, "C", CValueX, CValueY);

    // Save Last Value touched
    m_LastTouchedValue = valueTouched;

    // Display UPnP GameState URLs
    pTextWriter->SetCursor(UpnpInfoX, UpnpInfoY);
    pTextWriter->SetTextColor(UpnpInfoColor);
    pTextWriter->Print("UPnP Device        : %s\n", pGamePlayer->GetUdn());
    pTextWriter->Print("UPnP Event URL   : %s\n", pGamePlayer->GetEventUrl());
    pTextWriter->Print("UPnP Control URL : %s\n", pGamePlayer->GetControlUrl());

    // Instructions
    pTextWriter->SetCursor(UpnpInstructX, UpnpInstructY);
    pTextWriter->SetTextColor(UpnpInstructColor);
    pTextWriter->Print("INSTRUCTIONS: Touch box A, B, or C to send a UPnP SOAP Action Request to change the box color.\n");
    pTextWriter->Print("                        To Exit this Sample, Touch 'E X I T' at bottom right\n" );

    // Display UI Messages
    pTextWriter->SetCursor(UIMessagesX, UIMessagesY);
    pTextWriter->SetTextColor(UIMessagesColor);
    for(m_MessagesIter = m_Messages.begin();
        m_MessagesIter != m_Messages.end();
        m_MessagesIter++)
    {
        pTextWriter->Print(m_MessagesIter->c_str());
    }

    // Display Program Exit
    pTextWriter->SetCursor(ExitValueX, ExitValueY);
    pTextWriter->SetTextColor(ExitValueColor);
    pTextWriter->Print("E X I T");

    // Display location of Touch Indicator
    if (m_DisplayTouchBox == true)
    {
        pTextWriter->SetTextColor(TouchBoxColor);
        PrintBox(pTextWriter, "T", m_TouchBoxX, m_TouchBoxY);
    }

    // If Program Exit is touched - Exit!
    if (BoxIsTouched(ExitValueX, ExitValueY, m_TouchBoxX, m_TouchBoxY) == true)
    {
        NN_LOG("Exit Box touched - Exiting!\n");
        m_GameUIShouldRun = false;
    }

    return(0);

}    // NOLINT(impl/function_size)

void UpnpDeviceGameUI::ProcessTouchScreenState(const nns::hid::TouchScreen* const pTouchScreen) NN_NOEXCEPT
{
    if (pTouchScreen == 0)
    {
        m_LastTouchedValue = GameUIValueUnset;
        m_TouchBoxX = TouchBoxInitialX;
        m_TouchBoxY = TouchBoxInitialY;
        return;
    }

    if (pTouchScreen->GetTouchStates().size() == 0)
    {
        m_LastTouchedValue = GameUIValueUnset;
        m_TouchBoxX = TouchBoxInitialX;
        m_TouchBoxY = TouchBoxInitialY;
        return;
    }

    const std::vector<nns::hid::TouchScreen::TouchState> & states = pTouchScreen->GetTouchStates();

    for (size_t i = 0; i < states.size(); ++i)
    {
        const nns::hid::TouchScreen::TouchState & state = states[i];

        m_TouchBoxX = state.position.x;
        m_TouchBoxY = state.position.y;
    }
}

void UpnpDeviceGameUI::ProcessDebugPadState(const nns::hid::DebugPad* const pDebugPad) NN_NOEXCEPT
{
    if (pDebugPad == 0)
    {
        return;
    }

    if (pDebugPad->HasAnyButtons(nns::hid::Button::Up::Mask))
    {
        m_TouchBoxY = m_TouchBoxY - 10;
        if (m_TouchBoxY < 0)
        {
            m_TouchBoxY = 0;
        }
    }

    if (pDebugPad->HasAnyButtons(nns::hid::Button::Down::Mask))
    {
        m_TouchBoxY = m_TouchBoxY + 10;
        if (m_TouchBoxY > FrameBufferWidth)
        {
            m_TouchBoxY = FrameBufferWidth;
        }
    }

    if (pDebugPad->HasAnyButtons(nns::hid::Button::Left::Mask))
    {
        m_TouchBoxX = m_TouchBoxX - 10;
        if (m_TouchBoxX < 0)
        {
            m_TouchBoxX = 0;
        }
    }

    if (pDebugPad->HasAnyButtons(nns::hid::Button::Right::Mask))
    {
        m_TouchBoxX = m_TouchBoxX + 10;
        if (m_TouchBoxX > FrameBufferWidth)
        {
            m_TouchBoxX = FrameBufferWidth;
        }
    }
}

int UpnpDeviceGameUI::MainLoop(UpnpDeviceGamePlayer* pGamePlayer) NN_NOEXCEPT
{
    int ret = 0;

    do
    {
        // If Already started?
        if (m_GameUIShouldRun == true)
        {
            NN_LOG("UpnpDeviceGameUI object has already been started!\n");
            ret = -1;
            break;
        }

        // Specfic to NX
        nv::SetGraphicsAllocator(UpnpDeviceGameUI::NvAllocate, UpnpDeviceGameUI::NvFree, UpnpDeviceGameUI::NvReallocate, nullptr);
        nv::SetGraphicsDevtoolsAllocator(UpnpDeviceGameUI::NvAllocate, UpnpDeviceGameUI::NvFree, UpnpDeviceGameUI::NvReallocate, nullptr);
        nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);

        // Create the HID Controller
        m_pControllerManager = new nns::hid::ControllerManager();
        if (m_pControllerManager == nullptr)
        {
            NN_LOG("Failed calling new on 'nns::hid::ControllerManager'\n");
            ret = -1;
            break;
        }
        nns::hid::util::SetControllerManagerWithDefault(m_pControllerManager);

        // Create the Application Heap
        m_pApplicationHeap = new ApplicationHeap();
        if (m_pApplicationHeap == nullptr)
        {
            NN_LOG("Failed calling new on 'ApplicationHeap'\n");
            ret = -1;
            break;
        }
        m_pApplicationHeap->Initialize(ApplicationHeapSize);

        // Create a new Graphics System
        m_pGraphicsSystem = new GraphicsSystem();
        if (m_pGraphicsSystem == nullptr)
        {
            NN_LOG("Failed calling new on 'GraphicsSystem'\n");
            ret = -1;
            break;
        }
        m_pGraphicsSystem->Initialize(m_pApplicationHeap, FrameBufferWidth, FrameBufferHeight);

        // Enable Window Message
        EnableWindowMessage( m_pGraphicsSystem->GetNativeWindowHandle() );

        // Create a new Font System
        m_pFontSystem = new FontSystem();
        if (m_pFontSystem == nullptr)
        {
            NN_LOG("Failed calling new on 'FontSystem'\n");
            ret = -1;
            break;
        }
        m_pFontSystem->Initialize(m_pApplicationHeap, m_pGraphicsSystem);

        // Get a Debug Text Writer
        nn::gfx::util::DebugFontTextWriter & textWriter = m_pFontSystem->GetDebugFontTextWriter();

        // Main Loop
        m_GameUIShouldRun = true;
        while(m_GameUIShouldRun == true)
        {

            switch (GetWindowMessage(m_pGraphicsSystem->GetNativeWindowHandle()))
            {
            case WindowMessage_Close:
                NN_LOG("WindowMessage_Close received - Exiting!\n" );
                m_GameUIShouldRun = false;
                break;
            default:
                break;
            }

            // Update the Control Manager
            m_pControllerManager->Update();

            // Process any plugged in Debug Control Pad
            ProcessDebugPadState(reinterpret_cast<nns::hid::DebugPad*>(m_pControllerManager->GetController(nns::hid::ControllerId_DebugPad, 0)) );

            // Process any Touch Screen Input
            ProcessTouchScreenState(reinterpret_cast<nns::hid::TouchScreen*>(m_pControllerManager->GetController(nns::hid::ControllerId_TouchScreen, 0)));

            // Place Target Boxes
            DrawBoxes(&textWriter, pGamePlayer);

            //Render the Scene
            m_pGraphicsSystem->BeginDraw();
            m_pFontSystem->Draw();
            m_pGraphicsSystem->EndDraw();

            // Synchronize the Frame Rate
            m_pGraphicsSystem->Synchronize(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));
        }

        // All Done
        break;

    } while (NN_STATIC_CONDITION(false));

    // Free Font System
    if (m_pFontSystem != nullptr)
    {
        m_pFontSystem->Finalize();
        delete m_pFontSystem;
        m_pFontSystem = nullptr;
    }

    // Free Graphic System
    if (m_pGraphicsSystem != nullptr)
    {
        m_pGraphicsSystem->Finalize();
        delete m_pGraphicsSystem;
        m_pGraphicsSystem = nullptr;
    }

    // Free Application Heap
    if (m_pApplicationHeap != nullptr)
    {
        m_pApplicationHeap->Finalize();
        delete m_pApplicationHeap;
        m_pApplicationHeap = nullptr;
    }

    // Free HID Control Manager
    if (m_pControllerManager != nullptr)
    {
        delete m_pControllerManager;
        m_pControllerManager = nullptr;
    }

    return(ret);
}

}} // namespace nns / libupnp
