﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include <cstring>

#include "UpnpLanDevice.h"

namespace nns {
namespace libupnp {

//***************************************************************
//*
//*                    G L O B A L s
//*
//***************************************************************

const char*   UpnpIGWDevices::IGWDeviceType = "urn:schemas-upnp-org:device:InternetGatewayDevice:1";
const char*   UpnpIGWDevices::IGWWANIPConnection = "urn:schemas-upnp-org:service:WANIPConnection:1";

//***************************************************************
//*
//* UpnpLanDevice
//*
//***************************************************************
UpnpLanDevice::UpnpLanDevice() NN_NOEXCEPT
{
    memset(m_Udn, 0, sizeof(m_Udn));
    memset(m_DeviceType, 0, sizeof(m_DeviceType));
    memset(m_FriendlyName, 0, sizeof(m_FriendlyName));
    memset(m_BaseUrl, 0, sizeof(m_BaseUrl));
    memset(m_RelativeUrl, 0, sizeof(m_RelativeUrl));
    memset(m_PresentationUrl, 0, sizeof(m_PresentationUrl));
    memset(m_WanControlUrl, 0, sizeof(m_WanControlUrl));
    m_AdvertiseTimeout = -1;
}

UpnpLanDevice::UpnpLanDevice(const UpnpLanDevice& rhs) NN_NOEXCEPT
{
    memcpy(m_Udn, rhs.m_Udn, sizeof(rhs.m_Udn));
    memcpy(m_DeviceType, rhs.m_DeviceType, sizeof(rhs.m_DeviceType));
    memcpy(m_FriendlyName, rhs.m_FriendlyName, sizeof(rhs.m_FriendlyName));
    memcpy(m_BaseUrl, rhs.m_BaseUrl, sizeof(rhs.m_BaseUrl));
    memcpy(m_RelativeUrl, rhs.m_RelativeUrl, sizeof(rhs.m_RelativeUrl));
    memcpy(m_PresentationUrl, rhs.m_PresentationUrl, sizeof(rhs.m_PresentationUrl));
    memcpy(m_WanControlUrl, rhs.m_WanControlUrl, sizeof(rhs.m_WanControlUrl));
    m_AdvertiseTimeout = rhs.m_AdvertiseTimeout;
}

UpnpLanDevice & UpnpLanDevice::operator=(const UpnpLanDevice& rhs) NN_NOEXCEPT
{
    memcpy(m_Udn, rhs.m_Udn, sizeof(rhs.m_Udn));
    memcpy(m_DeviceType, rhs.m_DeviceType, sizeof(rhs.m_DeviceType));
    memcpy(m_FriendlyName, rhs.m_FriendlyName, sizeof(rhs.m_FriendlyName));
    memcpy(m_BaseUrl, rhs.m_BaseUrl, sizeof(rhs.m_BaseUrl));
    memcpy(m_RelativeUrl, rhs.m_RelativeUrl, sizeof(rhs.m_RelativeUrl));
    memcpy(m_PresentationUrl, rhs.m_PresentationUrl, sizeof(rhs.m_PresentationUrl));
    memcpy(m_WanControlUrl, rhs.m_WanControlUrl, sizeof(rhs.m_WanControlUrl));
    m_AdvertiseTimeout = rhs.m_AdvertiseTimeout;

    return(*this);
}

UpnpLanDevice::~UpnpLanDevice() NN_NOEXCEPT
{
    memset(m_Udn, 0, sizeof(m_Udn));
    memset(m_DeviceType, 0, sizeof(m_DeviceType));
    memset(m_FriendlyName, 0, sizeof(m_FriendlyName));
    memset(m_BaseUrl, 0, sizeof(m_BaseUrl));
    memset(m_RelativeUrl, 0, sizeof(m_RelativeUrl));
    memset(m_PresentationUrl, 0, sizeof(m_PresentationUrl));
    memset(m_WanControlUrl, 0, sizeof(m_WanControlUrl));
    m_AdvertiseTimeout = -1;
}

//***************************************************************
//*
//* UpnpIGWDevices
//*
//***************************************************************

UpnpIGWDevices::UpnpIGWDevices() NN_NOEXCEPT
{
    nn::os::InitializeMutex(&m_MapMutex, false, 0);
}

UpnpIGWDevices::~UpnpIGWDevices() NN_NOEXCEPT
{
    nn::os::LockMutex(&m_MapMutex);
    m_Map.clear();
    nn::os::UnlockMutex(&m_MapMutex);
    nn::os::FinalizeMutex(&m_MapMutex);
}

size_t UpnpIGWDevices::GetMapSize() NN_NOEXCEPT
{
    size_t mapSize = 0;

    nn::os::LockMutex(&m_MapMutex);
    mapSize = m_Map.size();
    nn::os::UnlockMutex(&m_MapMutex);

    return(mapSize);
}

int UpnpIGWDevices::GetDevice(UpnpLanDevice& lanDevice) NN_NOEXCEPT
{
    int  ret = 0;

    nn::os::LockMutex(&m_MapMutex);

    do
    {
        auto myIter = m_Map.begin();
        if (myIter == m_Map.end())
        {
            ret = -1;
            break;
        }

        lanDevice = myIter->second;

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    nn::os::UnlockMutex(&m_MapMutex);

    return(ret);
}

IXML_NodeList* UpnpIGWDevices::GetNthServiceList(IXML_Document* pDesc, unsigned int servicePos) NN_NOEXCEPT
{
    IXML_NodeList*  pServiceList = nullptr;
    IXML_NodeList*  pServlistnodelist = nullptr;
    IXML_Node*      pServlistnode = nullptr;

    pServlistnodelist = ixmlDocument_getElementsByTagName(pDesc, "serviceList");

    if ( (pServlistnodelist != nullptr)                           &&
         (ixmlNodeList_length(pServlistnodelist) != 0)            &&
         (servicePos < ixmlNodeList_length(pServlistnodelist))    )
    {
        pServlistnode = ixmlNodeList_item(pServlistnodelist, servicePos);
        if (pServlistnode != nullptr)
        {
            pServiceList = ixmlElement_getElementsByTagName(reinterpret_cast<IXML_Element*>(pServlistnode), "service");
        }
        else
        {
            pServiceList = nullptr;
        }
    }

    if (pServlistnodelist != nullptr)
    {
        ixmlNodeList_free(pServlistnodelist);
        pServlistnodelist = nullptr;
    }

    return(pServiceList);
}

int UpnpIGWDevices::GetFirstElementItem(char* pOutItem, size_t outSize, IXML_Element* pElement, const char* pItem) NN_NOEXCEPT
{
    IXML_NodeList*  pNodeList = nullptr;
    IXML_Node*      pTextNode = nullptr;
    IXML_Node*      pTmpNode = nullptr;
    size_t          itemSize = 0;
    int             ret = 0;

    do
    {
        pNodeList = ixmlElement_getElementsByTagName(pElement, reinterpret_cast<const DOMString>(pItem));
        if (pNodeList == nullptr)
        {
            NN_LOG("Error finding %s in XML Node\n", pItem);
            ret = -1;
            break;
        }

        pTmpNode = ixmlNodeList_item(pNodeList, 0);
        if (pTmpNode == nullptr)
        {
            NN_LOG("Error finding %s value in XML Node\n", pItem);
            ret = -1;
            break;
        }

        pTextNode = ixmlNode_getFirstChild(pTmpNode);
        if (pTextNode == nullptr)
        {
            NN_LOG("Error failed to get 'textnode' from IXML_Node\n" );
            ret = -1;
            break;
        }

        itemSize = strlen(ixmlNode_getNodeValue(pTextNode));
        if (itemSize > outSize - 1)
        {
            NN_LOG( "ixmlNode_getNodeValue(): Error text length: %ld exceeds (out variable) max size: %ld\n", itemSize, outSize - 1);
            ret = -1;
            break;
        }

        memset(pOutItem, 0, outSize);
        memcpy(pOutItem, ixmlNode_getNodeValue(pTextNode), itemSize);

        // Free the Nodelist
        ixmlNodeList_free(pNodeList);
        pNodeList = nullptr;

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    if (pNodeList != nullptr)
    {
        ixmlNodeList_free(pNodeList);
        pNodeList = nullptr;
    }

    return(ret);
}

int UpnpIGWDevices::GetFirstDocumentItem(char* pOutItem, size_t outSize, IXML_Document* pDesc, const char* pItem) NN_NOEXCEPT
{
    IXML_NodeList*  pNodeList = nullptr;
    IXML_Node*      pTextNode = nullptr;
    IXML_Node*      pTmpNode = nullptr;
    size_t          itemSize = 0;
    int             ret = 0;

    do
    {
        if (pOutItem == nullptr || outSize < 2)
        {
            NN_LOG("Return value should be NULL before use\n");
            ret = -1;
            break;
        }

        pNodeList = ixmlDocument_getElementsByTagName(pDesc, reinterpret_cast<const DOMString>(pItem));
        if (pNodeList == nullptr)
        {
            NN_LOG( "ixmlDocument_getElementsByTagName(): Error finding %s in XML Node\n", pItem);
            ret = -1;
            break;
        }

        pTmpNode = ixmlNodeList_item(pNodeList, 0);
        if (pTmpNode == nullptr)
        {
            NN_LOG( "ixmlNodeList_item(): Error getting Node from Nodelist\n" );
            ret = -1;
            break;
        }

        pTextNode = ixmlNode_getFirstChild(pTmpNode);
        if (pTextNode == nullptr)
        {
            NN_LOG( "ixmlNode_getFirstChild(): Error getting First Child from Node\n" );
            ret = -1;
            break;
        }

        itemSize = strlen(ixmlNode_getNodeValue(pTextNode));
        if (itemSize > outSize - 1)
        {
            NN_LOG( "ixmlNode_getNodeValue(): Error text length: %ld exceeds (out variable) max size: %ld\n", itemSize, outSize - 1);
            ret = -1;
            break;
        }

        memset(pOutItem, 0, outSize);
        memcpy(pOutItem, ixmlNode_getNodeValue(pTextNode), itemSize);

        // Free the Nodelist
        ixmlNodeList_free(pNodeList);
        pNodeList = nullptr;

        // All Done
        break;

    } while (NN_STATIC_CONDITION(false));

    if (pNodeList != nullptr)
    {
        ixmlNodeList_free(pNodeList);
        pNodeList = nullptr;
    }

    return(ret);
}

int UpnpIGWDevices::AddDevice(IXML_Document* pDescDoc, struct Upnp_Discovery* pDiscovery, bool* isIGWDevice) NN_NOEXCEPT
{
    UpnpLanDevice   lanDevice;
    IXML_NodeList*  pServiceList = nullptr;
    IXML_Element*   pService = nullptr;
    int             ret = 0;
    int             length = 0;
    int             idx = 0;
    int             idx2 = 0;
    char            serviceType[200];
    char            wanRelUrl[200];
    bool            haveWanCtrlUrl = false;

    // Get the Map Lock
    nn::os::LockMutex(&m_MapMutex);

    do
    {
        if (pDescDoc == nullptr || pDiscovery == nullptr)
        {
            NN_LOG( "Input arguments are NULL\n" );
            ret = -1;
            break;
        }
        *isIGWDevice = false;

        ret = GetFirstDocumentItem(lanDevice.m_Udn, sizeof(lanDevice.m_Udn), pDescDoc, "UDN");
        if (ret != 0)
        {
            NN_LOG("UpnpGetFirstDocumentItem() did not find UDN!\n" );
            ret = -1;
            break;
        }

        ret = GetFirstDocumentItem(lanDevice.m_DeviceType, sizeof(lanDevice.m_DeviceType), pDescDoc, "deviceType");
        if (ret != 0)
        {
            NN_LOG("UpnpGetFirstDocumentItem() did not find deviceType!\n" );
            ret = -1;
            break;
        }

        // If this device (IS NOT) an Internet Gateway Device (IGW)
        if (strcmp(lanDevice.m_DeviceType, IGWDeviceType) != 0)
        {
            break;   // Not an IGW device
        }
        *isIGWDevice = true;

        ret = GetFirstDocumentItem(lanDevice.m_FriendlyName, sizeof(lanDevice.m_FriendlyName), pDescDoc, "friendlyName");
        if (ret != 0)
        {
            NN_LOG("UpnpGetFirstDocumentItem() did not find friendlyName!\n" );
            ret = -1;
            break;
        }

        ret = GetFirstDocumentItem(lanDevice.m_RelativeUrl, sizeof(lanDevice.m_RelativeUrl), pDescDoc, "presentationURL");
        if (ret != 0)
        {
            NN_LOG("UpnpGetFirstDocumentItem() did not find presentationURL!\n" );
            ret = -1;
            break;
        }

        ret = GetFirstDocumentItem(lanDevice.m_BaseUrl, sizeof(lanDevice.m_BaseUrl), pDescDoc, "URLBase");
        if (ret == 0)
        {
            ret = UpnpResolveURL(lanDevice.m_BaseUrl, lanDevice.m_RelativeUrl, lanDevice.m_PresentationUrl);
            if (ret != UPNP_E_SUCCESS)
            {
                NN_LOG("Error generating PresentationURL from (BaseURL) %s + %s\n", lanDevice.m_BaseUrl, lanDevice.m_RelativeUrl);
                ret = -1;
                break;
            }
        }
        else
        {
            ret = UpnpResolveURL(pDiscovery->Location, lanDevice.m_RelativeUrl, lanDevice.m_PresentationUrl);
            if (ret != UPNP_E_SUCCESS)
            {
                NN_LOG("Error generating PresentationURL from (Discovery Location) %s + %s\n", pDiscovery->Location, lanDevice.m_RelativeUrl);
                ret = -1;
                break;
            }
        }

        // Save when Advertised Service is going to timeout
        lanDevice.m_AdvertiseTimeout = pDiscovery->Expires;

        // Find:  WAN Control URL
        haveWanCtrlUrl = false;
        for(idx = 0; (pServiceList = GetNthServiceList(pDescDoc, idx)) != nullptr; idx++)
        {
            length = ixmlNodeList_length(pServiceList);

            for (idx2 = 0; idx2 < length; idx2++)
            {
                pService = reinterpret_cast<IXML_Element*>(ixmlNodeList_item(pServiceList, idx2));
                if (pService == nullptr)
                {
                    NN_LOG("Failed to get 'service' at level: %d\n", idx2);
                    ret = -1;
                    break;
                }

                ret = GetFirstElementItem(serviceType, sizeof(serviceType), pService, "serviceType");
                if (ret != 0)
                {
                    NN_LOG("Failed to get 'serviceType'\n");
                    ret = -1;
                    break;
                }

                // Is this the right service type?
                if (strcmp(serviceType, IGWWANIPConnection) != 0)
                {
                    continue;   // No
                }

                ret = GetFirstElementItem(wanRelUrl, sizeof(wanRelUrl), pService, "controlURL");
                if (ret != 0)
                {
                    NN_LOG("Failed to get 'controlURL' from service: %s\n", serviceType);
                    ret = -1;
                    break;
                }

                ret = UpnpResolveURL(pDiscovery->Location, wanRelUrl, lanDevice.m_WanControlUrl);
                if (ret != UPNP_E_SUCCESS)
                {
                    NN_LOG("Failed calling 'UpnpResolveURL' to resolve the WAN Control URL - Error: %d/%s\n",
                                                                 ret, UpnpGetErrorMessage(ret));
                    ret = -1;
                    break;
                }

                // We have the WAN Control URL
                haveWanCtrlUrl = true;

                // All done!
                break;
            }
        }

        // If error while searching for WAN Control URL..
        if (ret < 0)
        {
            break;
        }

        // If we did not find the WAN Control URL
        if (haveWanCtrlUrl == false)
        {
            NN_LOG( "Failed to find WAN Control URL: %s, element <%s>\n", IGWWANIPConnection, "controlURL" );
            break;
        }

        // Display the Record
        {
            NN_LOG( "UDN              : %s\n", lanDevice.m_Udn);
            NN_LOG( "DeviceType       : %s\n", lanDevice.m_DeviceType);
            NN_LOG( "FriendlyName     : %s\n", lanDevice.m_FriendlyName);
            NN_LOG( "BaseUrl          : %s\n", lanDevice.m_BaseUrl);
            NN_LOG( "RelativeUrl      : %s\n", lanDevice.m_RelativeUrl);
            NN_LOG( "PresentationUrl  : %s\n", lanDevice.m_PresentationUrl);
            NN_LOG( "WANCtrlUrl       : %s\n", lanDevice.m_WanControlUrl);
            NN_LOG( "Advert Tmout     : %d\n", lanDevice.m_AdvertiseTimeout);
        }

        // Otherwise add the Device
        m_Map.insert(std::pair<const char *,UpnpLanDevice>(lanDevice.m_Udn, lanDevice));

        NN_LOG("Device: [%s] - Created\n", lanDevice.m_Udn);

        // All done
        break;

    } while(1);

    // Unlock the Map
    nn::os::UnlockMutex(&m_MapMutex);

    return(ret);

}  // NOLINT(impl/function_size)

}}  // namespace nns / libupnp
