﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Assert.h>
#include <nn/ldn.h>
#include "LdnTask.h"

namespace nns { namespace ldn
{
    LdnInitializeTask::LdnInitializeTask() NN_NOEXCEPT
    {
    }

    LdnInitializeTask::~LdnInitializeTask() NN_NOEXCEPT
    {
    }

    TaskResult LdnInitializeTask::RunImpl(nn::Result* pOutResult) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutResult);
        *pOutResult = nn::ldn::Initialize();
        return TaskResult_Finished;
    }

    LdnFinalizeTask::LdnFinalizeTask() NN_NOEXCEPT
    {
    }

    LdnFinalizeTask::~LdnFinalizeTask() NN_NOEXCEPT
    {
    }

    TaskResult LdnFinalizeTask::RunImpl(nn::Result* pOutResult) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutResult);
        if (nn::ldn::GetState() != nn::ldn::State_None)
        {
            nn::ldn::Finalize();
        }
        *pOutResult = nn::ResultSuccess();
        return TaskResult_Finished;
    }

    LdnOpenAccessPointTask::LdnOpenAccessPointTask() NN_NOEXCEPT
    {
    }

    LdnOpenAccessPointTask::~LdnOpenAccessPointTask() NN_NOEXCEPT
    {
    }

    TaskResult LdnOpenAccessPointTask::RunImpl(nn::Result* pOutResult) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutResult);
        nn::ldn::State state = nn::ldn::GetState();
        if (state == nn::ldn::State_AccessPoint)
        {
            *pOutResult = nn::ResultSuccess();
        }
        else
        {
            *pOutResult = nn::ldn::OpenAccessPoint();
        }
        return TaskResult_Finished;
    }

    LdnOpenStationTask::LdnOpenStationTask() NN_NOEXCEPT
    {
    }

    LdnOpenStationTask::~LdnOpenStationTask() NN_NOEXCEPT
    {
    }

    TaskResult LdnOpenStationTask::RunImpl(nn::Result* pOutResult) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutResult);
        if (nn::ldn::GetState() == nn::ldn::State_Station)
        {
            *pOutResult = nn::ResultSuccess();
        }
        else
        {
            *pOutResult = nn::ldn::OpenStation();
        }
        return TaskResult_Finished;
    }

    LdnCloseAccessPointTask::LdnCloseAccessPointTask() NN_NOEXCEPT
    {
    }

    LdnCloseAccessPointTask::~LdnCloseAccessPointTask() NN_NOEXCEPT
    {
    }

    TaskResult LdnCloseAccessPointTask::RunImpl(nn::Result* pOutResult) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutResult);
        nn::ldn::State state = nn::ldn::GetState();
        if (state == nn::ldn::State_Initialized)
        {
            *pOutResult = nn::ResultSuccess();
        }
        else
        {
            *pOutResult = nn::ldn::CloseAccessPoint();
        }
        return TaskResult_Finished;
    }

    LdnCloseStationTask::LdnCloseStationTask() NN_NOEXCEPT
    {
    }

    LdnCloseStationTask::~LdnCloseStationTask() NN_NOEXCEPT
    {
    }

    TaskResult LdnCloseStationTask::RunImpl(nn::Result* pOutResult) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutResult);
        nn::ldn::State state = nn::ldn::GetState();
        if (state == nn::ldn::State_Initialized)
        {
            *pOutResult = nn::ResultSuccess();
        }
        else
        {
            *pOutResult = nn::ldn::CloseStation();
        }
        return TaskResult_Finished;
    }

    LdnScanTask::LdnScanTask() NN_NOEXCEPT
        : m_ScanResultCount(0),
          m_Channel(nn::ldn::AutoChannel)
    {
        std::memset(&m_ScanFilter, 0, sizeof(nn::ldn::ScanFilter));
    }

    LdnScanTask::~LdnScanTask() NN_NOEXCEPT
    {
    }

    void LdnScanTask::SetChannel(int channel) NN_NOEXCEPT
    {
        m_Channel = channel;
    }

    void LdnScanTask::SetScanFilter(const nn::ldn::ScanFilter& filter) NN_NOEXCEPT
    {
        m_ScanFilter = filter;
    }

    int LdnScanTask::GetNetworkCount() const NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(GetState(), TaskState_Finished);
        return m_ScanResultCount;
    }

    const nn::ldn::NetworkInfo& LdnScanTask::GetNetwork(int index) const NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(GetState(), TaskState_Finished);
        NN_ASSERT_RANGE(index, 0, m_ScanResultCount);
        return m_Networks[index];
    }

    TaskResult LdnScanTask::RunImpl(nn::Result* pOutResult) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutResult);
        *pOutResult = nn::ldn::Scan(
            m_Networks, &m_ScanResultCount, nn::ldn::ScanResultCountMax, m_ScanFilter, m_Channel);
        return TaskResult_Finished;
    }

    LdnConnectTask::LdnConnectTask() NN_NOEXCEPT
        : m_LocalCommunicationVersion(0),
          m_RetryCount(1),
          m_IsCanceled(false)
    {
        std::memset(&m_NetworkInfo, 0, sizeof(nn::ldn::NetworkInfo));
        std::memset(&m_SecurityConfig, 0, sizeof(nn::ldn::SecurityConfig));
        std::memset(&m_UserConfig, 0, sizeof(nn::ldn::UserConfig));
    }

    LdnConnectTask::~LdnConnectTask() NN_NOEXCEPT
    {
    }

    void LdnConnectTask::SetLocalCommunicationVersion(int version) NN_NOEXCEPT
    {
        m_LocalCommunicationVersion = version;
    }

    void LdnConnectTask::SetNetworkInfo(const nn::ldn::NetworkInfo& network) NN_NOEXCEPT
    {
        m_NetworkInfo = network;
    }

    void LdnConnectTask::SetSecurityConfig(const nn::ldn::SecurityConfig& security) NN_NOEXCEPT
    {
        m_SecurityConfig = security;
    }

    void LdnConnectTask::SetUserConfig(const nn::ldn::UserConfig& user) NN_NOEXCEPT
    {
        m_UserConfig = user;
    }

    void LdnConnectTask::SetRetryCount(int retryCount) NN_NOEXCEPT
    {
        NN_ASSERT(0 < retryCount);
        m_RetryCount = retryCount;
    }

    nn::ldn::Ipv4Address LdnConnectTask::GetIpv4Address() const NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(GetState(), TaskState_Finished);
        return m_Ipv4Address;
    }

    nn::ldn::SubnetMask LdnConnectTask::GetSubnetMask() const NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(GetState(), TaskState_Finished);
        return m_SubnetMask;
    }

    TaskResult LdnConnectTask::RunImpl(nn::Result* pOutResult) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutResult);

        // ネットワークへの接続を試行し、失敗した場合には指定された回数だけリトライします。
        nn::Result result = nn::ResultSuccess();
        for (int i = 0; i < m_RetryCount; ++i)
        {
            // Cancel() を実行された場合には中断します。
            if (m_IsCanceled)
            {
                return TaskResult_Canceled;
            }

            // ネットワークへの接続を試行します。
            result = nn::ldn::Connect(
                m_NetworkInfo, m_SecurityConfig, m_UserConfig, m_LocalCommunicationVersion,
                nn::ldn::ConnectOption_None);
            if (result.IsSuccess())
            {
                // 接続に成功しました。
                break;
            }
            else if (nn::ldn::ResultInvalidState::Includes(result) ||
                     nn::ldn::ResultDeviceNotAvailable::Includes(result) ||
                     nn::ldn::ResultNodeCountLimitation::Includes(result) ||
                     nn::ldn::ResultIncompatibleVersion::Includes(result))
            {
                // リトライしても成功する見込みがありません。
                break;
            }
            else if (nn::ldn::ResultConnectionFailed::Includes(result))
            {
                // 接続をリトライします。
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
            }
            else
            {
                // 想定外の Result です。
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }

        // 接続に成功した場合は IPv4 アドレスを取得します。
        if (result.IsSuccess())
        {
            result = nn::ldn::GetIpv4Address(&m_Ipv4Address, &m_SubnetMask);
        }
        *pOutResult = result;
        return TaskResult_Finished;
    }

    void LdnConnectTask::CancelImpl() NN_NOEXCEPT
    {
        m_IsCanceled = true;
    }

    LdnCreateNetworkTask::LdnCreateNetworkTask() NN_NOEXCEPT
    {
        std::memset(&m_NetworkConfig, 0, sizeof(nn::ldn::NetworkConfig));
        std::memset(&m_SecurityConfig, 0, sizeof(nn::ldn::SecurityConfig));
        std::memset(&m_UserConfig, 0, sizeof(nn::ldn::UserConfig));
    }

    LdnCreateNetworkTask::~LdnCreateNetworkTask() NN_NOEXCEPT
    {
    }

    void LdnCreateNetworkTask::SetNetworkConfig(const nn::ldn::NetworkConfig& network) NN_NOEXCEPT
    {
        m_NetworkConfig = network;
    }

    void LdnCreateNetworkTask::SetSecurityConfig(
        const nn::ldn::SecurityConfig& security) NN_NOEXCEPT
    {
        m_SecurityConfig = security;
    }

    void LdnCreateNetworkTask::SetUserConfig(const nn::ldn::UserConfig& user) NN_NOEXCEPT
    {
        m_UserConfig = user;
    }

    nn::ldn::Ipv4Address LdnCreateNetworkTask::GetIpv4Address() const NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(GetState(), TaskState_Finished);
        return m_Ipv4Address;
    }

    nn::ldn::SubnetMask LdnCreateNetworkTask::GetSubnetMask() const NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(GetState(), TaskState_Finished);
        return m_SubnetMask;
    }

    TaskResult LdnCreateNetworkTask::RunImpl(nn::Result* pOutResult) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutResult);
        nn::Result result;
        result = nn::ldn::CreateNetwork(m_NetworkConfig, m_SecurityConfig, m_UserConfig);
        if (result.IsSuccess())
        {
            result = nn::ldn::GetIpv4Address(&m_Ipv4Address, &m_SubnetMask);
        }
        *pOutResult = result;
        return TaskResult_Finished;
    }

    LdnRejectTask::LdnRejectTask() NN_NOEXCEPT
        : m_Ipv4Address(nn::ldn::ZeroIpv4Address)
    {
    }

    LdnRejectTask::LdnRejectTask(nn::ldn::Ipv4Address ipv4Address) NN_NOEXCEPT
        : m_Ipv4Address(ipv4Address)
    {
    }

    LdnRejectTask::LdnRejectTask(const nn::ldn::NodeInfo& node) NN_NOEXCEPT
        : m_Ipv4Address(node.ipv4Address)
    {
    }

    LdnRejectTask::~LdnRejectTask() NN_NOEXCEPT
    {
    }

    void LdnRejectTask::SetStation(nn::ldn::Ipv4Address ipv4Address) NN_NOEXCEPT
    {
        m_Ipv4Address = ipv4Address;
    }

    void LdnRejectTask::SetStation(const nn::ldn::NodeInfo& node) NN_NOEXCEPT
    {
        SetStation(node.ipv4Address);
    }

    TaskResult LdnRejectTask::RunImpl(nn::Result* pOutResult) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutResult);
        *pOutResult = nn::ldn::Reject(m_Ipv4Address);
        return TaskResult_Finished;
    }

}} // namespace nns::ldn
