﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/err.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/socket/socket_Api.h>
#include <nn/util/util_FormatString.h>
#include "LdnDrawer.h"

namespace nns { namespace ldn { namespace
{
    #define NNS_LDN_DRAW_TEXT(...) NN_LOG(__VA_ARGS__)

    #define NNS_LDN_DRAW_LINEFEED NNS_LDN_DRAW_TEXT("\n")

    #define NNS_LDN_DRAW_LINE(...) do\
        {\
            NNS_LDN_DRAW_TEXT(__VA_ARGS__);\
            NNS_LDN_DRAW_LINEFEED;\
        } while (NN_STATIC_CONDITION(0))

    #define NNS_LDN_DRAW_MENU_ITEM(cond, ...) do\
        {\
            NNS_LDN_DRAW_TEXT("[%c] ", (cond) ? '>' : ' ');\
            NNS_LDN_DRAW_LINE(__VA_ARGS__);\
        } while (NN_STATIC_CONDITION(0))

    #define NNS_LDN_DRAW_NEWPAGE(app) do\
        {\
            NNS_LDN_DRAW_TEXT("\n\n");\
            NNS_LDN_DRAW_LINE("FRAME   : %lld", (app).totalFrame);\
        } while (NN_STATIC_CONDITION(0))

    void ToString(char* out, size_t bufferSize, const nn::ldn::Ipv4Address ip) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(out);
        NN_ASSERT(16 <= bufferSize);
        uint32_t address = nn::socket::InetHtonl(ip.raw);
        const uint8_t* p = reinterpret_cast<uint8_t*>(&address);
        nn::util::SNPrintf(out, bufferSize, "%d.%d.%d.%d", p[0], p[1], p[2], p[3]);
    }

    void ToString(char* out, size_t bufferSize, int linkLevel) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(out);
        NN_ASSERT(4 <= bufferSize);
        int i;
        for (i = 0; i < linkLevel; ++i)
        {
            out[i] = '*';
        }
        out[i] = 0;
    }

}}} // namespace nns::ldn::<unnamed>

namespace nns { namespace ldn
{
    LdnDrawer::LdnDrawer() NN_NOEXCEPT
    {
    }

    LdnDrawer::~LdnDrawer() NN_NOEXCEPT
    {
    }

    void LdnDrawer::Setup(const ApplicationStatus& app) NN_NOEXCEPT
    {
        NN_UNUSED(app);
    }

    void LdnDrawer::Cleanup(const ApplicationStatus& app) NN_NOEXCEPT
    {
        NN_UNUSED(app);
    }

    void LdnDrawer::Draw(const ApplicationStatus& app) NN_NOEXCEPT
    {
        const auto& data = *static_cast<const LdnData*>(app.pSceneData);
        switch (data.subScene)
        {
        case LdnSubScene_Initializing:
            DrawInitializing(app, data);
            break;
        case LdnSubScene_Initialized:
            DrawInitialized(app, data);
            break;
        case LdnSubScene_Station:
            DrawStation(app, data);
            break;
        case LdnSubScene_StationScanning:
            DrawStationScanning(app, data);
            break;
        case LdnSubScene_StationClosing:
            DrawStationClosing(app, data);
            break;
        case LdnSubScene_StationConnecting:
            DrawStationConnecting(app, data);
            break;
        case LdnSubScene_StationConnected:
            DrawStationConnected(app, data);
            break;
        case LdnSubScene_StationDisconnected:
            DrawStationDisconnected(app, data);
            break;
        case LdnSubScene_AccessPointCreating:
            DrawAccessPointCreating(app, data);
            break;
        case LdnSubScene_AccessPointCreated:
            DrawAccessPointCreated(app, data);
            break;
        case LdnSubScene_AccessPointScanning:
            DrawAccessPointScanning(app, data);
            break;
        case LdnSubScene_AccessPointRejecting:
            DrawAccessPointRejecting(app, data);
            break;
        case LdnSubScene_AccessPointClosing:
            DrawAccessPointClosing(app, data);
            break;
        case LdnSubScene_Finalizing:
            DrawFinalizing(app, data);
            break;
        case LdnSubScene_Error:
            DrawError(app, data);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void LdnDrawer::DrawInitializing(
        const ApplicationStatus& app, const LdnData& data) NN_NOEXCEPT
    {
        if (data.isUpdated)
        {
            NNS_LDN_DRAW_NEWPAGE(app);
            NNS_LDN_DRAW_LINE("ローカル通信の準備中です・・・。");
        }
    }

    void LdnDrawer::DrawInitialized(
        const ApplicationStatus& app, const LdnData& data) NN_NOEXCEPT
    {
        if (data.isUpdated)
        {
            NNS_LDN_DRAW_NEWPAGE(app);
            NNS_LDN_DRAW_MENU_ITEM(data.y == 0, "ネットワークを探す");
            NNS_LDN_DRAW_MENU_ITEM(data.y == 1, "ネットワークを構築する");
            NNS_LDN_DRAW_LINE("(A) 決定 (+) 終了");
        }
    }

    void LdnDrawer::DrawStation(
        const ApplicationStatus& app, const LdnData& data) NN_NOEXCEPT
    {
        if (data.isUpdated)
        {
            NNS_LDN_DRAW_NEWPAGE(app);
            if (0 < data.scanResultCount)
            {
                if (nn::ldn::ResultConnectionFailed::Includes(data.lastError))
                {
                    NNS_LDN_DRAW_LINE("ネットワークへの接続に失敗しました。");
                    NNS_LDN_DRAW_LINE("%d", data.lastError.GetDescription());
                }
                NNS_LDN_DRAW_LINE(
                    "    %-10s %-7s %-5s %-5s %-32s",
                    "NAME", "CHANNEL", "NODES", "LEVEL", "SSID");
                char linkLevelString[4];
                for (int i = 0; i < data.scanResultCount; ++i)
                {
                    const auto& network = data.scanResult[i];
                    ToString(linkLevelString, sizeof(linkLevelString), network.common.linkLevel);
                    NNS_LDN_DRAW_MENU_ITEM(
                        i == data.y, "%-10s %-7d %d/%d   %-5s %s",
                        network.ldn.nodes[0].userName, network.common.channel,
                        network.ldn.nodeCount, network.ldn.nodeCountMax,
                        linkLevelString, network.common.ssid.raw);
                }
                NNS_LDN_DRAW_LINE("(A) 選択したネットワークに接続 (B) 戻る (X) ネットワークを探す");
            }
            else
            {
                NNS_LDN_DRAW_LINE("周囲にネットワークが見つかりませんでした。");
                NNS_LDN_DRAW_LINE("(B) 戻る (X) ネットワークを探す");
            }

        }
    }

    void LdnDrawer::DrawStationScanning(
        const ApplicationStatus& app, const LdnData& data) NN_NOEXCEPT
    {
        if (data.isUpdated)
        {
            NNS_LDN_DRAW_NEWPAGE(app);
            NNS_LDN_DRAW_LINE("周囲のネットワークを探索中です・・・。");
        }
    }

    void LdnDrawer::DrawStationConnecting(
        const ApplicationStatus& app, const LdnData& data) NN_NOEXCEPT
    {
        if (data.isUpdated)
        {
            NNS_LDN_DRAW_NEWPAGE(app);
            NNS_LDN_DRAW_LINE("ネットワークに接続中です・・・。");
        }
    }

    void LdnDrawer::DrawStationConnected(
        const ApplicationStatus& app, const LdnData& data) NN_NOEXCEPT
    {
        if (data.isUpdated || data.subSceneFrame % 60 == 0)
        {
            NNS_LDN_DRAW_NEWPAGE(app);
            NNS_LDN_DRAW_LINE("%-2s %-4s %-10s %-16s %-10s %-7s %-11s",
                "ID", "TYPE", "NAME", "ADDRESS", "COUNT", "LOSS[%]", "DESCRIPTION");
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                const auto& node = data.network.ldn.nodes[i];
                char ipString[16];
                ToString(ipString, sizeof(ipString), node.ipv4Address);
                if (node.isConnected)
                {
                    const auto& counter = data.counters[i];
                    if (node.ipv4Address == data.address)
                    {
                        NNS_LDN_DRAW_LINE(
                            "%-2d %-4s %-10s %-16s %-10s %-7s %-11s",
                            node.nodeId, i == 0 ? "AP" : "STA", node.userName, ipString,
                            "-", "-", "YOU");
                    }
                    else
                    {
                        NNS_LDN_DRAW_LINE(
                            "%-2d %-4s %-10s %-16s %-10d %-7.3f %-11s",
                            node.nodeId, i == 0 ? "AP" : "STA", node.userName, ipString,
                            counter.count, counter.loss * 100.0f, "-");
                    }
                }
                else
                {
                    NNS_LDN_DRAW_LINE(
                        "%-2d %-4s %-10s %-16s %-10s %-7s %-11s",
                        i, "-", "-", "-", "-", "-", "-");
                }
            }
            NNS_LDN_DRAW_LINE("(B) ネットワークから抜ける (X) ネットワークを探す");
        }
    }

    void LdnDrawer::DrawStationDisconnected(
        const ApplicationStatus& app, const LdnData& data) NN_NOEXCEPT
    {
        if (data.isUpdated)
        {
            NNS_LDN_DRAW_NEWPAGE(app);
            switch (static_cast<nn::ldn::DisconnectReason>(data.disconnectReason))
            {
            case nn::ldn::DisconnectReason_None:
                NN_ASSERT(false);
                break;
            case nn::ldn::DisconnectReason_DisconnectedByUser:
                NNS_LDN_DRAW_LINE("ネットワークから離脱しました。");
                break;
            case nn::ldn::DisconnectReason_DisconnectedBySystem:
                NNS_LDN_DRAW_LINE("システムの要求によってネットワークから切断されました。");
                break;
            case nn::ldn::DisconnectReason_DestroyedByUser:
                NNS_LDN_DRAW_LINE("ネットワークの管理者がネットワークを破棄しました。");
                break;
            case nn::ldn::DisconnectReason_DestroyedBySystem:
                NNS_LDN_DRAW_LINE("システムの要求によってネットワークが破棄されました。");
                break;
            case nn::ldn::DisconnectReason_Rejected:
                NNS_LDN_DRAW_LINE("ネットワークの管理者によって切断されました。");
                break;
            case nn::ldn::DisconnectReason_SignalLost:
                NNS_LDN_DRAW_LINE("通信状況が悪いためネットワークから切断されました。");
                break;
            case nn::ldn::DisconnectReason_Unknown:
                NNS_LDN_DRAW_LINE("ネットワークから切断されました。");
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            NNS_LDN_DRAW_LINE("(B) 戻る");
        }
    }

    void LdnDrawer::DrawStationClosing(
        const ApplicationStatus& app, const LdnData& data) NN_NOEXCEPT
    {
        NN_UNUSED(app);
        NN_UNUSED(data);
    }

    void LdnDrawer::DrawAccessPointCreating(
        const ApplicationStatus& app, const LdnData& data) NN_NOEXCEPT
    {
        if (data.isUpdated)
        {
            NNS_LDN_DRAW_NEWPAGE(app);
            NNS_LDN_DRAW_LINE("ネットワークを構築中です・・・。");
        }
    }

    void LdnDrawer::DrawAccessPointCreated(
        const ApplicationStatus& app, const LdnData& data) NN_NOEXCEPT
    {
        if (data.isUpdated || data.subSceneFrame % 60 == 0)
        {
            NNS_LDN_DRAW_NEWPAGE(app);
            NNS_LDN_DRAW_LINE("SSID     : %s", data.network.common.ssid.raw);
            NNS_LDN_DRAW_LINE("CHANNEL  : %d ch", data.network.common.channel);
            NNS_LDN_DRAW_LINE("SCAN RET : %d networks", data.scanResultCount);
            NNS_LDN_DRAW_LINE("    %-2s %-4s %-10s %-16s %-10s %-7s %-11s",
                "ID", "TYPE", "NAME", "ADDRESS", "COUNT", "LOSS[%]", "DESCRIPTION");
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                const auto& node = data.network.ldn.nodes[i];
                char ipString[16];
                ToString(ipString, sizeof(ipString), node.ipv4Address);
                if (node.isConnected)
                {
                    const auto& counter = data.counters[i];
                    if (node.ipv4Address == data.address)
                    {
                        NNS_LDN_DRAW_MENU_ITEM(
                            i == data.y, "%-2d %-4s %-10s %-16s %-10s %-7s %-11s",
                            node.nodeId, i == 0 ? "AP" : "STA", node.userName, ipString,
                            "-", "-", "YOU");
                    }
                    else
                    {
                        NNS_LDN_DRAW_MENU_ITEM(
                            i == data.y, "%-2d %-4s %-10s %-16s %-10d %-7.3f %-11s",
                            node.nodeId, i == 0 ? "AP" : "STA", node.userName, ipString,
                            counter.count, counter.loss * 100.0f, "-");
                    }
                }
                else
                {
                    NNS_LDN_DRAW_MENU_ITEM(
                        i == data.y, "%-2d %-4s %-10s %-16s %-10s %-7s %-11s",
                        i, "-", "-", "-", "-", "-", "-");
                }
            }
            NNS_LDN_DRAW_LINE("(A) 選択した参加者を切断 (B) ネットワークの破棄 (X) ネットワークを探す");
        }
    }

    void LdnDrawer::DrawAccessPointScanning(
        const ApplicationStatus& app, const LdnData& data) NN_NOEXCEPT
    {
        if (data.isUpdated)
        {
            NNS_LDN_DRAW_NEWPAGE(app);
            NNS_LDN_DRAW_LINE("周囲のネットワークを探索中です・・・。");
        }
    }

    void LdnDrawer::DrawAccessPointRejecting(
        const ApplicationStatus& app, const LdnData& data) NN_NOEXCEPT
    {
        if (data.isUpdated)
        {
            const auto& node = data.network.ldn.nodes[data.rejectNodeId];
            NNS_LDN_DRAW_NEWPAGE(app);
            NNS_LDN_DRAW_LINE("%s (ID: %d) を切断しています・・・。", node.userName, node.nodeId);
        }
    }

    void LdnDrawer::DrawAccessPointClosing(
        const ApplicationStatus& app, const LdnData& data) NN_NOEXCEPT
    {
        if (data.isUpdated)
        {
            NNS_LDN_DRAW_NEWPAGE(app);
            NNS_LDN_DRAW_LINE("ネットワークを破棄しています・・・。");
        }
    }

    void LdnDrawer::DrawFinalizing(
        const ApplicationStatus& app, const LdnData& data) NN_NOEXCEPT
    {
        NN_UNUSED(app);
        NN_UNUSED(data);
    }

    void LdnDrawer::DrawError(
        const ApplicationStatus& app, const LdnData& data) NN_NOEXCEPT
    {
        if (data.isUpdated)
        {
            NNS_LDN_DRAW_NEWPAGE(app);
            nn::Result result = data.lastError;
            if (nn::ldn::ResultDeviceNotAvailable::Includes(result) ||
                nn::ldn::ResultNodeCountLimitation::Includes(result) ||
                nn::ldn::ResultIncompatibleVersion::Includes(result))
            {
                nn::err::ShowError(result);
            }
            NNS_LDN_DRAW_LINE("(A) やりなおす (B) 終了");
        }
    }

}} // namespace nns::ldn
