﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "MarkerPositionReaderModeState.h"

#include <nn/nn_Assert.h>
#include <algorithm>
#include <sstream>
#include <iomanip>

// Check if joycon is learning too much where camera can not see markers properly
bool CheckJoyconLeaning(nn::irsensor::MarkerPositionReaderState& state)
{
    // Rotation/translation matrix:
    // | R        | T  |
    // | Xx Yx Zx | Tx |
    // | Xy Yy Zy | Ty |
    // | Xz Yz Zz | Tz |

    float const warningThreshold = 0.5f;
    float const YzMax = 0.3f;

    nn::util::Float4x4 matrix;
    MatrixStore(&matrix, state.cameraMatrix);

    float const Yz = matrix.m[2][1]; // rotation around X axis indicator
    float const Zx = matrix.m[0][2];
    float const Zy = matrix.m[1][2];
    float const leaningFactor = Zx*Zx + Zy*Zy;

    if (leaningFactor > warningThreshold * warningThreshold || Yz > YzMax)
    {
        // "Leaning too much";
        return true;
    }
    else
    {
        // OK
        return false;
    }
}

PositionReaderModeState::PositionReaderModeState(IrSensorMode* pNextProcessor, int* pMenuSelection, nn::irsensor::IrCameraHandle irCameraHandle)
    : IrSensorModeState(pNextProcessor, pMenuSelection, irCameraHandle)
{
    float* pDrawScale = &m_DrawScale;
    *pDrawScale = 1.0f;
    m_ReadWriteMenu.emplace_back("Pen Speed",
        [pDrawScale](std::stringstream& sstr) {
        sstr << *pDrawScale;
    },
        [pDrawScale](int8_t delta) {
        if (delta > 0)
            *pDrawScale += 0.4f;
        else if (delta < 0)
            *pDrawScale -= 0.4f;
        *pDrawScale = (*pDrawScale >= 4.0f) ? 4.0f : *pDrawScale;
        *pDrawScale = (*pDrawScale < 1.0f) ? 1.0f : *pDrawScale;
    }
    );

    m_ReadOnlyMenu.emplace_back("Position",
        [&](std::stringstream& sstr) {
        auto const n = m_lastState.pentipPosition;
        sstr << "position (" << n.x << ", " << n.y << ");";
    },
        [](int8_t delta) { NN_UNUSED(delta); }
    );

    m_ReadOnlyMenu.emplace_back("Touch",
        [&](std::stringstream& sstr) {
        auto const n = m_lastState.touchFactor;
        sstr << "touch (" << n << ");";
    },
        [](int8_t delta) { NN_UNUSED(delta); }
    );

    m_ReadOnlyMenu.emplace_back("Guidance",
        [&](std::stringstream& sstr) {
        if (m_lastState.isMarkerDetected)
        {
            if (m_LernWarning == true)
                sstr << "Learning too much";
            else
                sstr << "OK";
        }
    },
        [](int8_t delta) { NN_UNUSED(delta); }
    );
}

void PositionReaderModeState::Start()
{
    nn::irsensor::RunMarkerPositionReader(m_IrCameraHandle);

    m_CurrentPosition.x = nn::irsensor::MarkerPositionWidth / 2;
    m_CurrentPosition.y = nn::irsensor::MarkerPositionHeight / 2;
    m_LernWarning = false;
    m_PositionList.clear();
}

// Add pen trajectory to vector
void PositionReaderModeState::Update()
{
    int returnCount;
    nn::irsensor::GetMarkerPositionReaderStates(&m_lastState, &returnCount, 1, m_IrCameraHandle);

    if (m_lastState.isMarkerDetected)
    {
        m_CurrentPosition.x = (m_lastState.pentipPosition.x - nn::irsensor::MarkerPositionWidth / 2.0f) / static_cast<float>(nn::irsensor::MarkerPositionWidth) * m_DrawScale;
        m_CurrentPosition.y = (m_lastState.pentipPosition.y - nn::irsensor::MarkerPositionHeight / 2.0f) / static_cast<float>(nn::irsensor::MarkerPositionHeight) * m_DrawScale;

        m_CurrentPosition.x = std::max(std::min(m_CurrentPosition.x, 0.5f), -0.5f) * nn::irsensor::MarkerPositionWidth + nn::irsensor::MarkerPositionWidth / 2.0f;
        m_CurrentPosition.y = std::max(std::min(m_CurrentPosition.y, 0.5f), -0.5f) * nn::irsensor::MarkerPositionHeight + nn::irsensor::MarkerPositionHeight / 2.0f;

        if (m_lastState.touchFactor > 0.5f)
        {
            if (m_PositionList.size() < PositionMaxCount)
                m_PositionList.emplace_back(std::make_pair(m_CurrentPosition, true));
            else
            {
                for (size_t i = 1; i < m_PositionList.size(); ++i)
                    m_PositionList[i - 1] = m_PositionList[i];
                m_PositionList.back() = std::make_pair(m_CurrentPosition, true);
            }
        }
        else
        {
            if (m_PositionList.size() >= 1)
                m_PositionList[m_PositionList.size() - 1].second = false;
        }

        m_LernWarning = CheckJoyconLeaning(m_lastState);
        memcpy(&m_RotationMatrix, &m_lastState.cameraMatrix, sizeof(nn::util::Matrix4x3fType));
    }
}

void PositionReaderModeState::Render(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer, int index)
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_UNUSED(index);

    //    const nn::util::Uint8x4 Red = { { 255, 0, 0, 255 } };
    const nn::util::Uint8x4 White = { { 255, 255, 255, 255 } };

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;
    pPrimitiveRenderer->SetLineWidth(5.f);
    pPrimitiveRenderer->SetColor(White);

    // Position Reader Demo
    const float ratioScale = 1.0f; // Please change this value freely
    const float xBorderStart = -0.95f, xBorderEnd = 0.95f;
    const float yHalfSize = (xBorderEnd - xBorderStart) / 2.0f * ratioScale;
    const float yBorderStart = -yHalfSize, yBorderEnd = yHalfSize;

    nn::util::VectorSet(&begin, xBorderStart, yBorderStart, 0.0f);
    nn::util::VectorSet(&end, xBorderStart, yBorderEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xBorderEnd, yBorderEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&end, xBorderEnd, yBorderStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xBorderStart, yBorderStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);

    // Render pen trajectory
    for (size_t i = 1; i < m_PositionList.size(); i++)
    {
        if (m_PositionList[i - 1].second == false)
            continue;
        float xPenPosition = xBorderStart + (xBorderEnd - xBorderStart) * m_PositionList[i - 1].first.x / static_cast<float>(nn::irsensor::MarkerPositionWidth);
        float yPenPosition = yBorderStart + (yBorderEnd - yBorderStart) * m_PositionList[i - 1].first.y / static_cast<float>(nn::irsensor::MarkerPositionHeight);
        nn::util::VectorSet(&begin, xPenPosition, yPenPosition, 0.0f);
        xPenPosition = xBorderStart + (xBorderEnd - xBorderStart) * m_PositionList[i].first.x / static_cast<float>(nn::irsensor::MarkerPositionWidth);
        yPenPosition = yBorderStart + (yBorderEnd - yBorderStart) * m_PositionList[i].first.y / static_cast<float>(nn::irsensor::MarkerPositionHeight);
        nn::util::VectorSet(&end, xPenPosition, yPenPosition, 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    }

    // Render touch level as crossing cursor size
    if (m_lastState.isMarkerDetected)
    {
        const nn::util::Uint8x4 cursorColor = {
            static_cast<uint8_t>((1.0 - m_lastState.touchFactor) * 255),
            static_cast<uint8_t>(m_lastState.touchFactor * 255), 0, 255 };
        const float xPenPosition = xBorderStart + (xBorderEnd - xBorderStart) * m_CurrentPosition.x / static_cast<float>(nn::irsensor::MarkerPositionWidth);
        const float yPenPosition = yBorderStart + (yBorderEnd - yBorderStart) * m_CurrentPosition.y / static_cast<float>(nn::irsensor::MarkerPositionHeight);
        const float xStartCursor = xPenPosition - 0.02f - m_lastState.touchFactor * 0.06f; const float xEndCursor = xPenPosition + 0.02f + m_lastState.touchFactor * 0.06f;
        const float yStartCursor = yPenPosition - 0.03f - m_lastState.touchFactor * 0.1f; const float yEndCursor = yPenPosition + 0.03f + m_lastState.touchFactor * 0.1f;

        pPrimitiveRenderer->SetColor(cursorColor);
        nn::util::VectorSet(&begin, xStartCursor, yStartCursor, 0.0f);
        nn::util::VectorSet(&end, xEndCursor, yEndCursor, 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&begin, xStartCursor, yEndCursor, 0.0f);
        nn::util::VectorSet(&end, xEndCursor, yStartCursor, 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    }
}
