﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "HeartRateModeState.h"

#include <nn/nn_Assert.h>
#include <sstream>
#include <iomanip>

namespace Color
{
    const nn::util::Unorm8x4 White = { { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 Blue = { { 0, 60, 160, 255 } };
    const nn::util::Unorm8x4 Green = { { 0, 255, 0, 255 } };
    const nn::util::Unorm8x4 Orange = { { 255, 153, 0, 255 } };
    const nn::util::Unorm8x4 Red = { { 255, 0, 0, 255 } };
};

HeartRateModeState::HeartRateModeState(IrSensorMode* pNextProcessor, int* pMenuSelection, nn::irsensor::IrCameraHandle irCameraHandle)
    : IrSensorModeState(pNextProcessor, pMenuSelection, irCameraHandle)
    , m_pulseSignalSlidingWindowView(m_pulseSignal, 180, 32)
{
    m_HeartRateCaptureConfig.measurementDuration = ::nn::TimeSpanType::FromSeconds(30);
    m_HeartRateCaptureConfig.measurementDurationExtension = ::nn::TimeSpanType::FromSeconds(5);


    m_ReadOnlyMenu.emplace_back("Capture Id",
        [&](std::stringstream& sstr) {
        auto const n = m_lastState.userId;
        sstr << (n ? std::to_string(n) : "__");
    },
        [](int8_t delta) { NN_UNUSED(delta); }
    );

    m_ReadOnlyMenu.emplace_back("Status",
        [&](std::stringstream& sstr) {
            switch (m_lastState.status)
            {
            case nn::irsensor::HeartRateStatus::HeartRateStatus_WaitingForUser:
                sstr << "Waiting for user";
                break;
            case nn::irsensor::HeartRateStatus::HeartRateStatus_MeasurementInProgress:
                sstr << "In progress";
                break;
            case nn::irsensor::HeartRateStatus::HeartRateStatus_MeasurementInProgressExtension:
                sstr << "In progress (extension)";
                break;
            case nn::irsensor::HeartRateStatus::HeartRateStatus_MeasurementCompleted:
                sstr << "Completed";
                break;
            case nn::irsensor::HeartRateStatus::HeartRateStatus_MeasurementAborted:
                sstr << "Aborted";
                break;
            default:
                sstr << "__";
                break;
            }
    },
        [](int8_t delta) { NN_UNUSED(delta); }
    );

    m_ReadOnlyMenu.emplace_back("Progress",
        [&](std::stringstream& sstr) {
        sstr << std::fixed << std::setprecision(2) << m_lastState.progress * 100 << "%";
    },
        [](int8_t delta) { NN_UNUSED(delta); }
    );

    m_ReadOnlyMenu.emplace_back("BPM",
        [&](std::stringstream& sstr) {
        sstr << std::fixed << std::setprecision(2) << m_lastState.bpm;
    },
        [](int8_t delta) { NN_UNUSED(delta); }
    );

    m_ReadOnlyMenu.emplace_back("Confidence",
        [&](std::stringstream& sstr) {
        sstr << std::fixed << std::setprecision(2) << m_lastState.confidence * 100 << "%";
    },
        [](int8_t delta) { NN_UNUSED(delta); }
    );

    m_ReadOnlyMenu.emplace_back("Signal Quality Instant ",
        [&](std::stringstream& sstr) {
        sstr << std::fixed << std::setprecision(2) << m_lastState.signalQualityInstant * 100 << "%";
    },
        [](int8_t delta) { NN_UNUSED(delta); }
    );

    m_ReadOnlyMenu.emplace_back("Signal Quality Average",
        [&](std::stringstream& sstr) {
        sstr << std::fixed << std::setprecision(2) << m_lastState.signalQualityAverage * 100 << "%";
    },
        [](int8_t delta) { NN_UNUSED(delta); }
    );
}

const nn::util::Unorm8x4 HeartRateModeState::GetCaptureStateColor()
{
    if (m_lastState.progress == 0)
        return Color::White;
    if (m_lastState.status == nn::irsensor::HeartRateStatus::HeartRateStatus_MeasurementInProgress)
        return Color::Blue;
    if (m_lastState.status == nn::irsensor::HeartRateStatus::HeartRateStatus_MeasurementInProgressExtension)
        return Color::Orange;
    return m_lastState.confidence >= 0.6f ? Color::Green : Color::Red;
}

void HeartRateModeState::Start()
{
    nn::irsensor::RunHeartRateProcessor(m_IrCameraHandle, m_HeartRateCaptureConfig, m_HeartRateWorkBuffer);
    Reset();
}

void HeartRateModeState::Reset()
{
    m_lastState.status = nn::irsensor::HeartRateStatus::HeartRateStatus_Invalid;
    m_lastState.userId = 0;
    m_lastState.progress = 0;
    m_lastState.bpm = 0;
    m_lastState.confidence = 0;
    m_lastState.newSignalValuesCount = 0;
    m_lastState.newSignalValues = nullptr;

    m_pulseSignal.clear();

    nn::irsensor::ResetHeartRateProcessorState(m_IrCameraHandle);
}

void HeartRateModeState::Update()
{
    if (m_lastState.status != nn::irsensor::HeartRateStatus::HeartRateStatus_MeasurementInProgress && m_lastState.confidence >= 0.8f)
    {
        m_lastState.status = nn::irsensor::HeartRateStatus::HeartRateStatus_MeasurementCompleted;
        m_lastState.progress = 1.f;
        return;
    }

    nn::irsensor::HeartRateProcessorState state{};
    nn::Result result = nn::irsensor::GetHeartRateProcessorState(&state, m_IrCameraHandle);
    HandleResult(result);

    if ( result.IsSuccess() )
    {
        NN_SDK_ASSERT(state.status != nn::irsensor::HeartRateStatus::HeartRateStatus_Invalid);

        switch (state.status)
        {
        case nn::irsensor::HeartRateStatus::HeartRateStatus_MeasurementInProgressExtension:
        case nn::irsensor::HeartRateStatus::HeartRateStatus_MeasurementInProgress:
        case nn::irsensor::HeartRateStatus::HeartRateStatus_MeasurementCompleted:
        {
            m_pulseSignal.insert(m_pulseSignal.end(), state.newSignalValues, state.newSignalValues + state.newSignalValuesCount);
            break;
        }
        case nn::irsensor::HeartRateStatus::HeartRateStatus_MeasurementAborted:
            Reset();
            break;
        default:
            break;
        }

        m_lastState = state;
    }
}

void HeartRateModeState::Render(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer, int screenIndex)
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    const nn::util::Uint8x4 White = { { 255, 255, 255, 255 } };
    const nn::util::Uint8x4 Red = { { 255, 0, 0, 255 } };

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // Heart Rate Demo
    const float xStart = -0.8f, xEnd = -0.2f;
    float yStart, yEnd;
    if (screenIndex == 0)
        yStart = 0.9f, yEnd = 0.1f;
    else
        yStart = -0.1f, yEnd = -0.9f;
    float const progress = m_lastState.progress;
    float const progressBarHeight = 0.04f;

    pPrimitiveRenderer->SetColor(GetCaptureStateColor());
    nn::util::Vector3fType size;
    nn::util::Vector3fType corner;
    nn::util::VectorSet(&corner,
        xStart,
        yStart - progressBarHeight,
        0.0f);
    nn::util::VectorSet(&size,
        (xEnd - xStart) * progress,
        progressBarHeight,
        0.0f);

    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;
    pPrimitiveRenderer->DrawQuad(pCommandBuffer, corner, size);
    nn::util::VectorSet(&begin, xStart, yStart - progressBarHeight, 0.0f);
    nn::util::VectorSet(&end, xStart, yEnd, 0.0f);
    pPrimitiveRenderer->SetColor(White);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xEnd, yStart - progressBarHeight, 0.0f);
    nn::util::VectorSet(&end, xEnd, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    nn::util::VectorSet(&end, xEnd, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xStart, yEnd, 0.0f);
    nn::util::VectorSet(&end, xEnd, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    pPrimitiveRenderer->SetColor(White);
    nn::util::VectorSet(&begin, xStart, yStart - progressBarHeight, 0.0f);
    nn::util::VectorSet(&end, xEnd, yStart - progressBarHeight, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);


    size_t count;
    float scale;
    float const *data = m_pulseSignalSlidingWindowView.Get(&count, &scale);
    nn::util::Vector3f pA;
    for (size_t i = 0; i < count; ++i)
    {
        float const xvalue = xStart + (xEnd - xStart) * i / count;
        float const yRange = yEnd - yStart;
        float const yCenter = yStart + 0.5f * yRange;
        float const yvalue = data[i] * scale * yRange * 0.5f + yCenter;

        nn::util::Vector3f const pB(
            xvalue,
            yvalue,
            0.0f);

        if (i > 0)
        {
            pPrimitiveRenderer->SetColor(Red);
            pPrimitiveRenderer->DrawLine(pCommandBuffer, pA, pB);
        }
        pA = pB;
    }
}
