﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/TargetConfigs/build_Platform.h>
#include <nns/hid.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include "HidbusRondeHardwareTestTool_ApplicationHeap.h"
#include "HidbusRondeHardwareTestTool_Color.h"
#include "HidbusRondeHardwareTestTool_FontSystem.h"
#include "HidbusRondeHardwareTestTool_GraphicsSystem.h"
#include "HidbusRondeHardwareTestTool_Ronde.h"
#include "HidbusRondeHardwareTestTool_WindowMessage.h"

namespace {

const size_t ApplicationHeapSize = 64 * 1024 * 1024;

const int FrameBufferWidth = 1280;

const int FrameBufferHeight = 720;

const int FrameRate = 60;

void WriteRondeInformation(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const RondeInfo* const pRondeInfo,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetScale(2, 2);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Ronde Hardware Test Tool");

    pTextWriter->SetCursor(offsetX, offsetY + 50);
    pTextWriter->Print("Info");

    // RONDE State
    pTextWriter->SetCursor(offsetX + 40, offsetY + 90);
    pTextWriter->Print("Ronde State : ");

    pTextWriter->SetCursor(offsetX + 290, offsetY + 90);
    switch (pRondeInfo->state)
    {
    case HidbusState_NotGotHandle:
    case HidbusState_GotHandle:
        pTextWriter->SetTextColor(Color::Gray);
        pTextWriter->Print("Not Connected");
        break;
    case HidbusState_Enabled:
        pTextWriter->Print("Enabled");
        break;
    default:
        pTextWriter->SetTextColor(Color::Gray);
        pTextWriter->Print("Not Connected");
        break;
    }

    // FW Ver
    pTextWriter->SetCursor(offsetX + 40, offsetY + 130);
    pTextWriter->Print("FW Ver : %x.%x", pRondeInfo->fwVer >> 8, pRondeInfo->fwVer & 0xff);

    // UniqueId
    pTextWriter->SetCursor(offsetX + 40, offsetY + 170);

    uint32_t uniqueId32bit[3];
    for(int i = 0; i < 3; i++)
    {
        uniqueId32bit[i] = pRondeInfo->uniqueId[4 * i]           |
                           pRondeInfo->uniqueId[4 * i + 1] <<  8 |
                           pRondeInfo->uniqueId[4 * i + 2] << 16 |
                           pRondeInfo->uniqueId[4 * i + 3] << 24;
    }

    pTextWriter->Print("Unique ID");

    pTextWriter->SetCursor(offsetX + 80, offsetY + 210);
    pTextWriter->Print("%x %x %x", uniqueId32bit[2], uniqueId32bit[1], uniqueId32bit[0]);

    // Manufacture Cal
    pTextWriter->SetCursor(offsetX, offsetY + 290);
    pTextWriter->Print("Manufacture Cal");

    pTextWriter->SetCursor(offsetX + 40, offsetY + 330);
    pTextWriter->Print("Max Push : %d", pRondeInfo->maxPush);

    pTextWriter->SetCursor(offsetX + 40, offsetY + 370);
    pTextWriter->Print("Max Pull : %d", pRondeInfo->maxPull);

    pTextWriter->SetCursor(offsetX + 40, offsetY + 410);
    pTextWriter->Print("Zero Min : %d", pRondeInfo->zeroMin);

    pTextWriter->SetCursor(offsetX + 40, offsetY + 450);
    pTextWriter->Print("Zero Max : %d", pRondeInfo->zeroMax);



    pTextWriter->SetCursor(offsetX + 640, offsetY + 50);
    pTextWriter->Print("RONDE Error Count");

    pTextWriter->SetCursor(offsetX + 680, offsetY + 90);
    pTextWriter->Print("ADS Reset : %d", pRondeInfo->errorInfo.resetAdsCount);

    pTextWriter->SetCursor(offsetX + 680, offsetY + 130);
    pTextWriter->Print("UART Error");

    pTextWriter->SetCursor(offsetX + 720, offsetY + 170);
    pTextWriter->Print("Parity Error : %d", pRondeInfo->errorInfo.uartParityErrorCount);

    pTextWriter->SetCursor(offsetX + 720, offsetY + 210);
    pTextWriter->Print("Noise Error : %d", pRondeInfo->errorInfo.uartNoiseErrorCount);

    pTextWriter->SetCursor(offsetX + 720, offsetY + 250);
    pTextWriter->Print("Frame Error : %d", pRondeInfo->errorInfo.uartFrameErrorCount);

    pTextWriter->SetCursor(offsetX + 720, offsetY + 290);
    pTextWriter->Print("Overrun Error : %d", pRondeInfo->errorInfo.uartOverrunErrorCount);

    pTextWriter->SetCursor(offsetX + 720, offsetY + 330);
    pTextWriter->Print("DMA Error : %d", pRondeInfo->errorInfo.uartDmaErrorCount);



    pTextWriter->SetCursor(offsetX + 640, offsetY + 410);
    pTextWriter->Print("Polling Data");

    pTextWriter->SetCursor(offsetX + 680, offsetY + 450);
    pTextWriter->Print("Sensor : %d", pRondeInfo->sensor);

    pTextWriter->SetCursor(offsetX + 680, offsetY + 490);
    pTextWriter->Print("Thermistor : %d", pRondeInfo->thermistor);

    pTextWriter->SetCursor(offsetX + 680, offsetY + 530);
    pTextWriter->Print("SamplingNumber : %llu", pRondeInfo->samplingNumber);
}



#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;

void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

} // namespace

extern "C" void nnMain()
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif

    // RONDE 情報の構造体
    RondeInfo rondeInfo;

    // RONDE 用スレッドの作成
    nn::os::ThreadType rondeThread;
    static NN_ALIGNAS(4096) char s_RondeThreadStack[4096];
    nn::os::CreateThread(&rondeThread, RondeThread, nullptr, s_RondeThreadStack, sizeof(s_RondeThreadStack), 0);
    nn::os::StartThread(&rondeThread);

    ApplicationHeap applicationHeap;
    applicationHeap.Initialize(ApplicationHeapSize);

    GraphicsSystem* pGraphicsSystem = new GraphicsSystem();
    pGraphicsSystem->Initialize(
        &applicationHeap, FrameBufferWidth, FrameBufferHeight);

    EnableWindowMessage(pGraphicsSystem->GetNativeWindowHandle());

    FontSystem* pFontSystem = new FontSystem();
    pFontSystem->Initialize(&applicationHeap, pGraphicsSystem);

    nn::gfx::util::DebugFontTextWriter& textWriter =
        pFontSystem->GetDebugFontTextWriter();

    while (NN_STATIC_CONDITION(true))
    {
        bool quits = false;

        switch (GetWindowMessage(pGraphicsSystem->GetNativeWindowHandle()))
        {
        case WindowMessage_Close:
            quits = true;
            break;

        case WindowMessage_Active:
        case WindowMessage_Inactive:
        default:
            break;
        }

        if (quits)
        {
            break;
        }

        rondeInfo = GetRondeInfo();

        WriteRondeInformation(&textWriter,
                              &rondeInfo,
                              25,
                              15 + 175 * static_cast<float>(0));

        pGraphicsSystem->BeginDraw();
        pFontSystem->Draw();
        pGraphicsSystem->EndDraw();

        pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));
    }

    pFontSystem->Finalize();
    delete pFontSystem;

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    applicationHeap.Finalize();
}
