﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Color.h"
#include "ControllerManager.h"
#include "Audio.h"
#include "Demo.h"

namespace
{
    VirtualBallDescriptor g_BallDesc[DemoEnvironmentPattern_CountMax];

    const char* NameList[3] = {
        "[Bouncy]  Metal   Stone",
        " Bouncy  [Metal]  Stone",
        " Bouncy   Metal  [Stone]",
    };

    const int FrameBufferWidth = 1280;
    const int FrameBufferHeight = 720;
    const float FieldWidth = FrameBufferWidth;
    const float FieldHalfWidth = FieldWidth / 2.0f;
    const float FieldHalfHeight = FrameBufferHeight / 2.0f;
    const float GroundThickness = 20.0f;
    const float CameraOffset = FieldHalfHeight;
    const float WallThickness = 20.0f;
    const float WallHeight = 60.0f;

}

Demo& Demo::GetInstance() NN_NOEXCEPT
{
    static Demo instance;
    return instance;
}

const char* Demo::GetEnvironmentName() const NN_NOEXCEPT
{
    return NameList[m_Pattern];
}

void Demo::InitializeEnv() NN_NOEXCEPT
{
    auto pDesc = &(g_BallDesc[DemoEnvironmentPattern_BouncyBall]);
    pDesc->radius              = 30.0f;
    pDesc->weight              = 0.5f;
    pDesc->coefStaticFriction  = 0.10f;
    pDesc->coefDynamicFriction = 0.05f;
    pDesc->coefRestitutionWall = 0.80f;
    pDesc->coefRestitutionBall = 0.50f;

    pDesc->fieldVibrationAmplifierLimit = 0.20f;
    pDesc->fieldModulation = nn::hid::VibrationModulation::Make(0.00f, 0.50f, 1.00f, 0.90f);
    pDesc->wallVibrationAmplifierLimit  = 0.30f;
    pDesc->wallModulation  = nn::hid::VibrationModulation::Make(0.50f, 0.50f, 1.00f, 0.90f);

    pDesc->wallVibrationPattern  = "simple";
    pDesc->fieldVibrationPattern = "simple";
    pDesc->baseColor = Color::White;
    pDesc->colorPattern = Sphere::VertexColorPattern_RgbStripeSqueeze;
    pDesc->isRolling = true;

    AddEnvironment(DemoEnvironmentPattern_BouncyBall, pDesc, "Bouncy Ball");

    pDesc = &(g_BallDesc[DemoEnvironmentPattern_Metal]);
    pDesc->radius              = 30.0f;
    pDesc->weight              = 1.0f;
    pDesc->coefStaticFriction  = 0.30f;
    pDesc->coefDynamicFriction = 0.20f;
    pDesc->coefRestitutionWall = 0.40f;
    pDesc->coefRestitutionBall = 0.50f;

    pDesc->fieldVibrationAmplifierLimit = 0.50f;
    pDesc->fieldModulation = nn::hid::VibrationModulation::Make(0.30f, 0.20f, 0.60f, 1.00f);
    pDesc->wallVibrationAmplifierLimit  = 0.70f;
    pDesc->wallModulation  = nn::hid::VibrationModulation::Make(0.50f, 0.50f, 0.80f, 1.00f);

    pDesc->wallVibrationPattern  = "simple";
    pDesc->fieldVibrationPattern = "simple";
    pDesc->baseColor = Color::Yellow;
    pDesc->colorPattern = Sphere::VertexColorPattern_VerticalGradient;
    pDesc->isRolling = false;

    AddEnvironment(DemoEnvironmentPattern_Metal, pDesc, "Metal Ball");

    pDesc = &(g_BallDesc[DemoEnvironmentPattern_Stone]);
    pDesc->radius              = 30.0f;
    pDesc->weight              = 2.0f;
    pDesc->coefStaticFriction  = 0.60f;
    pDesc->coefDynamicFriction = 0.30f;
    pDesc->coefRestitutionWall = 0.10f;
    pDesc->coefRestitutionBall = 0.50f;

    pDesc->fieldVibrationAmplifierLimit = 0.50f;
    pDesc->fieldModulation = nn::hid::VibrationModulation::Make(0.80f, 0.10f, 1.00f, 0.50f);
    pDesc->wallVibrationAmplifierLimit  = 0.80f;
    pDesc->wallModulation  = nn::hid::VibrationModulation::Make(0.80f, 0.10f, 1.00f, 0.50f);

    pDesc->wallVibrationPattern  = "simple";
    pDesc->fieldVibrationPattern = "simple";
    pDesc->baseColor = Color::White;
    pDesc->colorPattern = Sphere::VertexColorPattern_VerticalGradient;
    pDesc->isRolling = false;

    AddEnvironment(DemoEnvironmentPattern_Stone, pDesc, "Stone Ball");

    m_Pattern = DemoEnvironmentPattern_BouncyBall;
}

// デモに使用するオブジェクトの初期化
void Demo::InitializeScene() NN_NOEXCEPT
{
    // 物体の設定
    m_Sphere.Make();
    m_Ball.SetFieldWidth(FieldWidth - WallThickness * 2.0f);
    m_Ball.SetDescriptor(GetVirtualBallDesc());
    m_Ball.SetVibrationMixMode(m_IsVibrationEnabled ? nn::hid::VibrationMixMode_MaxAmplitude : nn::hid::VibrationMixMode_None);

    // 振動ターゲットの設定
    auto pTargetL = m_Ball.GetVibrationTarget(0);
    auto pTargetR = m_Ball.GetVibrationTarget(1);
    ControllerManager::GetInstance().SetActiveVibrationTarget(pTargetL, pTargetR);
}

//コントローラによるデモ設定の変更
void Demo::UpdateEnv() NN_NOEXCEPT
{
    static nn::hid::NpadButtonSet previousButtons;
    const auto& currentButtons = ControllerManager::GetInstance().GetActiveController()->GetNpadButtonSet();
    nn::hid::NpadButtonSet downButtons = currentButtons & ~previousButtons;
    previousButtons = currentButtons;

    const nn::hid::NpadButtonSet ButtonMask = nn::hid::NpadButton::B::Mask;

    // Bでリセット
    if ((downButtons & ButtonMask) == ButtonMask)
    {
        ResetEnvironment();
    }
    else
    {
        // Rボタンで次のデモ環境に変更
        if (downButtons.Test(nn::hid::NpadButton::R::Index))
        {
            MoveNextEnvironment();
        }
        // Lボタンで次のデモ環境に変更
        else if (downButtons.Test(nn::hid::NpadButton::L::Index))
        {
            MovePreviousEnvironment();
        }
    }

    // Xボタンで振動のミキシングを変更する
    if (downButtons.Test(nn::hid::NpadButton::X::Index))
    {
        m_IsVibrationEnabled = !m_IsVibrationEnabled;
    }

    // Aボタンで衝突時の音声を再生するかどうかを変更する
    if (downButtons.Test(nn::hid::NpadButton::A::Index))
    {
        m_IsAudioEnabled = !m_IsAudioEnabled;
    }

    // Yボタンで物体の状態をグラフィックで表示するかどうかを変更する
    if (downButtons.Test(nn::hid::NpadButton::Y::Index))
    {
        m_IsGraphicsEnabled = !m_IsGraphicsEnabled;
    }
}

// ボールの位置の更新
void Demo::UpdateScene() NN_NOEXCEPT
{
    const auto& pController = ControllerManager::GetInstance().GetActiveController();
    const SixAxisState& sixAxisState = pController->GetSixAxisSensorState(0);
    nn::util::Vector3f gravity = sixAxisState.gravityVector;

    if (pController->GetSixAxisSensorCount() > 1)
    {
        //６軸センサーが２つある状態の場合、2つめが静止判定されていなければ合算する
        const SixAxisState& state = pController->GetSixAxisSensorState(1);
        auto isMotionLess = nn::hid::IsSixAxisSensorAtRest(state.sensorHandle);
        if (isMotionLess == false)
        {
            gravity += state.gravityVector;
        }
    }

    // ボールの座標を更新
    m_Ball.Update(gravity.GetX(), gravity.GetY(), gravity.GetZ());

    // 振動を再生するかどうか
    m_Ball.SetVibrationMixMode(m_IsVibrationEnabled ? nn::hid::VibrationMixMode_MaxAmplitude : nn::hid::VibrationMixMode_None);

    // 壁に衝突した際の音を再生する
    if (m_IsAudioEnabled)
    {
        if (m_Ball.IsHitWall())
        {
            auto pDesc = m_Ball.GetDesc();
            float volume = (1.0f - pDesc->coefRestitutionWall) * m_Ball.GetHitPowerWall() / 2.0f;
            volume = (volume < 0.0f) ? 0.0f : (volume > 3.0f) ? 3.0f : volume;
            float pitch = pDesc->coefRestitutionWall;

            if (volume > 0.1f)
            {
                AudioPlayer::GetInstance().PlaySound("hit",volume, pitch, m_Ball.GetPosition());
            }
        }
    }
}

void Demo::DrawEnv(GraphicsSystem* pGraphicsSystem) const NN_NOEXCEPT
{
    auto pTextWriter = &pGraphicsSystem->GetDebugFont();
    auto pDesc = m_Ball.GetDesc();

    pTextWriter->Draw(&pGraphicsSystem->GetCommandBuffer());
    pTextWriter->SetScale(1.0f, 1.0f);

    float yOffset = 160.0f;
    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetCursor(10.0f, yOffset);
    pTextWriter->Print("KeyConfig:");
    yOffset += 20.0f;

    // change ball type
    pTextWriter->SetCursor(10.0f, yOffset);
    pTextWriter->Print("(L/R) Ball Type:");
    pTextWriter->SetCursor(200.0f, yOffset);
    pTextWriter->Print("%s", GetEnvironmentName());
    yOffset += 20.0f;

    // VibrationMixMode
    pTextWriter->SetCursor(10.0f, yOffset);
    pTextWriter->Print("(X) Vibration:");
    pTextWriter->SetCursor(200.0f, yOffset);
    pTextWriter->Print(m_IsVibrationEnabled ? "[Enable]  Disable" : " Enable  [Disable]");
    yOffset += 20.0f;

    // PlaySound
    pTextWriter->SetCursor(10.0f, yOffset);
    pTextWriter->Print("(A) Audio:");
    pTextWriter->SetCursor(200.0f, yOffset);
    pTextWriter->Print(m_IsAudioEnabled ? "[Enable]  Disable" : " Enable  [Disable]");
    yOffset += 20.0f;

    // DisplayGraphics
    pTextWriter->SetCursor(10.0f, yOffset);
    pTextWriter->Print("(Y) Graphics:");
    pTextWriter->SetCursor(200.0f, yOffset);
    pTextWriter->Print(m_IsGraphicsEnabled ? "[Enable]  Disable" : " Enable  [Disable]");
    yOffset += 20.0f;

    pTextWriter->SetCursor(10.0f, yOffset);
    pTextWriter->Print("(B) Reset All configuration.");
    yOffset += 20.0f;

    pTextWriter->SetCursor(10.0f, yOffset);
    pTextWriter->Print("(+) and (-) Exit");
    yOffset += 40.0f;

    pTextWriter->SetCursor(10.0f, yOffset);
    pTextWriter->Print("Ball:");
    pTextWriter->SetCursor(200.0f, yOffset);
    pTextWriter->Print("%s", m_Ball.IsMoving() ? "M" : "S");
    pTextWriter->SetCursor(250.0f, yOffset);
    pTextWriter->Print("X = %4.3f", m_Ball.GetPosition());
    pTextWriter->SetCursor(500.0f, yOffset);
    pTextWriter->Print("V = %4.3f", m_Ball.GetVelocity());
    yOffset += 20.0f;

    pTextWriter->SetCursor(10.0f, yOffset);
    pTextWriter->Print("Weight:");
    pTextWriter->SetCursor(200.0f, yOffset);
    pTextWriter->Print("%4.3f", pDesc->weight);
    yOffset += 20.0f;

    pTextWriter->SetCursor(10.0f, yOffset);
    pTextWriter->Print("Friction:");
    pTextWriter->SetCursor(200.0f, yOffset);
    pTextWriter->Print("S: %4.3f", pDesc->coefStaticFriction);
    pTextWriter->SetCursor(300.0f, yOffset);
    pTextWriter->Print("D: %4.3f", pDesc->coefDynamicFriction);
    yOffset += 20.0f;

    pTextWriter->SetCursor(10.0f, yOffset);
    pTextWriter->Print("Restitution:");
    pTextWriter->SetCursor(200.0f, yOffset);
    pTextWriter->Print("Wall: %4.3f", pDesc->coefRestitutionWall);
    yOffset += 20.0f;

    pTextWriter->SetCursor(10.0f, yOffset);
    pTextWriter->Print("Field Vibration:");
    pTextWriter->SetCursor(200.0f, yOffset);
    pTextWriter->Print("Limit: %4.3f", pDesc->fieldVibrationAmplifierLimit);
    pTextWriter->SetCursor(350.0f, yOffset);
    pTextWriter->Print("AmpL: %4.3f", pDesc->fieldModulation.gainLow);
    pTextWriter->SetCursor(500.0f, yOffset);
    pTextWriter->Print("AmpH: %4.3f", pDesc->fieldModulation.gainHigh);
    yOffset += 20.0f;

    pTextWriter->SetCursor(200.0f, yOffset);
    pTextWriter->Print("FreqL: %4.3f", pDesc->fieldModulation.pitchLow);
    pTextWriter->SetCursor(350.0f, yOffset);
    pTextWriter->Print("FreqH: %4.3f", pDesc->fieldModulation.pitchHigh);
    yOffset += 20.0f;

    pTextWriter->SetCursor(10.0f, yOffset);
    pTextWriter->Print("Wall Vibration:");
    pTextWriter->SetCursor(200.0f, yOffset);
    pTextWriter->Print("Limit: %4.3f", pDesc->wallVibrationAmplifierLimit);
    pTextWriter->SetCursor(350.0f, yOffset);
    pTextWriter->Print("AmpL: %4.3f", pDesc->wallModulation.gainLow);
    pTextWriter->SetCursor(500.0f, yOffset);
    pTextWriter->Print("AmpH: %4.3f", pDesc->wallModulation.gainHigh);
    yOffset += 20.0f;

    pTextWriter->SetCursor(200.0f, yOffset);
    pTextWriter->Print("FreqL: %4.3f", pDesc->wallModulation.pitchLow);
    pTextWriter->SetCursor(350.0f, yOffset);
    pTextWriter->Print("FreqH: %4.3f", pDesc->wallModulation.pitchHigh);
    yOffset += 20.0f;
}

void Demo::DrawScene(GraphicsSystem* pGraphicsSystem) const NN_NOEXCEPT
{
    auto pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    auto pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();
    auto pDesc = m_Ball.GetDesc();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::Vector3fType cameraPosition = { 0.f, -5.0f, CameraOffset };
    nn::util::Vector3fType cameraTarget = { 0.f, 0.f, CameraOffset };
    nn::util::Vector3fType cammeraUp = { 0.f, 0.f, 1.f };
    nn::util::MatrixLookAtRightHanded(&viewMatrix, cameraPosition, cameraTarget, cammeraUp);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);

    nn::util::MatrixOrthographicOffCenterRightHanded(&projectionMatrix, -FieldHalfWidth, FieldHalfWidth, -FieldHalfHeight, FieldHalfHeight, -100.0f, 100.0f);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);

    // 地面と壁
    modelMatrix = nn::util::Matrix4x3f::Identity();
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
    pPrimitiveRenderer->SetColor(Color::Green);
    pPrimitiveRenderer->SetLineWidth(1.0f);
    pPrimitiveRenderer->DrawCube(
        pCommandBuffer,
        nns::gfx::PrimitiveRenderer::Surface_Solid,
        nn::util::MakeVector3fType(0.0f, 0.0f, GroundThickness / 2.0f),
        nn::util::MakeVector3fType(FieldWidth, 1.0f, GroundThickness)
    );
    pPrimitiveRenderer->DrawCube(
        pCommandBuffer,
        nns::gfx::PrimitiveRenderer::Surface_Solid,
        nn::util::MakeVector3fType(-FieldHalfWidth + WallThickness / 2.0f, 0.0f, GroundThickness + (WallHeight / 2.0f)),
        nn::util::MakeVector3fType(WallThickness, 1.0f, WallHeight)
    );
    pPrimitiveRenderer->DrawCube(
        pCommandBuffer,
        nns::gfx::PrimitiveRenderer::Surface_Solid,
        nn::util::MakeVector3fType(FieldHalfWidth - WallThickness / 2.0f, 0.0f, GroundThickness + (WallHeight / 2.0f)),
        nn::util::MakeVector3fType(WallThickness, 1.0f, WallHeight)
    );
    // 天井
    pPrimitiveRenderer->DrawCube(
        pCommandBuffer,
        nns::gfx::PrimitiveRenderer::Surface_Solid,
        nn::util::MakeVector3fType(0.0f, 0.0f, GroundThickness + WallHeight + GroundThickness / 2.0f),
        nn::util::MakeVector3fType(FieldWidth, 1.0f, GroundThickness)
    );
    // 物体
    pPrimitiveRenderer->SetColor(pDesc->baseColor);
    pPrimitiveRenderer->SetLineWidth(3.0f);
    nns::gfx::PrimitiveRenderer::PrimitiveMesh meshBuffer;

    meshBuffer.Initialize(
        pPrimitiveRenderer->GetGpuBuffer(),
        Sphere::VertexArraySize,
        Sphere::PolygonFaceCount * 3,
        nns::gfx::PrimitiveRenderer::VertexFormat_Default
    );

    uint32_t* pIndexData = meshBuffer.GetIndexBufferCpuAddress();
    memcpy(pIndexData, m_Sphere.FaceTable, Sphere::PolygonFaceCount * 3 * sizeof(uint32_t));

    nn::util::Float3* pPos = static_cast<nn::util::Float3*>(meshBuffer.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Pos));
    nn::util::Float2* pUv = static_cast<nn::util::Float2*>(meshBuffer.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Uv));
    nn::util::Float4* pColor = static_cast<nn::util::Float4*>(meshBuffer.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Color));
    memcpy(pPos, m_Sphere.VertexPos, Sphere::VertexArraySize * sizeof(nn::util::Float3));
    memcpy(pUv, m_Sphere.VertexUv, Sphere::VertexArraySize * sizeof(nn::util::Float2));
    memcpy(pColor, &(m_Sphere.VertexColor[pDesc->colorPattern]), Sphere::VertexArraySize * sizeof(nn::util::Float4));

    modelMatrix = nn::util::Matrix4x3f::Identity();
    nn::util::Vector3f scale;
    nn::util::VectorSet(&scale, pDesc->radius, pDesc->radius, pDesc->radius);

    if (pDesc->isRolling)
    {
        nn::util::Vector3f rotate;
        nn::util::VectorSet(&rotate, 0.0f, m_Ball.GetPosition() / pDesc->radius, 0.0f);
        nn::util::MatrixSetScaleRotateXyz(&modelMatrix, scale, rotate);
    }
    else
    {
        nn::util::MatrixSetScale(&modelMatrix, scale);
    }

    nn::util::Vector3f translate;
    nn::util::VectorSet(&translate, m_Ball.GetPosition(), 0.0f, GroundThickness + pDesc->radius);
    nn::util::MatrixSetTranslate(&modelMatrix, translate);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
    pPrimitiveRenderer->DrawUserMesh(
        pCommandBuffer,
        nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList,
        &meshBuffer);
}
