﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Assert.h>

#include "File.h"
#include "Audio.h"

namespace
{
    NN_ALIGNAS(4096) char g_AudioWorkBuffer[AudioPlayer::DefaultAudioWorkBufferSize];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_AudioWaveBuffer[AudioPlayer::DefaultAudioWaveBufferSize];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_AudioEffectBuffer[AudioPlayer::DefaultAudioEffectBufferSize];
}

void AudioPlayer::Update() NN_NOEXCEPT
{
    for (auto it = m_AudioHandles.begin(); it != m_AudioHandles.end(); ++it)
    {
        auto pVoiceType = &(it->second->voiceType);
        nn::audio::SetVoiceVolume(pVoiceType, m_VoiceVolume);
        nn::audio::SetVoicePitch(pVoiceType, m_VoicePitch);
    }

    if (m_SystemEvent.TryWait())
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::audio::RequestUpdateAudioRenderer(m_Handle, &m_Config));
    }
}

AudioPlayer& AudioPlayer::GetInstance()
{
    static AudioPlayer instance;
    return instance;
}

void AudioPlayer::Initialize() NN_NOEXCEPT
{
    //アロケータの初期化
    m_WorkBufferAllocator.Initialize(g_AudioWorkBuffer, sizeof(g_AudioWorkBuffer));

    m_WaveBuffer = g_AudioWaveBuffer;
    m_WaveBufferSize = sizeof(g_AudioWaveBuffer);
    m_WaveBufferAllocator.Initialize(m_WaveBuffer, m_WaveBufferSize);

    m_EffectBuffer = g_AudioEffectBuffer;
    m_EffectBufferSize = sizeof(g_AudioEffectBuffer);
    m_EffectBufferAllocator.Initialize(m_EffectBuffer, m_EffectBufferSize);

    //レンダラのパラメータを指定します。
    nn::audio::InitializeAudioRendererParameter(&m_Parameter);
    m_Parameter.sampleRate = DefaultRenderRate;
    m_Parameter.sampleCount = DefaultRenderRate / 200;
    m_Parameter.mixBufferCount = FinalMixBufferCount + SubMixBufferCount;
    m_Parameter.voiceCount = DefaultVoiceCount;
    m_Parameter.sinkCount = DefaultSinkCount;
    m_Parameter.effectCount = DefaultEffectCount;
    m_Parameter.subMixCount = SubMixBufferCount;

    m_MainBus[nn::audio::ChannelMapping_FrontLeft] = 4;
    m_MainBus[nn::audio::ChannelMapping_FrontRight] = 5;
    m_AuxBusA[nn::audio::ChannelMapping_FrontLeft] = 0;
    m_AuxBusA[nn::audio::ChannelMapping_FrontRight] = 1;
    m_AuxBusB[nn::audio::ChannelMapping_FrontLeft] = 2;
    m_AuxBusB[nn::audio::ChannelMapping_FrontRight] = 3;

    // パラメータがシステムでサポートされているかどうかを確認します。
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(m_Parameter),
        "Invalid AudioRendererParameter specified."
    );

    m_WorkBufferSize = nn::audio::GetAudioRendererWorkBufferSize(m_Parameter);
    m_WorkBuffer = m_WorkBufferAllocator.Allocate(m_WorkBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(m_WorkBuffer);

    // AudioRendererの初期化
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(&m_Handle, &m_SystemEvent, m_Parameter, m_WorkBuffer, m_WorkBufferSize).IsSuccess(),
        "Failed to open AudioRenderer"
    );

    // AudioRendererConfigの初期化
    m_ConfigBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(m_Parameter);
    m_ConfigBuffer = m_WorkBufferAllocator.Allocate(m_ConfigBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(m_ConfigBuffer);
    nn::audio::InitializeAudioRendererConfig(&m_Config, m_Parameter, m_ConfigBuffer, m_ConfigBufferSize);

    // FinalMix / SubMixの初期化
    nn::audio::AcquireFinalMix(&m_Config, &m_FinalMix, FinalMixBufferCount);
    nn::audio::AcquireSubMix(&m_Config, &m_SubMix0, m_Parameter.sampleRate, 1);
    nn::audio::AcquireSubMix(&m_Config, &m_SubMix1, m_Parameter.sampleRate, 1);

    // オーディオ出力デバイスへの入力を設定します。
    // mainBus に指定したミックスバッファのインデックスに応じて出力チャンネルが決定されます。
    // mainBus[nn::audio::ChannelMapping_FrontLeft] が L チャンネルに、
    // mainBus[nn::audio::ChannelMapping_FrontRight] が R チャンネルにそれぞれ出力されます。
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::audio::AddDeviceSink(&m_Config, &m_DeviceSink, &m_FinalMix, m_MainBus, ChannelCount, "MainAudioOut"));
    // 初期設定ではSubMixをそのままFinalMixに流し込む
    nn::audio::SetSubMixDestination(&m_Config, &m_SubMix0, &m_FinalMix);
    nn::audio::SetSubMixMixVolume(&m_SubMix0, &m_FinalMix, 1.0f, 0, m_MainBus[0]);
    nn::audio::SetSubMixDestination(&m_Config, &m_SubMix1, &m_FinalMix);
    nn::audio::SetSubMixMixVolume(&m_SubMix1, &m_FinalMix, 1.0f, 0, m_MainBus[1]);

    // メモリプール
    NN_ABORT_UNLESS(
        nn::audio::AcquireMemoryPool(&m_Config, &m_WaveBufferMemoryPool, m_WaveBuffer, m_WaveBufferSize));
    NN_ABORT_UNLESS(
        nn::audio::RequestAttachMemoryPool(&m_WaveBufferMemoryPool));

    NN_ABORT_UNLESS(
        nn::audio::AcquireMemoryPool(&m_Config, &m_EffectBufferMemoryPool, m_EffectBuffer, m_EffectBufferSize));
    NN_ABORT_UNLESS(
        nn::audio::RequestAttachMemoryPool(&m_EffectBufferMemoryPool));


    // 設定したパラメータをレンダラに反映させます。
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::audio::RequestUpdateAudioRenderer(m_Handle, &m_Config));

    // レンダリングを開始します。
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::audio::StartAudioRenderer(m_Handle));
}

void AudioPlayer::Finalize() NN_NOEXCEPT
{
    nn::audio::StopAudioRenderer(m_Handle);
    nn::audio::CloseAudioRenderer(m_Handle);
    nn::os::DestroySystemEvent(m_SystemEvent.GetBase());

    for (auto it = m_AudioHandles.begin(); it != m_AudioHandles.end(); ++it)
    {
        m_WaveBufferAllocator.Free(it->second->pData);
        delete it->second;
    }
    m_AudioHandles.clear();

    if (m_WorkBuffer)
    {
        m_WorkBufferAllocator.Free(m_WorkBuffer);
        m_WorkBuffer = nullptr;
    }

    if (m_ConfigBuffer)
    {
        m_WorkBufferAllocator.Free(m_ConfigBuffer);
        m_ConfigBuffer = nullptr;
    }

    if (m_WaveBuffer)
    {
        m_WaveBufferAllocator.Free(m_WaveBuffer);
        m_WaveBuffer = nullptr;
    }

    if (m_EffectBuffer)
    {
        m_EffectBufferAllocator.Free(m_EffectBuffer);
        m_EffectBuffer = nullptr;
    }
}

void AudioPlayer::PlaySound(const FileLabel& label, float volume, float pitch, float pos) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(volume >= 0.0f);
    NN_ABORT_UNLESS(pitch >= 0.0f);

    auto pData = m_AudioHandles.at(label);
    nn::audio::VoiceType* pVoice = &(pData->voiceType);
    nn::audio::WaveBuffer* pWaveBuffer = &(pData->waveBuffer);

    nn::audio::SetVoicePitch(pVoice, pitch);

    if (pos < 0.0f)
    {
        nn::audio::SetVoiceDestination(&m_Config, pVoice, &m_SubMix0);
        nn::audio::SetVoiceMixVolume(pVoice, &m_SubMix0, volume, 0, 0);
    }
    else
    {
        nn::audio::SetVoiceDestination(&m_Config, pVoice, &m_SubMix1);
        nn::audio::SetVoiceMixVolume(pVoice, &m_SubMix1, volume, 0, 0);
    }

    nn::audio::SetVoicePlayState(pVoice, nn::audio::VoiceType::PlayState_Play);

    if (pWaveBuffer == nn::audio::GetReleasedWaveBuffer(pVoice))
    {
        nn::audio::AppendWaveBuffer(pVoice, pWaveBuffer);
    }
}

void AudioPlayer::LoadAudioFile(const FileLabel& label) NN_NOEXCEPT
{
    AudioFile* pData = File::GetInstance().GetAudioFile(label);
    AudioHandle* handle = new AudioHandle();
    LoadWavFile(handle, pData->data, pData->size);
    m_AudioHandles.insert(std::make_pair(label, handle));
}

void AudioPlayer::LoadWavFile(AudioHandle* handle, void* wavData, size_t wavDataSize) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(handle);
    NN_ASSERT_NOT_NULL(wavData);
    NN_ABORT_UNLESS(wavDataSize > 0);

    nn::audio::WaveBuffer* pBuffer = &(handle->waveBuffer);
    nn::audio::VoiceType*  pVoice  = &(handle->voiceType);
    nns::audio::WavFormat* pFormat = &(handle->wavFormat);
    void** pDataBuffer = &(handle->pData);

    handle->priority = 0;
    handle->isLoop = false;

    auto result = nns::audio::ParseWavFormat(
        pFormat, wavData, ((wavDataSize < 1024) ? wavDataSize : 1024));

    NN_ABORT_UNLESS_EQUAL(result, nns::audio::WavResult_Success);

    //アロケータからメモリを確保する
    size_t dataSize = pFormat->dataSize;
    *(pDataBuffer) = m_WaveBufferAllocator.Allocate(dataSize, nn::audio::BufferAlignSize);
    NN_ASSERT_NOT_NULL(*(pDataBuffer));
    memcpy(*(pDataBuffer), &(reinterpret_cast<uint8_t*>(wavData)[pFormat->dataOffset]), dataSize);

    pBuffer->buffer = *(pDataBuffer);
    pBuffer->size = dataSize;
    pBuffer->startSampleOffset = 0;
    pBuffer->loop = handle->isLoop;
    pBuffer->isEndOfStream = false;
    pBuffer->pContext = nullptr;
    pBuffer->contextSize = 0;

    nn::audio::SampleFormat sampleFormat;
    switch (pFormat->bitsPerSample)
    {
    case 8:
        sampleFormat = nn::audio::SampleFormat_PcmInt8;
        pBuffer->endSampleOffset = static_cast<int32_t>(pFormat->dataSize) / pFormat->channelCount;
        break;
    case 16:
        sampleFormat = nn::audio::SampleFormat_PcmInt16;
        pBuffer->endSampleOffset = static_cast<int32_t>(pFormat->dataSize / 2) / pFormat->channelCount;
        break;
    case 24:
        sampleFormat = nn::audio::SampleFormat_PcmInt24;
        pBuffer->endSampleOffset = static_cast<int32_t>(pFormat->dataSize / 3) / pFormat->channelCount;
        break;
    case 32:
        sampleFormat = nn::audio::SampleFormat_PcmInt32;
        pBuffer->endSampleOffset = static_cast<int32_t>(pFormat->dataSize / 4) / pFormat->channelCount;
        break;
    default:
        sampleFormat = nn::audio::SampleFormat_PcmFloat;
        pBuffer->endSampleOffset = static_cast<int32_t>(pFormat->dataSize / 4) / pFormat->channelCount;
        break;
    }

    nn::audio::AcquireVoiceSlot(&m_Config, pVoice, pFormat->sampleRate, pFormat->channelCount, sampleFormat, handle->priority, nullptr, 0);
    nn::audio::SetVoicePlayState(pVoice, nn::audio::VoiceType::PlayState_Stop);
    nn::audio::AppendWaveBuffer(pVoice, pBuffer);
}

