﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   SpeedChangeableVibrationPlayer クラスの定義
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_VibrationPlayer.h>

namespace nns { namespace hid {

/**
 * @brief   @ref nn::hid::VibrationPlayer に再生スピードを可変にする機能を加えた振動ノードです。
 *
 * @details
 *  @ref nn::hid::VibrationPlayer を継承した上で、再生スピード変更に関する機能を追加したクラスです。
 *  @ref nns:hid::SpeedChangeableVibrationPlayer::SetPlaySpeed 関数を利用して再生スピードを設定します。
 *
 */
class SpeedChangeableVibrationPlayer : public nn::hid::VibrationPlayer
{
    NN_DISALLOW_COPY(SpeedChangeableVibrationPlayer);
    NN_DISALLOW_MOVE(SpeedChangeableVibrationPlayer);

public:
    /**
     * @brief       コンストラクタです。
     *
     * @post
     *              - IsLoaded() == false
     *              - GetPlaySpeed() == 1.0f;
     */
    SpeedChangeableVibrationPlayer() NN_NOEXCEPT;

    /**
     * @brief       デストラクタです。
     */
    virtual ~SpeedChangeableVibrationPlayer() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       SpeedChangeableVibrationPlayer にロードされた振動ファイルの再生スピードを指定します。
     *
     * @param[in]   speed   振動ファイルを再生する際の速度の倍率を指定します。1.0f を指定すると本来のスピードで再生されます。
     *
     * @pre
     *              - speed >= 0.0f
     * @post
     *              - GetPlaySpeed() = speed
     */
    void SetPlaySpeed(float speed) NN_NOEXCEPT;

    /**
     * @brief       SpeedChangeableVibrationPlayer にロードされた振動ファイルの再生スピードを取得します。
     *
     * @return      振動ファイルを再生する際の速度の倍率を返します。
     */
    float GetPlaySpeed() const NN_NOEXCEPT;

protected:
    virtual void OnNextSampleRequired(
        nn::hid::VibrationValue* pValue,
        nn::hid::VibrationNodeConnection::List* pInputConnections) NN_NOEXCEPT NN_OVERRIDE;

private:
    float m_PlaySpeed;
    float m_Count;
};

}} // namespace nns::hid
