﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>

#include "Color.h"
#include "VibrationValueDrawer.h"

namespace
{
    const int AxisHeight = 100;
    const int AxisWidth  = 250;

    const float FrequencyLow = static_cast<float>(nn::hid::VibrationFrequencyLowDefault);
    const float FrequencyHigh = static_cast<float>(nn::hid::VibrationFrequencyHighDefault);

    float CalcXOffsetFromFrequency(float frequency) NN_NOEXCEPT
    {
        //グラフ上で周波数を対数スケールで表示するための変換区分テーブル
        const float thresholds[] = {
            100.43f, 102.63f, 104.88f, 107.17f, 109.52f, 111.92f, 114.37f, 116.87f,
            119.43f, 122.05f, 124.72f, 127.45f, 130.24f, 133.09f, 136.01f, 138.99f,
            142.03f, 145.14f, 148.32f, 151.57f, 154.88f, 158.28f, 161.74f, 165.28f,
            168.90f, 172.60f, 176.38f, 180.24f, 184.19f, 188.22f, 192.35f, 196.56f,
            200.86f, 205.26f, 209.75f, 214.35f, 219.04f, 223.84f, 228.74f, 233.75f,
            238.87f, 244.10f, 249.44f, 254.90f, 260.48f, 266.19f, 272.02f, 277.97f,
            284.06f, 290.28f, 296.64f, 303.13f, 309.77f, 316.55f, 323.48f, 330.57f,
            337.81f, 345.20f, 352.76f, 360.49f, 368.38f, 376.45f, 384.69f, 393.11f,
            401.72f, 410.52f, 419.51f, 428.69f, 438.08f, 447.67f, 457.48f, 467.49f,
            477.73f, 488.19f, 498.88f, 509.81f };

        const int ThresholdsCount = sizeof(thresholds) / sizeof(thresholds[0]);

        for (int idx = 0; idx < ThresholdsCount; idx++)
        {
            if (frequency < thresholds[idx])
            {
                return static_cast<float>(idx) / ThresholdsCount;
            }
        }

        return 1.0f;
    }
}

void VibrationValueDrawer::SetPosition(float x, float y) NN_NOEXCEPT
{
    m_X = x;
    m_Y = y;
}

void VibrationValueDrawer::SetScale(float scale) NN_NOEXCEPT
{
    m_Scale = scale;
}

void VibrationValueDrawer::SetBrightColor(bool isBrightColorEnabled) NN_NOEXCEPT
{
    m_ColorPatternId = isBrightColorEnabled ? 0 : 1;
}

void VibrationValueDrawer::DrawVibrationValue(const nn::hid::VibrationValue& vibrationValue, bool isTextPrinted) const NN_NOEXCEPT
{
    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);

    m_pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    m_pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    m_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    m_pPrimitiveRenderer->SetLineWidth(1.0f);


    DrawAxis();

    DrawAmplitudeAndFrequency(
        vibrationValue.amplitudeLow,
        vibrationValue.frequencyLow,
        true,
        isTextPrinted);

    DrawAmplitudeAndFrequency(
        vibrationValue.amplitudeHigh,
        vibrationValue.frequencyHigh,
        false,
        isTextPrinted);
}

void VibrationValueDrawer::DrawTimeSeriesGraph(
    const VibrationValueBuffer& buffer,
    bool isDrawLowFreqBand ) const NN_NOEXCEPT
{
    const nn::util::Unorm8x4 MainAxisColor[] = { Color::White, Color::DimGray};
    const nn::util::Unorm8x4 VibrationValueColor[] = { Color::Maroon, Color::Green };

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);

    m_pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    m_pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    m_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    float x = m_X + 5;
    float y = m_Y + 5;
    float w = AxisWidth * m_Scale;
    float h = AxisHeight * m_Scale;

    m_pPrimitiveRenderer->SetColor(MainAxisColor[m_ColorPatternId]);
    m_pPrimitiveRenderer->SetLineWidth(3.0f);
    m_pPrimitiveRenderer->Draw2DFrame(m_pCommandBuffer, x, y, w, h);

    for (int i = 0; i < VibrationValueBuffer::BufferSize - 1; i++)
    {
        nn::util::Float4 startPoint;
        nn::util::Float4 endPoint;

        buffer.GetAmplitudes(&startPoint, i);
        buffer.GetAmplitudes(&endPoint, i + 1);

        float x1 = x + w * i / VibrationValueBuffer::BufferSize;
        float x2 = x + w * (i + 1) / VibrationValueBuffer::BufferSize;
        float y1 = y + 3 + (h - 6) * (1.0f - (isDrawLowFreqBand ? startPoint.x : startPoint.y));
        float y2 = y + 3 + (h - 6) * (1.0f - (isDrawLowFreqBand ? endPoint.x : endPoint.y));

        m_pPrimitiveRenderer->SetColor(VibrationValueColor[0]);
        m_pPrimitiveRenderer->Draw2DLine(m_pCommandBuffer, x1, y1, x2, y2);

        y1 = y + 3 + (h - 6) * (1.0f - (isDrawLowFreqBand ? startPoint.z : startPoint.w));
        y2 = y + 3 + (h - 6) * (1.0f - (isDrawLowFreqBand ? endPoint.z : endPoint.w));

        m_pPrimitiveRenderer->SetColor(VibrationValueColor[1]);
        m_pPrimitiveRenderer->Draw2DLine(m_pCommandBuffer, x1, y1, x2, y2);
    }
}

void VibrationValueDrawer::DrawAxis() const NN_NOEXCEPT
{
    const nn::util::Unorm8x4 MainAxisColor[] = { Color::White ,Color::DimGray};
    const nn::util::Unorm8x4 SecondAxisColor[] = { Color::DimGray,{{ 4, 4, 4, 255 }}};

    float x = m_X + 5;
    float y = m_Y + 5;
    float w = AxisWidth * m_Scale;
    float h = AxisHeight * m_Scale;
    float xOffsetLow  = CalcXOffsetFromFrequency(FrequencyLow);
    float xOffsetHigh = CalcXOffsetFromFrequency(FrequencyHigh);

    m_pPrimitiveRenderer->SetColor(MainAxisColor[m_ColorPatternId]);
    m_pPrimitiveRenderer->Draw2DFrame(m_pCommandBuffer, x, y, w, h);

    m_pPrimitiveRenderer->SetColor(SecondAxisColor[m_ColorPatternId]);
    m_pPrimitiveRenderer->Draw2DLine(m_pCommandBuffer, x, y + h / 2.0f, x + w, y + h / 2.0f);
    m_pPrimitiveRenderer->Draw2DLine(m_pCommandBuffer, x + w * xOffsetLow, y, x + w * xOffsetLow, y + h);
    m_pPrimitiveRenderer->Draw2DLine(m_pCommandBuffer, x + w * xOffsetHigh, y, x + w * xOffsetHigh, y + h);
}

void VibrationValueDrawer::DrawAmplitudeAndFrequency(float amplitude, float frequency, bool isDrawLowFreqBand, bool isTextPrinted) const NN_NOEXCEPT
{
    const nn::util::Unorm8x4 VibrationValueColor[] = {{{ 0, 160, 0, 255 }},{{ 0, 20, 0, 255 }}};

    float x = m_X + 5.0f;
    float y = m_Y + 5.0f;
    float w = AxisWidth * m_Scale;
    float h = AxisHeight * m_Scale;
    float barHeight = amplitude * AxisHeight * m_Scale;
    float barWidth = 7.5f * m_Scale;

    if (isTextPrinted)
    {
        float xOffset = CalcXOffsetFromFrequency(isDrawLowFreqBand ? 80.0f : 240.0f);
        m_pWriter->SetTextColor(VibrationValueColor[m_ColorPatternId]);
        m_pWriter->SetScale(m_Scale * 0.8f, m_Scale * 0.8f);
        m_pWriter->SetCursor(x + w * xOffset , y + h);
        m_pWriter->Print("(%4.2f, %5.1fHz)", amplitude, frequency);
    }

    {
        float xOffset = CalcXOffsetFromFrequency(frequency);
        m_pPrimitiveRenderer->SetColor(VibrationValueColor[m_ColorPatternId]);
        m_pPrimitiveRenderer->Draw2DRect(m_pCommandBuffer,
            x + w * xOffset - barWidth / 2.0f,
            y + h - barHeight,
            barWidth,
            barHeight
        );
    }
}

void VibrationValueBuffer::AddVibrationValues(
    const nn::hid::VibrationValue& currentVibrationValue,
    const nn::hid::VibrationValue& actualVibrationValue) NN_NOEXCEPT
{
    m_BufferAddr = (m_BufferAddr + 1) % BufferSize;
    m_CurrentValues[m_BufferAddr] = currentVibrationValue;
    m_ActualValues[m_BufferAddr]  = actualVibrationValue;
}

void VibrationValueBuffer::GetAmplitudes(nn::util::Float4* values, int time) const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(values);
    NN_ASSERT_GREATER_EQUAL(time, 0);

    // time は最大でも BufferSize - 1 にする
    auto addrOffset = std::min(time, BufferSize - 1);

    // time が 0 のときは最新のデータを参照する。1の時は1つ前のデータを参照する。
    int addr = (m_BufferAddr + BufferSize - addrOffset) % BufferSize;
    values->x = m_CurrentValues[addr].amplitudeLow;
    values->y = m_CurrentValues[addr].amplitudeHigh;
    values->z = m_ActualValues[addr].amplitudeLow;
    values->w = m_ActualValues[addr].amplitudeHigh;
}
