﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Box.h"
#include "Color.h"

const float Box::LineLength    = 256.0f;
const float Box::LineClearance = LineLength / (LineCountMax - 1);

Box::Box() NN_NOEXCEPT
{
    SetPos(0.0f, 0.0f);

    //複数のGeneratorを1つのMixerに接続して出力する
    for (int i = 0; i < LineCountMax; i++)
    {
        m_Connection[i].Connect(&m_Generator[i], &m_Mixer);
    }

    // Mixerは振幅を加算する設定で使用する
    m_Mixer.SetMixMode(nn::hid::VibrationMixMode_AmplitudeSum);
}

void Box::SetPos(float baseX, float baseY) NN_NOEXCEPT
{
    for (int i = 0; i < LineCountMax; i++)
    {
        auto pos = m_Line[i].Pos;
        pos[0] = baseX + LineClearance * i;
        pos[1] = baseY;
        pos[2] = baseX + LineClearance * i;
        pos[3] = baseY + LineLength;

        m_Line[i].Intensity = 0.0f;
        m_Line[i].LineAttacked = false;
    }
}

void Box::Draw(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer) const NN_NOEXCEPT
{
    pPrimitiveRenderer->SetLineWidth(3.0f);

    for (int i = 0; i < LineCountMax; i++)
    {
        nn::util::Unorm8x4 color;

        auto& line = m_Line[i];
        auto x = line.Pos[0];

        if (line.LineAttacked)
        {
            if (line.Intensity > 0.95f)
            {
                color = Color::Red;
            }
            else
            {
                //線の色の輝度を薄くする
                for (int j = 0; j < 4; j++)
                {
                    color.v[j] = static_cast<uint8_t>(Color::Red.v[j] * line.Intensity);
                }
                //線を左右に移動する
                x += 5 * nn::util::CosEst(2.0f * nn::util::FloatPi * line.Intensity * 10.0f);
            }
        }
        else
        {
            color = Color::Aqua;
        }

        pPrimitiveRenderer->SetColor(color);
        pPrimitiveRenderer->Draw2DLine(pCommandBuffer, x, line.Pos[1], x, line.Pos[3]);
    }
}

void Box::Update(float x, float y) NN_NOEXCEPT
{
    //ポインタの座標
    int posX = static_cast<int>(x);
    int posY = static_cast<int>(y);

    for (int i = 0; i < LineCountMax; i++)
    {
        auto& line = m_Line[i];

        //接触状態の時
        if (line.LineAttacked)
        {
            //線の輝度は徐々に下げる。
            line.Intensity *= 0.975f;

            //線の輝度が一定以下になったら接触状態を解除する。
            if (line.Intensity < 0.10f)
            {
                line.LineAttacked = false;
                line.Intensity = 0.0f;
            }
        }
        else
        {
            //当たり判定の処理
            auto minX = static_cast<int>(line.Pos[0]);
            auto maxX = static_cast<int>(line.Pos[0]);
            auto minY = static_cast<int>(line.Pos[1]);
            auto maxY = static_cast<int>(line.Pos[3]);
            if (minY <= posY && posY <= maxY)
            {
                if ((m_PrePosX <= maxX && minX <= posX) || (posX <= maxX && minX <= m_PrePosX))
                {
                    line.LineAttacked = true;
                    line.Intensity = 1.0f;
                    m_Generator[i].Play();
                }
            }
        }
    }

    m_PrePosX = posX;
}

