﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/hid/hid_NpadJoy.h>

#include "HidNpadIntegrate_PluginManager.h"
#include "HidNpadIntegrate_SettingApi.h"
#include "HidNpadIntegrate_SettingApiPluginBase.h"

class NpadJoyHoldType : public ReadWriteBase
{
    NN_DISALLOW_COPY(NpadJoyHoldType);
    NN_DISALLOW_MOVE(NpadJoyHoldType);

public:
    NN_IMPLICIT NpadJoyHoldType(nn::hid::NpadJoyHoldType* pHoldType) NN_NOEXCEPT
    : m_pHoldType(pHoldType) {}

    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << (*m_pHoldType == nn::hid::NpadJoyHoldType_Vertical ? "Vertical" : "Horizontal");
    }

    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
        *m_pHoldType = static_cast<nn::hid::NpadJoyHoldType>(1 - *m_pHoldType);
    }

private:
    nn::hid::NpadJoyHoldType* m_pHoldType;
};

class SetNpadJoyHoldTypeApiState : public SettingApiPluginBase
{
    NN_DISALLOW_COPY(SetNpadJoyHoldTypeApiState);
    NN_DISALLOW_MOVE(SetNpadJoyHoldTypeApiState);

public:
    SetNpadJoyHoldTypeApiState() NN_NOEXCEPT {}

    ~SetNpadJoyHoldTypeApiState() NN_NOEXCEPT  NN_OVERRIDE {}

    void Initialize(int* pNextProcessor, int* pMenuSelection) NN_NOEXCEPT NN_OVERRIDE
    {
        SettingApiPluginBase::Initialize(pNextProcessor, pMenuSelection);
        m_HoldType = nn::hid::GetNpadJoyHoldType();
        m_pNpadJoyHoldType = new NpadJoyHoldType(&m_HoldType);
        MenuItem npadJoyHoldTypeMenuItem("  HoldType", m_pNpadJoyHoldType);
        m_ReadWriteMenu.push_back(npadJoyHoldTypeMenuItem);
    }

    void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        delete m_pNpadJoyHoldType;
        SettingApiPluginBase::Finalize();
    }

    void Start() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::SetNpadJoyHoldType(m_HoldType);
        m_ResultMsg = "Success";
    }

    void Explain(
        nn::gfx::util::DebugFontTextWriter* pTextWriter,
        const float OffsetX, const float OffsetY) NN_NOEXCEPT NN_OVERRIDE
    {
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetScale(1.0f, 1.0f);

        pTextWriter->SetCursor(OffsetX, OffsetY);
        pTextWriter->
            Print("This function sets the hold type of Joy-Con.");
        pTextWriter->SetCursor(OffsetX, OffsetY + 30);
        pTextWriter->
            Print("This setting is effective only to Joy-Con with AssignmentMode \"Single Mode\"");
        pTextWriter->SetCursor(OffsetX, OffsetY + 60);
        pTextWriter->
            Print("The hold type is common setting for all Npads, not for individual ones.");
    }

    bool DrawHoldType(
        nn::gfx::util::DebugFontTextWriter* pTextWriter,
        const float OffsetX, const float OffsetY) NN_NOEXCEPT NN_OVERRIDE
    {
        pTextWriter->SetCursor(OffsetX, OffsetY);
        pTextWriter->Print("HoldType: %s",
            nn::hid::GetNpadJoyHoldType() == nn::hid::NpadJoyHoldType_Vertical ?
            "Vertical" : "Horizontal");

        return true;
    }

    bool DrawAssignment(
        int index, nn::gfx::util::DebugFontTextWriter* pTextWriter,
        const float OffsetX, const float OffsetY) NN_NOEXCEPT NN_OVERRIDE
    {
        pTextWriter->SetCursor(OffsetX, OffsetY);
        pTextWriter->Print("AssignmentMode: %s",
            nn::hid::GetNpadJoyAssignment(NpadIds[index]) == nn::hid::NpadJoyAssignmentMode_Dual ?
            "Dual" : "Single");

        return true;
    }

private:
    nn::hid::NpadJoyHoldType m_HoldType;
    ReadWriteBase* m_pNpadJoyHoldType;
};

namespace {
SET_PLUGIN(SetNpadJoyHoldTypeApiState, "SetNpadJoyHoldType");
}
