﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{HidKeyboardAndMouse_Main.cpp,PageSampleHidKeyboardAndMouse}
 *
 * @brief
 *  nn::hid ライブラリのキーボード API とマウス API の使い方を示すサンプルプログラム
 */

/**
 * @page PageSampleHidKeyboardAndMouse キーボードとマウスの状態取得
 * @tableofcontents
 *
 * @brief
 * nn::hid ライブラリを用いてキーボードとマウスの状態を取得するサンプルプログラムの解説です。
 *
 * @section PageSampleHidKeyboardAndMouse_SectionBrief 概要
 * キーボードとマウスの状態を表示します。
 *
 * @section PageSampleHidKeyboardAndMouse_SectionFileStructure ファイル構成
 * 本サンプルプログラムは
 * @link ../../../Samples/Sources/Applications/HidKeyboardAndMouse
 * Samples/Sources/Applications/HidKeyboardAndMouse @endlink 以下にあります。
 *
 * @section PageSampleHidKeyboardAndMouse_SectionNecessaryEnvironment 必要な環境
 * Windows 環境においては、 キーボードとマウスの操作がそのまま入力として扱われます。
 *
 * @section PageSampleHidKeyboardAndMouse_SectionHowToOperate 操作方法
 * サンプルプログラムを実行するとウィンドウが立ち上がります。
 * ウインドウ上には、キーボードについては、 アルファベット、数字、
 * 代表的な編集キー、及び修飾キーの状態と、その操作ログが、
 * マウスについては、カーソルの座標値、ボタンの状態、
 * 及びホイールのスクロール値が表示されます。
 * キーボードやマウスを操作すると状態の変化が反映されます。
 * キー配列は QWERTY 配列と AZERTY 配列を Tab キーの押下によって
 * 切り替えることができます。
 * 操作ログには、キーの押下、開放、リピートが記録されますが、
 * 修飾キーについてはリピートは無効化されています。
 * マウスの左ボタンをダブルクリックするとカーソルの色が切り替わります。
 *
 * @section PageSampleHidKeyboardAndMouse_SectionPrecaution 注意事項
 * コンソールウィンドウには何も表示されません。
 *
 * @section PageSampleHidKeyboardAndMouse_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleHidKeyboardAndMouse_SectionDetail 解説
 * 処理の流れは nns::hid サンプルライブラリのソースコード中にコメントで
 * 示されています。
 */

#include <algorithm>
#include <cstdlib>
#include <list>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/TargetConfigs/build_Platform.h>
#include <nn/util/util_MathTypes.h>
#include <nns/hid.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include "HidKeyboardAndMouse_ApplicationHeap.h"
#include "HidKeyboardAndMouse_Color.h"
#include "HidKeyboardAndMouse_FontSystem.h"
#include "HidKeyboardAndMouse_GraphicsSystem.h"
#include "HidKeyboardAndMouse_WindowMessage.h"

namespace {

const size_t ApplicationHeapSize = 128 * 1024 * 1024;

const int FrameBufferWidth = 1280;

const int FrameBufferHeight = 720;

const int FrameRate = 60;

//!< キーボードのキーを表す構造体です。
struct Key
{
    char value[8];
    char upper[8];
    float x;
    float y;
    int numPadKey;
    bool isToggleKey;
};

//!< キーボードの QWERTY 配列の定義です。
const Key KeyboardLayoutQwerty[] =
{
    { "a", "A",  60,  80 },
    { "b", "B", 145, 102 },
    { "c", "C", 105, 102 },
    { "d", "D", 100,  80 },
    { "e", "E",  98,  57 },
    { "f", "F", 123,  80 },
    { "g", "G", 140,  80 },
    { "h", "H", 160,  80 },
    { "i", "I", 200,  57 },
    { "j", "J", 183,  80 },
    { "k", "K", 200,  80 },
    { "l", "L", 223,  80 },
    { "m", "M", 185, 102 },
    { "n", "N", 165, 102 },
    { "o", "O", 215,  57 },
    { "p", "P", 235,  57 },
    { "q", "Q",  55,  57 },
    { "r", "R", 118,  57 },
    { "s", "S",  80,  80 },
    { "t", "T", 137,  57 },
    { "u", "U", 175,  57 },
    { "v", "V", 125, 102 },
    { "w", "W",  75,  57 },
    { "x", "X",  85, 102 },
    { "y", "Y", 155,  57 },
    { "z", "Z",  65, 102 },
    { "1", " ",  50,  35 },
    { "2", " ",  70,  35 },
    { "3", " ",  90,  35 },
    { "4", " ", 110,  35 },
    { "5", " ", 130,  35 },
    { "6", " ", 150,  35 },
    { "7", " ", 170,  35 },
    { "8", " ", 190,  35 },
    { "9", " ", 213,  35 },
    { "0", " ", 232,  35 },
};

//!< キーボードの AZERTY 配列の定義です。
const Key KeyboardLayoutAzerty[] =
{
    { "q", "Q",  60,  80 },
    { "b", "B", 145, 102 },
    { "c", "C", 105, 102 },
    { "d", "D", 100,  80 },
    { "e", "E",  98,  57 },
    { "f", "F", 123,  80 },
    { "g", "G", 140,  80 },
    { "h", "H", 160,  80 },
    { "i", "I", 200,  57 },
    { "j", "J", 183,  80 },
    { "k", "K", 200,  80 },
    { "l", "L", 223,  80 },
    { " ", " ", 186, 102 },
    { "n", "N", 165, 102 },
    { "o", "O", 215,  57 },
    { "p", "P", 235,  57 },
    { "a", "A",  55,  57 },
    { "r", "R", 118,  57 },
    { "s", "S",  80,  80 },
    { "t", "T", 137,  57 },
    { "u", "U", 175,  57 },
    { "v", "V", 125, 102 },
    { "z", "Z",  75,  57 },
    { "x", "X",  85, 102 },
    { "y", "Y", 155,  57 },
    { "w", "W",  63, 102 },
    { " ", "1",  50,  35 },
    { " ", "2",  70,  35 },
    { " ", "3",  90,  35 },
    { " ", "4", 110,  35 },
    { " ", "5", 130,  35 },
    { " ", "6", 150,  35 },
    { " ", "7", 170,  35 },
    { " ", "8", 190,  35 },
    { " ", "9", 213,  35 },
    { " ", "0", 232,  35 },
};

//!< キーボードの修飾キー配列の定義です。
const Key Modifiers[] =
{
    { "Ctrl",  "",   5, 125 },
    { "Shift", "",   5, 102 },
    { "Alt",   "",  70, 125 },
    { " ",     "",   0,   0 },
    { "Ctrl",  "", 285, 125 },
    { "Shift", "", 277, 102 },
};

//!< キーボードの編集キー配列の定義です。
const Key EditingKeys[] =
{
    { "Enter", "", 270,  80, nn::hid::KeyboardKey::NumPadEnter::Index, false },
    { "Esc",   "",   5,  35, 0,                                        false },
    { "BS",    "", 290,  35, 0,                                        false },
    { "Tab",   "",   5,  57, 0,                                        true  },
    { "Space", "", 130, 125, 0,                                        false },
};

//!< キーボードの拡張編集キー配列の定義です。
const Key ExtraEditingKeys[] =
{
    { "Insert", "", 340,  35, nn::hid::KeyboardKey::NumPad0::Index   },
    { "Home",   "", 400,  35, nn::hid::KeyboardKey::NumPad7::Index   },
    { "PgUp",   "", 460,  35, nn::hid::KeyboardKey::NumPad9::Index   },
    { "Delete", "", 340,  57, nn::hid::KeyboardKey::NumPadDot::Index },
    { "End",    "", 400,  57, nn::hid::KeyboardKey::NumPad1::Index   },
    { "PgDn",   "", 460,  57, nn::hid::KeyboardKey::NumPad3::Index   },
    { "--",     "", 488, 105, nn::hid::KeyboardKey::NumPad6::Index   },
    { "--",     "", 453, 105, nn::hid::KeyboardKey::NumPad4::Index   },
    { "|",      "", 473, 125, nn::hid::KeyboardKey::NumPad2::Index   },
    { "|",      "", 473,  85, nn::hid::KeyboardKey::NumPad8::Index   },
};

//!< キーボードの状態を描画します。
void WriteKeyboardState(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const nns::hid::KeyboardAndMouse* const pKeyboard,
    float offsetX, float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    if (pKeyboard == 0)
    {
        return;
    }

    const nn::util::Unorm8x4& textColor =
        pKeyboard->IsKeyboardConnected() ? Color::White : Color::Gray;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(1, 1);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Keyboard");

    const bool isToggled =
        pKeyboard->GetKeysToggle().Test<nn::hid::KeyboardKey::Tab>();

    const bool isShifted = pKeyboard->IsCapsLockOn()
        ? !pKeyboard->IsShiftKeyPressed()
        :  pKeyboard->IsShiftKeyPressed();

    const Key* layout = isToggled ? KeyboardLayoutAzerty
                                  : KeyboardLayoutQwerty;
    int head = nn::hid::KeyboardKey::A::Index;
    int tail = nn::hid::KeyboardKey::D0::Index;
    for (int i = head; i <= tail; ++i)
    {
        const Key& key = layout[i - head];
        pTextWriter->SetCursor(offsetX + key.x, offsetY + key.y);
        pTextWriter->SetTextColor(pKeyboard->GetKeys().Test(i) ? Color::Orange
                                                               : textColor);
        pTextWriter->Print(isShifted ? key.upper : key.value);
    }

    if (isToggled)
    {
        pTextWriter->SetCursor(offsetX + 240, offsetY + 80);
        pTextWriter->SetTextColor(
            pKeyboard->GetKeys().Test<nn::hid::KeyboardKey::Semicolon>()
                ? Color::Orange : textColor);
        pTextWriter->Print(isShifted ? "M" : "m");
    }

    head = nn::hid::KeyboardKey::Return::Index;
    tail = nn::hid::KeyboardKey::Space::Index;
    for (int i = head; i <= tail; ++i)
    {
        const Key& key = EditingKeys[i - head];
        pTextWriter->SetCursor(offsetX + key.x, offsetY + key.y);
        pTextWriter->SetTextColor(
            key.isToggleKey
                ? (pKeyboard->GetKeysToggle().Test(i)
                      ? Color::Green : textColor)
                : (((pKeyboard->GetKeys().Test(i)) ||
                    (key.numPadKey != 0 &&
                     pKeyboard->GetKeys().Test(key.numPadKey)))
                      ? Color::Orange : textColor));
        pTextWriter->Print(key.value);
    }

    const bool isNumLocked = pKeyboard->IsNumLockOn() &&
                            !pKeyboard->IsShiftKeyPressed();

    head = nn::hid::KeyboardKey::Insert::Index;
    tail = nn::hid::KeyboardKey::UpArrow::Index;
    for (int i = head; i <= tail; ++i)
    {
        const Key& key = ExtraEditingKeys[i - head];
        pTextWriter->SetCursor(offsetX + key.x, offsetY + key.y);
        pTextWriter->SetTextColor(
            ((pKeyboard->GetKeys().Test(i)) ||
             (!isNumLocked &&
              key.numPadKey != 0 && pKeyboard->GetKeys().Test(key.numPadKey)))
                ? Color::Orange : textColor);
        pTextWriter->Print(key.value);
    }

    head = nn::hid::KeyboardKey::LeftControl::Index;
    tail = nn::hid::KeyboardKey::RightShift::Index;
    for (int i = head; i <= tail; ++i)
    {
        const Key& key = Modifiers[i - head];
        pTextWriter->SetCursor(offsetX + key.x, offsetY + key.y);
        pTextWriter->SetTextColor(pKeyboard->GetKeys().Test(i) ? Color::Orange
                                                               : textColor);
        pTextWriter->Print(key.value);
    }

    pTextWriter->SetCursor(offsetX + 218, offsetY + 125);
    pTextWriter->SetTextColor(
        pKeyboard->GetKeys().Test<nn::hid::KeyboardKey::RightAlt>()
            ? Color::Orange : textColor);
    pTextWriter->Print(isToggled ? "AltGr" : "Alt");
}

//!< キーイベントの履歴を描画します。
void WriteKeyboardEventLog(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    std::list<nns::hid::KeyEvent>* pLog,
    const nns::hid::KeyboardAndMouse* const pKeyboard,
    float offsetX, float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pLog);

    if (pKeyboard == 0)
    {
        return;
    }

    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetScale(1, 1);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Event Log");

    for (size_t i = 0; i < pKeyboard->GetKeyEvents().size(); ++i)
    {
        const nns::hid::KeyEvent& e = pKeyboard->GetKeyEvents()[i];

        const int usageId = e.GetUsageId();

        if ((usageId >= nn::hid::KeyboardKey::A::Index &&
             usageId <= nn::hid::KeyboardKey::Space::Index) ||
            (usageId == nn::hid::KeyboardKey::Semicolon::Index) ||
            (usageId >= nn::hid::KeyboardKey::Insert::Index &&
             usageId <= nn::hid::KeyboardKey::UpArrow::Index) ||
            (usageId >= nn::hid::KeyboardKey::NumPadEnter::Index &&
             usageId <= nn::hid::KeyboardKey::NumPadDot::Index) ||
            (usageId >= nn::hid::KeyboardKey::LeftControl::Index &&
             usageId <= nn::hid::KeyboardKey::LeftAlt::Index) ||
            (usageId >= nn::hid::KeyboardKey::RightControl::Index &&
             usageId <= nn::hid::KeyboardKey::RightAlt::Index))
        {
            pLog->push_back(e);
        }
    }

    for (size_t i = pLog->size(); i > 30; --i)
    {
        pLog->pop_front();
    }

    size_t count = 0;

    for (std::list<nns::hid::KeyEvent>::reverse_iterator
             iter = pLog->rbegin(); iter != pLog->rend(); ++iter)
    {
        const float y = offsetY + 35 + 22 * count;

        pTextWriter->SetCursor(offsetX + 5, y);

        if (iter->IsKeyDown())
        {
            pTextWriter->SetTextColor(Color::Orange);
            pTextWriter->Print("Down");
        }
        else if (iter->IsKeyUp())
        {
            pTextWriter->SetTextColor(Color::Green);
            pTextWriter->Print("Up");
        }
        else if (iter->IsKeyRepeat())
        {
            pTextWriter->SetTextColor(Color::Gray);
            pTextWriter->Print("Repeat");
        }

        pTextWriter->SetCursor(offsetX + 105, y);
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->Print("UsageId: 0x%02X", iter->GetUsageId());

        pTextWriter->SetCursor(offsetX + 265, y);
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->Print(
            "IsPrintingKey: %s", iter->IsPrintingKey() ? "true" : "false");

        ++count;
    }
}

//!< マウスのカーソル座標を描画します。
void WriteMousePosition(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const nns::hid::KeyboardAndMouse* const pMouse,
    const float offsetX, const float offsetY,
    const nn::util::Unorm8x4& textColor) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pMouse);

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Position:");

    pTextWriter->SetCursor(offsetX + 80, offsetY);
    pTextWriter->SetFixedWidthEnabled(true);
    pTextWriter->SetFixedWidth(12);

    if (pMouse->IsPointerOn())
    {
        pTextWriter->Print(
            "(%4.0f,%4.0f)", pMouse->GetPointer().x, pMouse->GetPointer().y);
    }
    else
    {
        pTextWriter->Print("( NaN, NaN)");
    }

    pTextWriter->SetFixedWidthEnabled(false);
}

//!< マウスのボタンの押下状態を描画します。
void WriteMouseButtons(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const nns::hid::KeyboardAndMouse* const pMouse,
    float offsetX, float offsetY,
    const nn::util::Unorm8x4& textColor) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pMouse);

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Buttons:");

    pTextWriter->SetTextColor(
        pMouse->GetMouseButtons().Test<nn::hid::MouseButton::Forward>()
            ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 85, offsetY);
    pTextWriter->Print("F");

    pTextWriter->SetTextColor(
        pMouse->GetMouseButtons().Test<nn::hid::MouseButton::Back>()
            ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 85, offsetY + 22);
    pTextWriter->Print("B");

    pTextWriter->SetTextColor(
        pMouse->GetMouseButtons().Test<nn::hid::MouseButton::Left>()
            ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 105, offsetY);
    pTextWriter->Print("L");

    pTextWriter->SetTextColor(
        pMouse->GetMouseButtons().Test<nn::hid::MouseButton::Middle>()
            ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 125, offsetY);
    pTextWriter->Print("M");

    pTextWriter->SetTextColor(
        pMouse->GetMouseButtons().Test<nn::hid::MouseButton::Right>()
            ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 150, offsetY);
    pTextWriter->Print("R");
}

//!< マウスのホイールを描画します。
void WriteMouseWheel(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    int wheel,
    float offsetX, float offsetY,
    const nn::util::Unorm8x4& textColor) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Wheel:");

    pTextWriter->SetCursor(offsetX + 80, offsetY);
    pTextWriter->SetFixedWidthEnabled(true);
    pTextWriter->SetFixedWidth(12);
    pTextWriter->Print("(%4d)", wheel);
    pTextWriter->SetFixedWidthEnabled(false);
}

//!< マウスのカーソルを描画します。
void WriteMouseCursor(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const nns::hid::KeyboardAndMouse* pMouse,
    const nn::util::Unorm8x4& textColor) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pMouse);

    if (!pMouse->IsPointerOn())
    {
        return;
    }

    const float x = pMouse->GetPointer().x;
    const float y = pMouse->GetPointer().y;

    const float barOffsetX   = pTextWriter->CalculateStringWidth("|") / 2;
    const float barOffsetY   = pTextWriter->CalculateStringHeight("|") / 2;
    const float minusOffsetX = pTextWriter->CalculateStringWidth("-") / 2;
    const float minusOffsetY = pTextWriter->CalculateStringHeight("-") / 2;
    const float plusOffsetX  = pTextWriter->CalculateStringWidth("+") / 2;
    const float plusOffsetY  = pTextWriter->CalculateStringHeight("+") / 2;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetCursor(x - barOffsetX - 15, y - barOffsetY);
    pTextWriter->Print("|");
    pTextWriter->SetCursor(x - barOffsetX + 15, y - barOffsetY);
    pTextWriter->Print("|");
    pTextWriter->SetCursor(x - minusOffsetX, y - minusOffsetY - 15);
    pTextWriter->Print("-");
    pTextWriter->SetCursor(x - minusOffsetX, y - minusOffsetY + 15);
    pTextWriter->Print("-");
    pTextWriter->SetCursor(x - plusOffsetX - 15, y - plusOffsetY - 15);
    pTextWriter->Print("+");
    pTextWriter->SetCursor(x - plusOffsetX - 15, y - plusOffsetY + 15);
    pTextWriter->Print("+");
    pTextWriter->SetCursor(x - plusOffsetX + 15, y - plusOffsetY - 15);
    pTextWriter->Print("+");
    pTextWriter->SetCursor(x - plusOffsetX + 15, y - plusOffsetY + 15);
    pTextWriter->Print("+");
}

//!< マウスの状態を描画します。
void WriteMouseState(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    bool* pIsDoubleClicked, int* pWheel,
    const nns::hid::KeyboardAndMouse* const pMouse,
    float offsetX, float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    NN_ASSERT_NOT_NULL(pIsDoubleClicked);
    NN_ASSERT_NOT_NULL(pWheel);

    if (pMouse == 0)
    {
        return;
    }

    const nn::util::Unorm8x4& textColor =
        pMouse->IsMouseConnected() ? Color::White : Color::Gray;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(1, 1);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Mouse");

    WriteMousePosition(
        pTextWriter, pMouse, offsetX + 5, offsetY + 35, textColor);

    WriteMouseButtons(
        pTextWriter, pMouse, offsetX + 5, offsetY + 79, textColor);

    *pWheel += pMouse->GetMouseWheel();

    *pWheel = std::min(std::max(-600, *pWheel), 600);

    WriteMouseWheel(
        pTextWriter,
        *pWheel,
        offsetX + 5,
        offsetY + 123 + 11 * (5 - *pWheel / 120),
        textColor);

    if (pMouse->GetMouseButtonsDoubleClick().Test<nn::hid::MouseButton::Left>())
    {
        // ダブルクリックでカーソルの色を変更

        *pIsDoubleClicked = !(*pIsDoubleClicked);
    }

    WriteMouseCursor(
        pTextWriter, pMouse, *pIsDoubleClicked ? Color::Orange : Color::Green);
}

//!< コントローラを設定します。
void ConfigureController(nns::hid::KeyboardAndMouse* pController) NN_NOEXCEPT
{
    if (pController == 0)
    {
        return;
    }

    // 修飾キーのキーリピートを無効化します。
    nn::hid::KeyboardKeySet modifiers = {};
    modifiers.Set<nn::hid::KeyboardKey::LeftControl>();
    modifiers.Set<nn::hid::KeyboardKey::LeftShift>();
    modifiers.Set<nn::hid::KeyboardKey::LeftAlt>();
    modifiers.Set<nn::hid::KeyboardKey::RightControl>();
    modifiers.Set<nn::hid::KeyboardKey::RightShift>();
    modifiers.Set<nn::hid::KeyboardKey::RightAlt>();
    pController->SetKeysRepeatOption(modifiers, 0, 0);

    // マウスのカーソル座標の有効範囲を設定します。
    nn::util::Float2 boundaryMin = {};
    nn::util::Float2 boundaryMax = {};
    boundaryMin.x = 0;
    boundaryMin.y = 0;
    boundaryMax.x = static_cast<float>(FrameBufferWidth);
    boundaryMax.y = static_cast<float>(FrameBufferHeight);
    pController->SetPointerBoundary(boundaryMin, boundaryMax);

#if defined(NN_BUILD_TARGET_PLATFORM_WIN)
    // Windows 環境ではマウス API の返す座標値をそのまま使用するよう設定します。
    pController->SetMouseCursorAbsoluteCoordinateUsed(true);
#endif
}

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;

void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}

#endif

} // namespace

extern "C" void nnMain()
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif

    nns::hid::ControllerManager controllerManager;
    nns::hid::util::SetControllerManagerWithDefault(&controllerManager);

    nns::hid::KeyboardAndMouse* const pController =
        reinterpret_cast<nns::hid::KeyboardAndMouse*>(
            controllerManager.GetController(
                nns::hid::ControllerId_KeyboardAndMouse, 0));

    ConfigureController(pController);

    ApplicationHeap applicationHeap;
    applicationHeap.Initialize(ApplicationHeapSize);

    GraphicsSystem* pGraphicsSystem = new GraphicsSystem();
    pGraphicsSystem->Initialize(
        &applicationHeap, FrameBufferWidth, FrameBufferHeight);

    EnableWindowMessage(pGraphicsSystem->GetNativeWindowHandle());

    FontSystem* pFontSystem = new FontSystem();
    pFontSystem->Initialize(&applicationHeap, pGraphicsSystem);

    nn::gfx::util::DebugFontTextWriter& textWriter =
        pFontSystem->GetDebugFontTextWriter();

    std::list<nns::hid::KeyEvent> log;

    bool isDoubleClicked = false;

    int wheel = 0;

    while (NN_STATIC_CONDITION(true))
    {
        bool quits = false;

        switch (GetWindowMessage(pGraphicsSystem->GetNativeWindowHandle()))
        {
        case WindowMessage_Close:
            quits = true;
            break;

        default:
            break;
        }

        if (quits)
        {
            break;
        }

        controllerManager.Update();

        WriteKeyboardState(&textWriter, pController, 25, 15);
        WriteKeyboardEventLog(&textWriter, &log, pController, 645, 15);

        WriteMouseState(
            &textWriter, &isDoubleClicked, &wheel, pController, 25, 375);

        pGraphicsSystem->BeginDraw();
        pFontSystem->Draw();
        pGraphicsSystem->EndDraw();

        pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));
    }

    pFontSystem->Finalize();
    delete pFontSystem;

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    applicationHeap.Finalize();
}
