﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{HidGesture_Main.cpp,PageSampleHidGesture}
 *
 * @brief
 *  nn::hid ライブラリのジェスチャ API の使い方を示すサンプルプログラム
 */

/**
 * @page PageSampleHidGesture ジェスチャの取得
 * @tableofcontents
 *
 * @brief
 * nn::hid ライブラリを用いてジェスチャを取得するサンプルプログラムの解説です。
 *
 * @section PageSampleHidGesture_SectionBrief 概要
 * ジェスチャの状態を表示します。
 *
 * @section PageSampleHidGesture_SectionFileStructure ファイル構成
 * 本サンプルプログラムは
 * @link ../../../Samples/Sources/Applications/HidGesture
 * Samples/Sources/Applications/HidGesture @endlink 以下にあります。
 *
 * @section PageSampleHidGesture_SectionNecessaryEnvironment 必要な環境
 * Windows 環境においては、タッチまたはマウス操作が入力として扱われます。
 *
 * @section PageSampleHidGesture_SectionHowToOperate 操作方法
 * サンプルプログラムを実行するとウィンドウが立ち上がります。
 * ウインドウ上には、左半分に矩形状に数字が、右半分には操作ログが表示され、
 * 矩形にはピンチや回転といった連続的な操作の結果が、
 * 操作ログにはタップやスワイプと言った単発的な操作の結果が反映されます。
 * 操作ログからはコンテキスト番号を確認することもできます。
 * また、タッチ位置にはカーソルが描画されます。
 *
 * @section PageSampleHidGesture_SectionPrecaution 注意事項
 * コンソールウィンドウには何も表示されません。
 *
 * @section PageSampleHidGesture_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleHidGesture_SectionDetail 解説
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - ジェスチャ API を Initialize する。
 * - 利用可能なジェスチャを全て取得する。
 * - eventNumber から新規に発生したジェスチャを選り分ける。
 * - 新規に発生したジェスチャを画面表示に反映させる。
 */

#include <cstdlib>
#include <list>
#include <vector>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/hid/hid_Gesture.h>
#include <nn/TargetConfigs/build_Platform.h>
#include <nn/util/util_Constant.h>
#include <nn/util/util_MathTypes.h>
#include <nn/util/util_Matrix.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include "HidGesture_ApplicationHeap.h"
#include "HidGesture_Color.h"
#include "HidGesture_FontSystem.h"
#include "HidGesture_GraphicsSystem.h"
#include "HidGesture_WindowMessage.h"

namespace {

const size_t ApplicationHeapSize = 128 * 1024 * 1024;

const int FrameBufferWidth = 1280;

const int FrameBufferHeight = 720;

const int FrameRate = 60;

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;

void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}

#endif

class GestureManager
{
    NN_DISALLOW_COPY(GestureManager);
    NN_DISALLOW_MOVE(GestureManager);

private:
    std::list<nn::hid::GestureState> m_Events;
    std::vector<nn::hid::GesturePoint> m_Points;
    float m_Scale;
    float m_Angle;

    int64_t m_EventNumber;
    nn::hid::GestureState* m_States;

public:
    GestureManager() NN_NOEXCEPT
        : m_Scale(1)
        , m_Angle(0)
        , m_EventNumber(-1)
    {
        m_States = new nn::hid::GestureState[nn::hid::GestureStateCountMax]();
    }

    ~GestureManager() NN_NOEXCEPT
    {
        delete[] m_States;
    }

    static void Initialize() NN_NOEXCEPT
    {
        nn::hid::InitializeGesture();
    }

    const std::list<nn::hid::GestureState>& GetEvents() const NN_NOEXCEPT
    {
        return m_Events;
    }

    const std::vector<nn::hid::GesturePoint>& GetPoints() const NN_NOEXCEPT
    {
        return m_Points;
    }

    float GetScale() const NN_NOEXCEPT
    {
        return m_Scale;
    }

    float GetAngle() const NN_NOEXCEPT
    {
        return m_Angle;
    }

    void Update() NN_NOEXCEPT
    {
        const int count = nn::hid::GetGestureStates(
            m_States, nn::hid::GestureStateCountMax);

        for (int i = count - 1; i >= 0; --i)
        {
            if (m_EventNumber < m_States[i].eventNumber)
            {
                this->Precess(m_States[i]);
            }
        }

        m_EventNumber = m_States[0].eventNumber;
    }

private:
    void Precess(const nn::hid::GestureState& state) NN_NOEXCEPT
    {
        m_Points.clear();

        for (int i = 0; i < state.pointCount; ++i)
        {
            m_Points.push_back(state.points[i]);
        }

        switch (state.GetGestureType())
        {
        case nn::hid::GestureType_Idle:
        case nn::hid::GestureType_Complete:
        case nn::hid::GestureType_Cancel:
        case nn::hid::GestureType_Touch:
        case nn::hid::GestureType_Press:
        case nn::hid::GestureType_Tap:
        case nn::hid::GestureType_Swipe:
            m_Events.push_back(state);
            break;

        case nn::hid::GestureType_Pinch:
            m_Scale *= state.scale;
            m_Scale = m_Scale > 0.5 ? m_Scale : 0.5f;
            m_Scale = m_Scale < 2.0 ? m_Scale : 2.0f;
            break;

        case nn::hid::GestureType_Rotate:
            m_Angle += state.rotationAngle;
            m_Angle = m_Angle > -360 ? m_Angle : m_Angle + 360;
            m_Angle = m_Angle <  360 ? m_Angle : m_Angle - 360;
            break;

        default:
            break;
        }

        for (size_t i = m_Events.size(); i > 30; --i)
        {
            m_Events.pop_front();
        }
    }
};

void WriteCursor(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    int32_t x, int32_t y) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const float barOffsetX   = pTextWriter->CalculateStringWidth("|") / 2;
    const float barOffsetY   = pTextWriter->CalculateStringHeight("|") / 2;
    const float minusOffsetX = pTextWriter->CalculateStringWidth("-") / 2;
    const float minusOffsetY = pTextWriter->CalculateStringHeight("-") / 2;
    const float plusOffsetX  = pTextWriter->CalculateStringWidth("+") / 2;
    const float plusOffsetY  = pTextWriter->CalculateStringHeight("+") / 2;

    pTextWriter->SetCursor(x - barOffsetX - 45, y - barOffsetY);
    pTextWriter->Print("|");
    pTextWriter->SetCursor(x - barOffsetX + 45, y - barOffsetY);
    pTextWriter->Print("|");
    pTextWriter->SetCursor(x - minusOffsetX, y - minusOffsetY - 45);
    pTextWriter->Print("-");
    pTextWriter->SetCursor(x - minusOffsetX, y - minusOffsetY + 45);
    pTextWriter->Print("-");
    pTextWriter->SetCursor(x - plusOffsetX - 45, y - plusOffsetY - 45);
    pTextWriter->Print("+");
    pTextWriter->SetCursor(x - plusOffsetX - 45, y - plusOffsetY + 45);
    pTextWriter->Print("+");
    pTextWriter->SetCursor(x - plusOffsetX + 45, y - plusOffsetY - 45);
    pTextWriter->Print("+");
    pTextWriter->SetCursor(x - plusOffsetX + 45, y - plusOffsetY + 45);
    pTextWriter->Print("+");
}

void WritePoints(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const std::vector<nn::hid::GesturePoint>& points) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    pTextWriter->SetTextColor(Color::Green);

    for (size_t i = 0; i < points.size(); ++i)
    {
        WriteCursor(pTextWriter, points[i].x, points[i].y);
    }
}

void WritePicture(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    float scale, float angle, float offsetX, float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    char chars[] = { '\0', '\0' };

    const ::nn::util::Float2 vertexs[] =
    {
        { { { -100, -100 } } }, { { {    0, -100 } } }, { { { 100, -100 } } },
        { { {  100,    0 } } }, { { {  100,  100 } } }, { { {   0,  100 } } },
        { { { -100,  100 } } }, { { { -100,    0 } } }
    };

    nn::util::Matrix4x3f matrix = nn::util::Matrix4x3f::MakeRotation(
        nn::util::Vector3f(0, 0, -angle / 180 * nn::util::FloatPi));

    pTextWriter->SetTextColor(Color::White);

    for (size_t i = 0; i < NN_ARRAY_SIZE(vertexs); ++i)
    {
        chars[0] = '1' + static_cast<char>(i);
        const float charOffsetX = pTextWriter->CalculateStringWidth(chars) / 2;
        const float charOffsetY = pTextWriter->CalculateStringHeight(chars) / 2;
        const float x = matrix.GetAxisX().GetX() * vertexs[i].x +
                        matrix.GetAxisX().GetY() * vertexs[i].y;
        const float y = matrix.GetAxisY().GetX() * vertexs[i].x +
                        matrix.GetAxisY().GetY() * vertexs[i].y;
        pTextWriter->SetCursor(
            scale * x - charOffsetX + offsetX,
            scale * y - charOffsetY + offsetY);
        pTextWriter->Print(chars);
    }
}

const char* GetGestureDirection(nn::hid::GestureDirection direction) NN_NOEXCEPT
{
    switch (direction)
    {
    case nn::hid::GestureDirection_None:
        return "None";
    case nn::hid::GestureDirection_Left:
        return "Left";
    case nn::hid::GestureDirection_Up:
        return "Up";
    case nn::hid::GestureDirection_Right:
        return "Right";
    case nn::hid::GestureDirection_Down:
        return "Down";
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void WriteEventLog(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const std::list<nn::hid::GestureState>& events,
    float offsetX, float offsetY) NN_NOEXCEPT
{
    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Event Log");

    size_t count = 0;

    for (std::list<nn::hid::GestureState>::const_reverse_iterator
             iter = events.rbegin(); iter != events.rend(); ++iter)
    {
        const float y = offsetY + 35 + 22 * count;

        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetCursor(offsetX + 5, y);
        pTextWriter->Print("%08X", iter->contextNumber);

        pTextWriter->SetCursor(offsetX + 165, y);

        switch (iter->GetGestureType())
        {
        case nn::hid::GestureType_Idle:
            pTextWriter->SetTextColor(Color::Gray);
            pTextWriter->Print("Idle");
            break;

        case nn::hid::GestureType_Complete:
            pTextWriter->SetTextColor(Color::Gray);
            pTextWriter->Print("Complete");
            break;

        case nn::hid::GestureType_Cancel:
            pTextWriter->SetTextColor(Color::Gray);
            pTextWriter->Print("Cancel");
            break;

        case nn::hid::GestureType_Touch:
            if (iter->attributes.Test<nn::hid::GestureAttribute::IsNewTouch>())
            {
                pTextWriter->SetTextColor(Color::White);
                pTextWriter->Print("NewTouch");
                break;
            }
            else
            {
                pTextWriter->SetTextColor(Color::Gray);
                pTextWriter->Print("Restart");
                break;
            }

        case nn::hid::GestureType_Press:
            pTextWriter->SetTextColor(Color::Orange);
            pTextWriter->Print("Press");
            break;

        case nn::hid::GestureType_Tap:
            if (iter->attributes.Test<nn::hid::GestureAttribute::IsDoubleTap>())
            {
                pTextWriter->SetTextColor(Color::Orange);
                pTextWriter->Print("DoubleTap");
                break;
            }
            else
            {
                pTextWriter->SetTextColor(Color::White);
                pTextWriter->Print("Tap");
                break;
            }

        case nn::hid::GestureType_Swipe:
            pTextWriter->SetTextColor(Color::White);
            pTextWriter->Print("Swipe");
            pTextWriter->SetCursor(offsetX + 285, y);
            pTextWriter->Print(
                GetGestureDirection(iter->GetGestureDirection()));
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        ++count;
    }
}

} // namespace

extern "C" void nnMain()
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif

    GestureManager::Initialize();
    GestureManager gestureManager;

    ApplicationHeap applicationHeap;
    applicationHeap.Initialize(ApplicationHeapSize);

    GraphicsSystem* pGraphicsSystem = new GraphicsSystem();
    pGraphicsSystem->Initialize(
        &applicationHeap, FrameBufferWidth, FrameBufferHeight);

    EnableWindowMessage(pGraphicsSystem->GetNativeWindowHandle());

    FontSystem* pFontSystem = new FontSystem();
    pFontSystem->Initialize(&applicationHeap, pGraphicsSystem);

    nn::gfx::util::DebugFontTextWriter& textWriter =
        pFontSystem->GetDebugFontTextWriter();

    while (NN_STATIC_CONDITION(true))
    {
        bool quits = false;

        switch (GetWindowMessage(pGraphicsSystem->GetNativeWindowHandle()))
        {
        case WindowMessage_Close:
            quits = true;
            break;

        default:
            break;
        }

        if (quits)
        {
            break;
        }

        gestureManager.Update();

        textWriter.SetTextColor(Color::White);
        textWriter.SetCursor(25, 15);
        textWriter.Print("Gesture");

        WritePoints(&textWriter, gestureManager.GetPoints());

        WritePicture(
            &textWriter,
            gestureManager.GetScale(), gestureManager.GetAngle(),
            320, 360);

        WriteEventLog(&textWriter, gestureManager.GetEvents(), 645, 15);

        pGraphicsSystem->BeginDraw();
        pFontSystem->Draw();
        pGraphicsSystem->EndDraw();

        pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));
    }

    pFontSystem->Finalize();
    delete pFontSystem;

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    applicationHeap.Finalize();
}
