﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#pragma once


#include <stdint.h>
#include <nn/nn_Result.h>


namespace nns
{


namespace detail
{
    // Helper class to get the maximum suported filter order based on the floating point type used
    template<typename T> struct ButterworthMaxOrder;
    template<> struct ButterworthMaxOrder<float>  {static const uint32_t MaxOrder =  5;};
    template<> struct ButterworthMaxOrder<double> {static const uint32_t MaxOrder = 10;};
}


/**
 * @brief Butterworth filter
 */
class Butterworth
{
private:
    typedef double Float; ///< The floating point type to be used when filtering
    typedef void (Butterworth::*ApplyFct)(const int16_t* srce, int16_t* dest, size_t length, size_t stride);

public:
    static const uint32_t MaxOrder = detail::ButterworthMaxOrder<Float>::MaxOrder; ///< Maximum order supported by the filter

public:
    Butterworth();

    nn::Result InitButterworthLowPass (double samplingRate, double cutoff, uint32_t order=4) NN_NOEXCEPT;
    nn::Result InitButterworthHighPass(double samplingRate, double cutoff, uint32_t order=4) NN_NOEXCEPT;

    void Apply(const int16_t* srce, int16_t* dest, size_t length, size_t stride=1) NN_NOEXCEPT;

private:
    nn::Result DoInit(double samplingRate, double cutoff, uint32_t order, bool highPass) NN_NOEXCEPT;

    void ApplyGeneric(const int16_t* srce, int16_t* dest, size_t length, size_t stride) NN_NOEXCEPT;
    void Apply2(const int16_t* srce, int16_t* dest, size_t length, size_t stride) NN_NOEXCEPT;
    void Apply4(const int16_t* srce, int16_t* dest, size_t length, size_t stride) NN_NOEXCEPT;
    void Apply6(const int16_t* srce, int16_t* dest, size_t length, size_t stride) NN_NOEXCEPT;

private:
    Float    m_a[MaxOrder + 1]; ///< Coefficients for the filter's poles
    Float    m_b[MaxOrder + 1]; ///< Coefficients for the filter's zeroes
    Float    m_x[MaxOrder + 1]; ///< Previous inputs values (m_x[0] is the oldest)
    Float    m_y[MaxOrder + 1]; ///< Previous output values (m_y[0] is the oldest)
    Float    m_invGain;         ///< The inverse of the filter's gain
    uint32_t m_order;           ///< Filter's order
    ApplyFct m_applyFunction;   ///< Pointer to the specialized `Apply()` function
};


inline Butterworth::Butterworth():
    m_order(0)
{
}


/**
 * @brief Creates a low-pass Butterworth filter
 *
 * @param [in] samplingRate The sampling rate (in Hz) of the waveform to be filtered
 * @param [in] cutoff       The cutoff frequency (in Hz)
 * @param [in] order        The filter's order (defaults to 4)
 */
inline nn::Result Butterworth::InitButterworthLowPass(double samplingRate, double cutoff, uint32_t order) NN_NOEXCEPT
{
    return DoInit(samplingRate, cutoff, order, false);
}


/**
 * @brief Creates a high-pass Butterworth filter
 *
 * @param [in] samplingRate The sampling rate (in Hz) of the waveform to be filtered
 * @param [in] cutoff       The cutoff frequency (in Hz)
 * @param [in] order        The filter's order (defaults to 4)
 */
inline nn::Result Butterworth::InitButterworthHighPass(double samplingRate, double cutoff, uint32_t order) NN_NOEXCEPT
{
    return DoInit(samplingRate, cutoff, order, true);
}


/**
 * @brief Applies the filter to audio data
 *
 * @param [in]  srce   The buffer containing the original waveform
 * @param [out] dest   The buffer to receive the filtered waveform
 * @param [in]  stride The distance (in number of `int16_t`) between each sample in `srce`
 */
inline void Butterworth::Apply(const int16_t* srce, int16_t* dest, size_t length, size_t stride) NN_NOEXCEPT
{
    (this->*m_applyFunction)(srce, dest, length, stride);
}


} // namespace nns
