﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define ENABLE_SKELETON_SSBO
#include <common.glsl>
#include <skeleton_block.glsl>
#include <shape_block.glsl>
#include <view_block.glsl>
#include <env_block.glsl>

#define c_Local             ( 0 )
#define c_World             ( c_Local + 1 )

#define c_None              ( 0 )
#define c_Blend             ( c_None + 1 )
#define c_Mul               ( c_Blend + 1 )
#define c_Add               ( c_Mul + 1 )

#define USE_VERTEX_COLOR    ( 0 )       // @@ id="vertex_color"
#define USE_ALBEDO_TEX      ( 1 )       // @@ id="albedo_tex"
#define MULTI_TEX           ( 0 )       // @@ id="multi_tex" choice="0: 無し, 1: 補間, 2: 乗算, 3: 加算"
#define USE_NORMAL_MAP      ( 0 )       // @@ id="normal_map"
#define USE_SPEC_MASK       ( 0 )       // @@ id="spec_mask"
#define USE_RIM_LIGHT       ( 0 )       // @@ id="rim_light"
#define SKINNING            ( 0 )       // @@ id="skinning" choice="[-1, 8]" type="dynamic"
#define COORDINATE          ( c_Local ) // @@ id="coord" choice="c_Local, c_World" type="dynamic"
#define NUM_LIGHT           ( 2 )

layout(std140) uniform Option // @@ id="opt" type="option"
{
    int useVtxColor;    // @@ id="vertex_color"
    int albedoTex;      // @@ id="albedo_tex"
    int multiTex;       // @@ id="multi_tex"
    int useNormMap;     // @@ id="normal_map"
    int useSpecMask;    // @@ id="spec_mask"
    int useRimLight;    // @@ id="rim_light"
    int combinerColor0; // @@ id="combiner_color0"
};

//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_VERTEX_SHADER

layout(location = 0)  in vec4   i_Position;     // @@ id="_p0"   hint="position0"
layout(location = 1)  in vec3   i_Normal;       // @@ id="_n0"   hint="normal0"
layout(location = 2)  in vec4   i_Tangent;      // @@ id="_t0"   hint="tangent0"
layout(location = 3)  in ivec4  i_Index0;       // @@ id="_i0"   hint="blendindex0"
layout(location = 4)  in ivec4  i_Index1;       // @@ id="_i1"   hint="blendindex1"
layout(location = 5)  in vec4   i_Weight0;      // @@ id="_w0"   hint="blendweight0"
layout(location = 6)  in vec4   i_Weight1;      // @@ id="_w1"   hint="blendweight1"
layout(location = 7)  in vec4   i_Color;        // @@ id="_c0"   hint="color0"
layout(location = 8)  in vec2   i_UV0;          // @@ id="_u0"   hint="uv0"
layout(location = 9)  in vec2   i_UV1;          // @@ id="_u1"   hint="uv0"
layout(location = 10) in vec2   i_UV2;          // @@ id="_u2"   hint="uv0"
layout(location = 11) in vec2   i_UV3;          // @@ id="_u3"   hint="uv0"

/* @@
    attrib_id="_p0"   label="位置座標"                    editable="false"
    attrib_id="_n0"   label="法線"                        editable="false"
    attrib_id="_t0"   label="接線"                        editable="false"
    attrib_id="_i0"   label="ブレンドインデックス1-4"     editable="false"
    attrib_id="_i1"   label="ブレンドインデックス5-8"     editable="false"
    attrib_id="_w0"   label="ブレンドウェイト1-4"         editable="false"
    attrib_id="_w1"   label="ブレンドウェイト5-8"         editable="false"
    attrib_id="_c0"   label="ディフューズカラー"
    attrib_id="_u0"   label="アルベドテクスチャ0用UV"
    attrib_id="_u1"   label="アルベドテクスチャ1用UV"
    attrib_id="_u2"   label="法線マップ用UV"
    attrib_id="_u3"   label="スペキュラマスク用UV"
    interleave="_p0 _n0 _t0 _i0 _i1 _w0 _w1 _c0 _u0 _u1 _u2 _u3"
*/

#endif // NN_G3D_VERTEX_SHADER

#ifndef NN_G3D_GEOMETRY_SHADER

NN_G3D_RASTER
{
    vec3 pos_w;
    vec3 pos_v;
    vec3 nrm_w;
    vec3 nrm_v;
    vec4 tan_w;
    vec4 tan_v;
    vec4 color;
    vec4 uv0_uv1;
    vec4 uv2_uv3;
    vec3 projXYW;
    vec4 debug;
} nn_g3d_raster;

#endif // !NN_G3D_GEOMETRY_SHADER

#ifdef NN_G3D_FRAGMENT_SHADER

layout(location = 0)    out vec4 o_Color;

#endif // NN_G3D_FRAGMENT_SHADER

//--------------------------------------------------------------------------------------------------

uniform sampler2D albedoTex0;   // @@ id="_a0" hint="albedo0"
uniform sampler2D albedoTex1;   // @@ id="_a1" hint="albedo1"
uniform sampler2D normalTex;    // @@ id="_n0" hint="normal0"
uniform sampler2D specularTex;  // @@ id="_s0" hint="specular0"

layout(std140) uniform Material // @@ id="mat" type="material"
{
    vec4 texsrt0[2];        // @@ id="texsrt0" type="texsrt" hint="albedo0"
    vec4 texsrt1[2];        // @@ id="texsrt1" type="texsrt" hint="albedo1"
    vec4 texsrt2[2];        // @@ id="texsrt2" type="texsrt" hint="normal0"
    vec4 texsrt3[2];        // @@ id="texsrt3" type="texsrt" hint="specular0"
    vec3 diffColor;         // @@ id="diffuse" default="1 1 1" item="color" hint="diffuse"
    float weightTex;        // @@ id="weight_tex" default="1.0" min="-1" max="1"
    float specInt;          // @@ id="spec_int" min="0" max="1" default="0.2"
    float specPower;        // @@ id="spec_power" min="0" default="50"
    float rimInt;           // @@ id="rim_int" min="0" max="1" default="0.5"
    float rimPower;         // @@ id="rim_power" min="0" default="2.5"
};

// @@ group="diffuse_group" label="ディフューズ" order="0"
/* @@ diffuse_group:
    uniform_id="diffuse"            order="1"   label="ディフューズカラー"
    option_id="vertex_color"        order="2"   label="頂点カラー"
    option_id="albedo_tex"          order="3"   label="アルベドテクスチャ"

    sampler_id="_a0"                order="4"   label="アルベドテクスチャ0"
    uniform_id="texsrt0"            order="5"   label="アルベドテクスチャ0用SRT"
*/

// @@ group="multi_tex_group" label="マルチテクスチャ" order="1" condition="$multi_tex != 0"
/* @@ multi_tex_group:
    option_id="multi_tex"           order="1"   label="マルチテクスチャブレンドモード"
    sampler_id="_a1"                order="2"   label="アルベドテクスチャ1"
    uniform_id="weight_tex"         order="3"   label="補間・加算時ウェイト"
    uniform_id="texsrt1"            order="4"   label="アルベドテクスチャ1用SRT"
*/

// @@ group="specular_group" label="スペキュラ" order="3"
/* @@ specular_group:
    uniform_id="spec_int"           order="1"   label="スペキュラ強度"
    uniform_id="spec_power"         order="2"   label="スペキュラ絞り"
    option_id="spec_mask"           order="3"   label="スペキュラマスク"
    sampler_id="_s0"                order="4"   label="スペキュラマスク"
    uniform_id="texsrt3"            order="5"   label="スペキュラマスク用SRT"
*/

// @@ group="rim_group" label="リムライト" order="4" condition="$rim_light"
/* @@ rim_group:
    option_id="rim_light"           order="1"   label="リムライトの有効化"
    uniform_id="rim_int"            order="2"   label="リムライト強度"
    uniform_id="rim_power"          order="3"   label="リムライト絞り"
*/

// @@ group="normal_map_group" label="法線" order="5" condition="$normal_map"
/* @@ normal_map_group:
    option_id="normal_map"          order="1"   label="法線マップの有効化"
    sampler_id="_n0"                order="2"   label="法線マップ"
    uniform_id="texsrt2"            order="3"   label="法線マップ用SRT"
*/

//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_VERTEX_SHADER

void main()
{
#if COORDINATE == c_Local
    vec4 pos_w = vec4(0, 0, 0, 1);
    vec3 nrm_w = vec3(0);
    vec3 tan_w = vec3(0);

    int skinning = SKINNING > 0 ? SKINNING - 1 : vtxSkinCount;
    if (skinning == 0)
    {
        // rigid body
        pos_w.xyz = NN_G3D_TRANSFORM_POS(shapeMtx, i_Position);
        nrm_w.xyz = NN_G3D_TRANSFORM_VEC(shapeMtx, i_Normal);
        tan_w.xyz = NN_G3D_TRANSFORM_VEC(shapeMtx, i_Tangent);
    }
    else if (skinning == 1)
    {
        // rigid skinning
        int offset = i_Index0.x * 3;
        pos_w.xyz = NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
        nrm_w.xyz = NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
        tan_w.xyz = NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
    }
    else if (skinning <= 4)
    {
        // smooth skinning [1, 4]
        for (int i = 0; i < skinning; ++i)
        {
            int offset = i_Index0[i] * 3;
            float weight = i_Weight0[i];
            pos_w.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            nrm_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            tan_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
        }
    }
    else if (skinning <= 8)
    {
        // smooth skinning [5, 8]
        for (int i = 0; i < 4; ++i)
        {
            int offset = i_Index0[i] * 3;
            float weight = i_Weight0[i];
            pos_w.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            nrm_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            tan_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
        }
        for (int i = 4; i < skinning; ++i)
        {
            int offset = i_Index1[i - 4] * 3;
            float weight = i_Weight1[i - 4];
            pos_w.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            nrm_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            tan_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
        }
    }
#elif COORDINATE == c_World
    vec4 pos_w = i_Position;
    vec3 nrm_w = i_Normal;
    vec3 tan_w = i_Tangent.xyz;
#endif

    vec4 pos_v = vec4(NN_G3D_TRANSFORM_POS(cameraMtx, pos_w), 1);
    vec3 nrm_v = NN_G3D_TRANSFORM_VEC(cameraMtx, nrm_w);
    vec3 tan_v = NN_G3D_TRANSFORM_VEC(cameraMtx, tan_w);

    gl_Position = NN_G3D_PROJECT(projMtx, pos_v);
    OUT.pos_w = pos_w.xyz;
    OUT.pos_v = pos_v.xyz;
    OUT.nrm_w = nrm_w;
    OUT.nrm_v = nrm_v;
    OUT.tan_w = vec4(tan_w, i_Tangent.w);
    OUT.tan_v = vec4(tan_v, i_Tangent.w);
    OUT.color = i_Color;
    OUT.uv0_uv1.xy = NN_G3D_TRANSFORM_UV(texsrt0, i_UV0);
    OUT.uv0_uv1.zw = NN_G3D_TRANSFORM_UV(texsrt1, i_UV1);
    OUT.uv2_uv3.xy = NN_G3D_TRANSFORM_UV(texsrt2, i_UV2);
    OUT.uv2_uv3.zw = NN_G3D_TRANSFORM_UV(texsrt3, i_UV3);
}

#endif // NN_G3D_VERTEX_SHADER

//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_FRAGMENT_SHADER

#include <NodeDefinitions/combiner_default_node_definitions.glsl>
#include <NodeDefinitions/combiner_node_definitions.glsl>
#include "combiner_option_definitions.glsl"

void main()
{
    vec3 nrm_w = normalize(IN.nrm_w);
    vec3 nrm_v = normalize(IN.nrm_v);

    vec3 normal = nrm_w;
    if (NN_G3D_TO_BOOL(USE_NORMAL_MAP))
    {
        vec3 nrm_l;
        nrm_l.xy = texture(normalTex, IN.uv2_uv3.xy).xy;
        float lengthSq = 1 - nrm_l.x * nrm_l.x - nrm_l.y * nrm_l.y;
        lengthSq = clamp(lengthSq, 0, 1);
        nrm_l.z =  sqrt(lengthSq);
        vec3 tan_w = normalize(IN.tan_w.xyz);
        vec3 binrm_w = cross(nrm_w, tan_w) * IN.tan_w.w;
        normal = nrm_l.x * tan_w + nrm_l.y * binrm_w + nrm_l.z * nrm_w;
    }

    // multi texture
    vec4 albedo = GetCombinerColor0();

    // combiner_color0 とは別のファイルで書いたケース
    vec4 albedo1;
    {
        vec4 combiner_output_color = vec4(0, 0, 0, 1);
        // @@ combiner_process="combiner_color1"
        albedo1 = combiner_output_color;
    }

    switch (MULTI_TEX)
    {
        case c_Blend:
            albedo = albedo * (1 - weightTex) + albedo1 * weightTex;
            break;
        case c_Mul:
            albedo *= albedo1;
            break;
        case c_Add:
            albedo += albedo1 * weightTex;
            break;
    }

    vec3 view_w = cameraMtx[2].xyz;
    vec3 refl_w = 2 * dot(view_w, normal) * normal - view_w;


    // light
    vec3 diffRefl = vec3(0);
    float specRefl = 0;

    // hemisphere
    {
        vec3 up = vec3(0, 1, 0);
        float rate = dot(normal, up) * 0.5 + 0.5;
        diffRefl += rate * skyColor + (1 - rate) * groundColor;
    }

    // directional
    {
        vec3 lightDir = normalize(-dirLightDir);
        diffRefl += dirLightColor * clamp(dot(normal, lightDir), 0, 1);
        specRefl += pow(clamp(dot(lightDir, refl_w), 0, 1), specPower);
    }

    // rim
    float rim = 0;
    if (NN_G3D_TO_BOOL(USE_RIM_LIGHT))
    {
        rim = rimInt * pow(1 - abs(clamp(dot(normal, view_w), 0, 1)), rimPower);
    }

    o_Color = albedo;
    o_Color.rgb *= diffRefl * (NN_G3D_TO_BOOL(USE_VERTEX_COLOR) ? IN.color.rgb * IN.color.a : diffColor);

    float specular = specRefl * specInt;
    if (NN_G3D_TO_BOOL(USE_SPEC_MASK))
    {
        float specMask = texture(specularTex, IN.uv2_uv3.zw).x;
        specular *= specMask;
    }

    o_Color.rgb += vec3(specular + rim);
    o_Color.rgb = clamp(o_Color.rgb, 0, 1);

    //o_Color = IN.debug;
}

#endif // NN_G3D_FRAGMENT_SHADER
