﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "g3ddemo_ModelUtility.h"

#include <nn/g3d/viewer/g3d_ViewerCallbackUtility.h>
#include <nn/os.h>

namespace nn { namespace g3d { namespace demo {
class DemoViewerBase : public nn::g3d::viewer::ViewerCallbacks
{
public:
    //! @brief 外から渡された任意のコールバック関数でビューアを初期化します。
    void Initialize(
        ResourceHolder& resourceHolder,
        nn::g3d::viewer::ViewerCallback callback,
        nn::g3d::viewer::ViewerTextureBindCallback textureBindCallback) NN_NOEXCEPT;

    //! @brief 自分自身をコールバッククラスとしてビューアを初期化します。
    void Initialize(ResourceHolder& resourceHolder) NN_NOEXCEPT;

    //! @brief 終了処理をします。
    void Finalize() NN_NOEXCEPT;

    //! @brief 通信スレッドを開始します。
    void StartPollThread() NN_NOEXCEPT;

    //! @brief 通信スレッドを停止します。
    void StopPollThread() NN_NOEXCEPT;

    //! @brief 通信スレッドを一時停止します。
    void PausePollThread() NN_NOEXCEPT;

    //! @brief 一時停止した通信スレッドを再開します。
    void ResumePollThread() NN_NOEXCEPT;

    //! @brief サーバーをオープンします。
    void Open() NN_NOEXCEPT;

    //! @brief サーバーをクローズします。
    void Close() NN_NOEXCEPT;

protected:
    DemoViewerBase() NN_NOEXCEPT
        : m_pResourceHolder(nullptr)
        , m_IsPollThreadLiving(false)
        , m_IsPollThreadPauseRequested(false)
        , m_IsPollThreadPausing(false)
    {
    }

    virtual void FinalizeInternal() NN_NOEXCEPT
    {
    }

    ResourceHolder* GetResourceHolder() NN_NOEXCEPT
    {
        return m_pResourceHolder;
    }

    const ResourceHolder* GetResourceHolder() const NN_NOEXCEPT
    {
        return m_pResourceHolder;
    }

    void SetModelLayout(const nn::g3d::viewer::ModelLayoutUpdatedArg& arg) NN_NOEXCEPT;

private:
    static void PollThreadEntryFunction(void *pArg) NN_NOEXCEPT;

    // 通信用スレッドを作成します。
    void CreatePollThread() NN_NOEXCEPT;

private:
    ResourceHolder* m_pResourceHolder;

    nn::os::ThreadType m_PollThread;
    volatile bool m_IsPollThreadLiving;
    volatile bool m_IsPollThreadPauseRequested;
    volatile bool m_IsPollThreadPausing;
};

//! @brief 3DEditor で移動がリセットされたかどうかを判定します。
NN_FORCEINLINE
bool IsInitialTranslate(const nn::g3d::viewer::ModelLayoutUpdatedArg& arg) NN_NOEXCEPT
{
    return nn::util::VectorGetX(arg.translate) == 0.0f
        && nn::util::VectorGetY(arg.translate) == 0.0f
        && nn::util::VectorGetZ(arg.translate) == 0.0f;
}

//! @brief pSourceResFile をコピーしたセットアップ済みの ResFile を作成します。
nn::g3d::ResFile* CreateCopiedResFile(nn::g3d::ResFile* pSourceResFile, size_t fileSize, size_t alignment) NN_NOEXCEPT;

//! @brief 3DEditor にメッセージを送ります。
void SendMessageTo3dEditor(const char* message) NN_NOEXCEPT;

//! @brief 3DEditor にエラーメッセージを送ります。
void SendErrorMessageTo3dEditor(const char* message) NN_NOEXCEPT;

}}} // namespace nn::g3d::demo

