﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <curl/curl.h>
#include <queue>
#include <list>

/*!
    @brief      ファイルダウンローダーです。

    @details
                本クラスは curl_multi を使用して HTTP 通信を並列化します。

                @ref AddRequest を使用してリクエストを複数設定した後、 @ref Execute を呼ぶことで通信処理が行われます。
*/
class FileDownloader
{
public:
    /*!
        @brief      ダウンロード完了コールバックです。

        @details
                    ダウンロードの成功と失敗の両方で本コールバックは呼び出されます。
                    通常、成功判定は以下で行います。

                    - curlCode == CURLE_OK && (statusCode / 100) == 2

                    その他の注意事項：

                    - リクエストがキャンセルされた場合、 curlCode は CURLE_ABORTED_BY_CALLBACK になります。
    */
    using CompletionCallback = void (*)(uint64_t requestId, CURLcode curlCode, long statusCode,
        void* pBuffer, size_t downloadedSize, void* pParam);

public:
    /*!
        @brief      コンストラクタです。

        @param[in]  sessionCountMax 最大同時実行数

        @pre
            - sessionCountMax > 0

        @details
                    最大同時実行数は、リクエストが大量にある時に同時実行数に上限を設けるために設定します。@n
                    アプリケーションが同時に使用できる SSL やソケットのオブジェクトの数には制限があるため、同時に実行する他の処理に合わせてパラメータを決定します。
    */
    explicit FileDownloader(int sessionCountMax = 4) NN_NOEXCEPT;

    /*!
        @brief      リクエストを追加します。

        @param[out] pOutRequestId   リクエスト ID
        @param[in]  pUrl            URL
        @param[in]  pBuffer         受信バッファ
        @param[in]  size            受信バッファのサイズ

        @pre
            - pOutRequestId != nullptr
            - pUrl != nullptr
            - pBuffer != nullptr
            - size > 0

        @details
                    リクエストの実行完了時、指定した受信バッファにデータが書き込まれた状態になります。
    */
    void AddRequest(uint64_t* pOutRequestId, const char* pUrl, void* pBuffer, size_t size);

    /*!
        @brief      リクエストを実行します。

        @details
                    @ref AddRequest で追加されたリクエストをすべて実行します。@n
                    本関数は、処理が完了するまでブロックします。
    */
    void Execute();

    /*!
        @brief      リクエストの実行をキャンセルします。

        @details
                    本関数はキャンセル用のイベントをシグナルするだけであり、即座に @ref Execute が完了することは保証しません。
    */
    void Cancel() NN_NOEXCEPT;

    /*!
        @brief      ダウンロード完了コールバックを設定します。

        @param[in]  pCallback   ダウンロード完了コールバック
        @param[in]  pParam      ダウンロード完了コールバックに渡すパラメータ

        @details
                    ダウンロード完了コールバックは、ファイルをダウンロードする度に呼び出されます。
    */
    void SetCompletionCallback(CompletionCallback pCallback, void* pParam) NN_NOEXCEPT;

private:
    //
    struct RequestItem
    {
        uint64_t requestId;
        const char* pUrl;
        void* pBuffer;
        size_t size;
    };

    //
    struct SessionItem
    {
        uint64_t requestId;
        CURL* pHandle;
        void* pBuffer;
        size_t size;
        size_t downloaded;
    };

private:
    //
    std::queue<RequestItem> m_RequestQueue;
    //
    int m_SessionCount;
    int m_SessionCountMax;
    //
    std::list<SessionItem> m_SessionList;
    //
    nn::os::Event m_CancelEvent;
    //
    CompletionCallback m_pCallback;
    void* m_pCallbackParam;

private:
    //
    void ClearRequests();
    void ClearSessions(CURLM* pMulti);
    //
    CURLcode DequeueRequests(CURLM* pMulti);
    //
    CURLcode CreateSession(SessionItem* pOutSession, const RequestItem& request) NN_NOEXCEPT;
    //
    int HandleCompletion(CURLM* pMulti);

private:
    //
    static CURLcode SslCtxFunction(CURL* pCurl, void* pSsl, void* pParam) NN_NOEXCEPT;
    static size_t HttpWriteFunction(char* pBuffer, size_t size, size_t count, void* pParam) NN_NOEXCEPT;
};
