﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace console {

/*!
    @brief      デフォルトのスクリーンの幅です。
*/
const int DefaultScreenWidth = 1280;

/*!
    @brief      デフォルトのスクリーンの高さです。
*/
const int DefaultScreenHeight = 720;

/*!
    @brief      コンソールの数です。
*/
const int ConsoleCount = 2;

/*!
    @brief      コンソールの生成パラメーターです。
*/
struct ConsoleCreateParam
{
    float lineHeight;        //!< ラインの高さ。
    int bufferWidth;         //!< 画面バッファの幅。
    int bufferHeight;        //!< 画面バッファの高さ。
    int windowWidth;         //!< ウィンドウの幅。
    int windowHeight;        //!< ウィンドウの高さ。
    int viewX;               //!< 表示領域の X 座標。
    int viewY;               //!< 表示領域の Y 座標。
    int viewWidth;           //!< 表示領域の幅。
    int viewHeight;          //!< 表示領域の高さ。
    bool isTimestampEnabled; //!< タイムスタンプを有効にするかどうか。
    bool isTeeEnabled;       //!< ターゲットマネージャーにもログを出すかどうか。
};

}

namespace console {

/*!
    @brief      コンソールモジュールを初期化します。

    @param[in]  screenWidth     スクリーンの幅。
    @param[in]  screenHeight    スクリーンの高さ。
*/
void Initialize(int screenWidth = DefaultScreenWidth, int screenHeight = DefaultScreenHeight) NN_NOEXCEPT;

/*!
    @brief      コンソールモジュールを終了します。
*/
void Finalize() NN_NOEXCEPT;

/*!
    @brief      画面の背景色を設定します。

    @param[in]  r   赤。
    @param[in]  g   緑。
    @param[in]  b   青。
*/
void SetScreenColor(float r, float g, float b) NN_NOEXCEPT;

/*!
    @brief      デフォルトのコンソール生成パラメーターを取得します。

    @return     コンソール生成パラメーター。
*/
const ConsoleCreateParam& GetDefaultConsoleCreateParam() NN_NOEXCEPT;

/*!
    @brief      コンソールを作成します。

    @param[in]  index       コンソール番号。
    @param[in]  createParam 生成パラメーター。

    @pre
        0 <= index < ConsoleCount
*/
void CreateConsole(int index, const ConsoleCreateParam& createParam) NN_NOEXCEPT;

/*!
    @brief      画面の表示設定を行います。

    @param[in]  index       コンソール番号。
    @param[in]  isVisible   表示するかどうか。

    @pre
        0 <= index < ConsoleCount
*/
void SetVisible(int index, bool isVisible) NN_NOEXCEPT;

/*!
    @brief      画面の表示設定を取得します。

    @param[in]  index   コンソール番号。

    @return     表示設定。

    @pre
        0 <= index < ConsoleCount
*/
bool GetVisible(int index) NN_NOEXCEPT;

/*!
    @brief      グラフィックスの 1 フレームの処理を実行します。
*/
void ProcessFrame() NN_NOEXCEPT;

/*!
    @brief      画面をクリアします。

    @param[in]  index   コンソール番号。

    @pre
        0 <= index < ConsoleCount
*/
void Clear(int index) NN_NOEXCEPT;

/*!
    @brief      テキストを表示します。

    @param[in]  index   コンソール番号。
    @param[in]  format  フォーマット文字列。
    @param[in]  ...     可変長引数。

    @pre
        0 <= index < ConsoleCount
*/
void Printf(int index, const char* format, ...) NN_NOEXCEPT;

/*!
    @brief      画面をスクロールします。

    @param[in]  index   コンソール番号。
    @param[in]  dy      スクロール量。

    @pre
        0 <= index < ConsoleCount
*/
void Scroll(int index, int dy) NN_NOEXCEPT;

/*!
    @brief      画面のスクロール位置を設定します。

    @param[in]  index   コンソール番号。
    @param[in]  y       スクロール位置。

    @pre
        0 <= index < ConsoleCount
*/
void SetScrollPosition(int index, int y) NN_NOEXCEPT;

}
