﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nns/hid.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/settings/settings_DebugPad.h>

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
// Windows プラットフォームにおける制約事項
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif

#include <nn/nn_Windows.h>
#endif

#if !defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#endif

//--------------------------------------------------------------------------------------------------
// Pad

class Pad
{
public:
    enum
    {
        BUTTON_LEFT = 0x00000001,
        BUTTON_RIGHT = 0x00000002,
        BUTTON_DOWN = 0x00000004,
        BUTTON_UP = 0x00000008,
        TRIGGER_Z = 0x00000010,
        TRIGGER_R = 0x00000020,
        TRIGGER_L = 0x00000040,
        BUTTON_A = 0x00000100,
        BUTTON_B = 0x00000200,
        BUTTON_X = 0x00000400,
        BUTTON_Y = 0x00000800,
        BUTTON_START = 0x00001000,
        STICK_R_LEFT = 0x00010000,
        STICK_R_RIGHT = 0x00020000,
        STICK_R_DOWN = 0x00040000,
        STICK_R_UP = 0x00080000,
        STICK_L_LEFT = 0x00100000,
        STICK_L_RIGHT = 0x00200000,
        STICK_L_DOWN = 0x00400000,
        STICK_L_UP = 0x00800000,
        BUTTON_STICK_L_LEFT = BUTTON_LEFT | STICK_L_LEFT,
        BUTTON_STICK_L_RIGHT = BUTTON_RIGHT | STICK_L_RIGHT,
        BUTTON_STICK_L_DOWN = BUTTON_DOWN | STICK_L_DOWN,
        BUTTON_STICK_L_UP = BUTTON_UP | STICK_L_UP
    };

    struct Stick
    {
        int8_t leftX, leftY, rightX, rightY;
    };
    struct AnalogStick
    {
        float leftX, leftY, rightX, rightY;
    };

    Pad()
        : m_InitNpad(false)
    {}

    void Initialize();
    void InitializeNpad();
    bool Reset();
    bool Read();

    bool IsHold(nn::Bit32 mask) const NN_NOEXCEPT
    {
        return 0 != (m_Button & mask);
    }
    bool IsTriggered(nn::Bit32 mask) const NN_NOEXCEPT
    {
        return 0 != (m_Triggered & mask);
    }
    bool IsReleased(nn::Bit32 mask) const NN_NOEXCEPT
    {
        return 0 != (m_Released & mask);
    }
    bool IsHoldAll(nn::Bit32 mask) const NN_NOEXCEPT
    {
        return mask == (m_Button & mask);
    }
    bool IsHoldAllAndTriggered(nn::Bit32 mask) const NN_NOEXCEPT
    {
        return IsHoldAll(mask) && IsTriggered(mask);
    }

    nn::Bit32 GetButton() const NN_NOEXCEPT
    {
        return m_Button;
    }
    nn::Bit32 GetLastButton() const NN_NOEXCEPT
    {
        return m_LastButton;
    }
    const Stick& GetStick() const NN_NOEXCEPT
    {
        return m_Stick;
    }
    const Stick& GetLastStick() const NN_NOEXCEPT
    {
        return m_LastStick;
    }
    const AnalogStick& GetAnalogStick() const NN_NOEXCEPT
    {
        return m_AnalogStick;
    }

private:
    enum NpadId
    {
        NPADID_NUM = 2,
    };

    enum NpadState
    {
        NpadState_Buttons = (1 << 0),
        NpadState_AnalogStickL = (1 << 1),
        NpadState_AnalogStickR = (1 << 2),
    };

    nn::hid::NpadIdType m_NpadIds[NPADID_NUM];
    bool m_InitNpad;
    void ReadNpad(nn::Bit32* pButton, Stick* pStick);
    nn::Bit32 GetNpadState(nn::hid::NpadButtonSet* pNpadButton,
        nn::hid::AnalogStickState* pAnalogStickL,
        nn::hid::AnalogStickState* pAnalogStickR);

private:
    nn::Bit32 m_Button;
    nn::Bit32 m_LastButton;
    Stick m_Stick;
    Stick m_LastStick;
    AnalogStick m_AnalogStick;
    nn::Bit32 m_Triggered;
    nn::Bit32 m_Released;
    nn::settings::DebugPadKeyboardMap m_DebugPadKeyboardMap;

    NN_DISALLOW_COPY(Pad);
};

Pad& GetPad() NN_NOEXCEPT;

nn::hid::MouseState& GetMouseState() NN_NOEXCEPT;
nns::hid::ControllerManager& GetControllerManager() NN_NOEXCEPT;


void InitializeInputController(int width, int height, void* hwnd);

void UpdateInputController();
