﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/font.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include "FontWordWrapping.h"

//------------------------------------------------------------------------------
//
// ワードラッピングのデモです。
//
// nn::font::WordWrapping クラスを用いて、行の表示幅に応じて文字列の適切な
// 位置に改行コードを挿入します。
//
//------------------------------------------------------------------------------

static const uint32_t Margin = 25;

// ワードラップ処理の対象となるソース文字列です。
static const uint16_t* g_Text;

// ワードラップ処理された文字列を格納するバッファです。
// ソース文字列＋挿入される改行コード分の容量が必要です。
static uint16_t* g_WordWrappedText;

// g_WordWrappedTextに格納可能な文字数(ヌル終端を含む)です。
static uint32_t g_WordWrappedTextSize;

// 表示する行数です。
static uint32_t g_LineCount;

static uint32_t g_WordWrappedCount = 0;

//------------------------------------------------------------------------------
void InitializeWordWrappingText()
{
    g_Text = reinterpret_cast<const uint16_t*>(
        NN_CHAR16LITERAL("\001nn::font::WordWrapping Sample\002\n")
        NN_CHAR16LITERAL("\n")
        NN_CHAR16LITERAL("\001Creating a Layout\002\n")
        NN_CHAR16LITERAL("\n")
        NN_CHAR16LITERAL("As a first step, here we provide a description of the basic work ")
        NN_CHAR16LITERAL("flow, which involves:\n")
        NN_CHAR16LITERAL("\n")
        NN_CHAR16LITERAL("* Creating a new layout\n")
        NN_CHAR16LITERAL("* Positioning the pane\n")
        NN_CHAR16LITERAL("* Previewing in the Viewer\n")
        NN_CHAR16LITERAL("* Saving the data\n")
        NN_CHAR16LITERAL("\n")
        NN_CHAR16LITERAL("\001Creating a New Layout\002\n")
        NN_CHAR16LITERAL("\n")
        NN_CHAR16LITERAL("Start LayoutEditor and create a new layout file by selecting ")
        NN_CHAR16LITERAL("File > Create New > Layout.")
        NN_CHAR16LITERAL("A window with the title \"Untitled\" will appear on the screen. ")
        NN_CHAR16LITERAL("This is called the Layout Window.")
        NN_CHAR16LITERAL("With LayoutEditor, you create game screen layouts by ")
        NN_CHAR16LITERAL("positioning images in the Layout Window.\n")
        NN_CHAR16LITERAL("\n")
        NN_CHAR16LITERAL("\001Description of the Layout Window Screen\002\n")
        NN_CHAR16LITERAL("\n")
        NN_CHAR16LITERAL("Look at the Layout Window and notice that a black square drawn ")
        NN_CHAR16LITERAL("on the screen.")
        NN_CHAR16LITERAL("This square is called the layout screen region (A).")
        NN_CHAR16LITERAL("The layout screen region corresponds to the screen shown on the ")
        NN_CHAR16LITERAL("actual hardware.")
        NN_CHAR16LITERAL("The red line drawn in the center of the screen is the coordinate ")
        NN_CHAR16LITERAL("origin of the layout screen region (B). ")
        NN_CHAR16LITERAL("(The layout coordinate system is a three-dimensional coordinate ")
        NN_CHAR16LITERAL("system consisting of a two-dimensional X-Y axis plus a ")
        NN_CHAR16LITERAL("one-dimensional Z axis.)\n")
    );
}

//------------------------------------------------------------------------------
// 初期化・終了
//------------------------------------------------------------------------------
void InitializeWordWrapping(int screenHeight, int fontHeight)
{
    // 表示する行数です。
    g_LineCount = (screenHeight - 2 * Margin) / fontHeight;

    // ワードラップ処理したテキストを格納する文字列バッファを確保
    g_WordWrappedTextSize = nn::font::CalculateWideCharString16Length(g_Text) + g_LineCount;
    g_WordWrappedText = static_cast<uint16_t*>(AllocateFromApplicationHeap(g_WordWrappedTextSize * sizeof(uint16_t)));
    g_WordWrappedCount = 0;
}

void FinalizeWordWrapping()
{
    DeallocateApplicationHeap(g_WordWrappedText);
}

//---------------------------------------------------------------------------
// 更新処理
//---------------------------------------------------------------------------
void CalcWordWrapping(nn::font::DispStringBuffer& dispStringBuffer, nn::font::Font& font, int screenWidth)
{
    nn::font::WideTextWriter writer;
    writer.SetDispStringBuffer( &dispStringBuffer );
    writer.SetFont( &font );
    writer.SetCursor( static_cast<float>(Margin), static_cast<float>(Margin) );
    writer.SetScale( 1.0f, 1.0f );

    MyTagProcessor tagProcessor;
    writer.SetTagProcessor( &tagProcessor );

    // 現在の設定を通常設定としてタグプロセッサに記憶させます。
    tagProcessor.SaveDefaultSetting(&writer);

    // 一行に表示できる幅を WideTextWriter に設定します。
    // これにより WideTextWriter::GetLineBreak() により一行に表示できる
    // 文字列の範囲を求めることができます。
    const uint32_t loop = 60 * 4;
    g_WordWrappedCount = (g_WordWrappedCount + loop - 1) % loop;
    float widthLimit = (static_cast<float>(screenWidth) - Margin * 2) * (g_WordWrappedCount + 1) / loop;
    writer.SetWidthLimit(widthLimit);

    // 一行の表示範囲に収まる文字列の範囲を判定するコールバックを作成します。
    nn::font::DefaultWordWrapCallback wordWrapCallback(&writer);

    // 画面に収まる行数に制限します。
    nn::font::WordWrapConfig config;
    config.SetLineCountMax(g_LineCount);

    // ワードラップ処理を行います。
    nn::font::WordWrapping::CalculateWordWrapping(NULL, g_WordWrappedText, g_WordWrappedTextSize, g_Text, wordWrapCallback, config);

    // 文字上端、下端カラーの設定を行います。
    // 上端、下端カラーは頂点カラーで実現されるため、
    // カラーを変更した場合、描画コマンドの生成を再度行わなければならない点に注意してください。
    nn::util::Unorm8x4 topColor = {{0, 255, 0, 255}};
    nn::util::Unorm8x4 bottomColor = {{0, 255, 0, 255}};
    writer.SetTextColor(topColor, bottomColor);

    writer.StartPrint();
    writer.Print(g_Text);
    writer.EndPrint();
}
