﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{EnsFsUtility.h,PageSampleEnsMessage}
 *
 * @brief
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/fs.h>

/**
 * @namespace nns::ens
 * @brief   ensのサンプル向け名前空間です。
 */
namespace nns { namespace ens {

/**
 * @brief   ファイルを読み込みます。
 *
 * @tparam  T   読み込む型
 * @param[out]  pOut    出力
 * @param[in]   path    ファイルパス
 *
 * @pre
 *  - ファイルシステムをマウント済
 *  - pOut != nullptr
 *  - path != nullptr
 *
 * @return  成否
 * @retval  true    成功
 * @retval  false   失敗(pathに該当するファイルがない)
 *
 * @details
 */
template <typename T>
bool ReadFile(T* pOut, const char* path) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    auto result = nn::fs::OpenFile(&handle, path, nn::fs::OpenMode::OpenMode_Read);

    if(nn::fs::ResultPathNotFound::Includes(result))
    {
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, pOut, sizeof(T)));

    nn::fs::CloseFile(handle);

    return true;
}

/**
 * @brief   ファイルを書き込み、コミットします。
 *
 * @tparam  T   書き込む型
 * @param[in]   value       書き込む値
 * @param[in]   mountName   マウント名
 * @param[in]   path        ファイルパス
 *
 * @pre
 *  - ファイルシステムをマウント済
 *  - mountName != nullptr
 *  - path != nullptr
 *
 * @details
 */
template <typename T>
void WriteFile(const T& value, const char* mountName, const char* path) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    auto result = nn::fs::OpenFile(&handle, path, nn::fs::OpenMode::OpenMode_Write);

    if(nn::fs::ResultPathNotFound::Includes(result))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateFile(path, sizeof(T)));

        result = nn::fs::OpenFile(&handle, path, nn::fs::OpenMode::OpenMode_Write);
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::WriteFile(
        handle, 0, &value, sizeof(T), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

    nn::fs::CloseFile(handle);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::Commit(mountName));
}

/**
 * @brief   ファイルを削除し、コミットします。
 *
 * @param[in]   mountName   マウント名
 * @param[in]   path        ファイルパス
 *
 * @pre
 *  - ファイルシステムをマウント済
 *  - mountName != nullptr
 *  - path != nullptr
 *
 * @details
 *  pathに該当するファイルが存在しない場合は何も行いません。
 */
void DeleteFile(const char* mountName ,const char* path) NN_NOEXCEPT
{
    auto result = nn::fs::DeleteFile(path);

    if(nn::fs::ResultPathNotFound::Includes(result))
    {
        // do nothing.
        return;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::Commit(mountName));
}

}} // end of namespace nns::ens
