﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <string>
#include <vector>

#include <nn/gfx.h>
#include <nn/hid.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>

#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/TargetConfigs/build_Platform.h>
#include <nn/util/util_Color.h>

#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeshRes.h>

#include "Demo1.h"
#include "Demo1Color.h"
#include "Demo1NpadSixAxisSensor.h"
#include "Demo1PluginManager.h"

namespace {

SET_PLUGIN( "SixAxisSensor", SixAxisSensorDemo, PluginProperty_Drawable );

const float DispWidth = 1280.0f;
const float DispHeight = 720.0f;
const float OffsetX = 20.0f;
const nn::hid::NpadIdType NpadIds1to4[] = {nn::hid::NpadId::No1,
                                       nn::hid::NpadId::No2,
                                       nn::hid::NpadId::No3,
                                       nn::hid::NpadId::No4};

const nn::util::Unorm8x4 ColorArray[]  = {Color::Red,
                                          Color::Orange,
                                          Color::Violet,
                                          Color::Green};

/**
 * @brief       描画する座標系の識別子です。
 */
enum CoordinateId
{
    CoordinateId_AccelerometerXy = 0,
    CoordinateId_AccelerometerYz,
    CoordinateId_AccelerometerZx,
    CoordinateId_AngularVelocityXy,
    CoordinateId_AngularVelocityYz,
    CoordinateId_AngularVelocityZx,
    CoordinateId_AngleX,
    CoordinateId_AngleY,
    CoordinateId_AngleZ,
    CoordinateId_DirectionXy,
    CoordinateId_DirectionYz,
    CoordinateId_DirectionZx,
};

// 6軸センサーの状態を座標系上に描画します。
void WriteSixAxisSensorStateFigure(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                                   const nn::hid::SixAxisSensorState& state,
                                   const nn::util::Unorm8x4& Color,
                                   const int playerNumber) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4& TextColor =
        state.attributes.Test<nn::hid::SixAxisSensorAttribute::IsConnected>() ?
        Color : Color::Gray;

    pTextWriter->SetTextColor(TextColor);

    const float DeltaY = NormalFontHeight;
    pTextWriter->SetScale(NormalFontScaleX, NormalFontScaleY);

    float posX = OffsetX + DispWidth / 2.0f + (DispWidth / 4.0f) * ((playerNumber - 1) % 2);
    float posY = DeltaY + (DispHeight / 2.0f) * ((playerNumber - 1) / 2);

    pTextWriter->SetTextColor(TextColor);
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("Player: %d", playerNumber);
    posY += DeltaY;

    // Accelerometer
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("Accelerometer");
    posY += DeltaY;

    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("(%f, %f, %f)",
        state.acceleration.x,
        state.acceleration.y,
        state.acceleration.z
    );
    posY += DeltaY;
    posY += DeltaY; // 空行

    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("AngularVelocity");
    posY += DeltaY;

    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("(%f, %f, %f)",
        state.angularVelocity.x,
        state.angularVelocity.y,
        state.angularVelocity.z
    );
    posY += DeltaY;
    posY += DeltaY; // 空行

    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("Angle");
    posY += DeltaY;

    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("(%f, %f, %f)",
        state.angle.x,
        state.angle.y,
        state.angle.z
    );
    posY += DeltaY;
    posY += DeltaY; // 空行

    // Direction
    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("Direction");
    posY += DeltaY;

    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("X (%f, %f, %f)",
        state.direction.x.x,
        state.direction.x.y,
        state.direction.x.z
    );
    posY += DeltaY;

    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("Y (%f, %f, %f)",
        state.direction.y.x,
        state.direction.y.y,
        state.direction.y.z
    );
    posY += DeltaY;

    pTextWriter->SetCursor(posX, posY);
    pTextWriter->Print("Z (%f, %f, %f)",
        state.direction.z.x,
        state.direction.z.y,
        state.direction.z.z
    );
    posY += DeltaY;
}


// ボタンのトリガー入力を取得します。
nn::hid::NpadButtonSet GetTriggerButtons(
    const nn::hid::NpadButtonSet current,
    const nn::hid::NpadButtonSet previous) NN_NOEXCEPT
{
    return (current ^ previous) & current;
}

class NpadSixAxisSensor
{
    NN_DISALLOW_COPY(NpadSixAxisSensor);
    NN_DISALLOW_MOVE(NpadSixAxisSensor);

public:
    explicit NpadSixAxisSensor(const nn::hid::NpadIdType& id) NN_NOEXCEPT
        : m_pId(&id)
    {
    }

    virtual ~NpadSixAxisSensor() NN_NOEXCEPT
    {
    }

    virtual void Initialize(nn::hid::NpadStyleSet style) NN_NOEXCEPT
    {
        nn::hid::GetSixAxisSensorHandles(&m_Handle,
            nn::hid::NpadSixAxisSensorHandleCountMax,
            *m_pId,
            style);

        nn::hid::StartSixAxisSensor(m_Handle);
        m_Style = style;
    }

    virtual bool CanPrint() NN_NOEXCEPT
    {
        return (GetTriggerButtons(m_ButtonState[0], m_ButtonState[1]).IsAnyOn());
    }

    virtual void Update() NN_NOEXCEPT
    {
        m_ButtonState[1] = m_ButtonState[0];

        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(*m_pId);

        // ボタンの状態を更新
        NpadPluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
        if(pNpad == NULL)
        {
            m_ButtonState[0].Reset();
            return;
        }

        m_ButtonState[0] = pNpad->GetNpadButtonState(*m_pId).buttons;

        nn::hid::GetSixAxisSensorState(&m_State, m_Handle);

        if (pNpad->IsEnableNpadStyle(m_Style) == false)
        {
            m_ButtonState[0].Reset();
        }
    }

    virtual void Draw(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
    {
        int i = static_cast<uint32_t>(*m_pId);

        WriteSixAxisSensorStateFigure(pTextWriter,
            m_State,
            ColorArray[i],
            i + 1);
    }

    virtual bool IsConnected() const NN_NOEXCEPT
    {
        //現在有効な操作形態(NpadStyleSet)を取得
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(*m_pId);

        // ボタンの状態を更新
        NpadPluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
        if(pNpad == NULL)
        {
            return false;
        }

        return pNpad->IsEnableNpadStyle(m_Style);
    }

    virtual nn::hid::NpadIdType GetNpadIdType() const NN_NOEXCEPT
    {
        return *m_pId;
    }

private:
    nn::hid::NpadStyleSet        m_Style;
    nn::hid::NpadButtonSet       m_ButtonState[2];
    nn::hid::SixAxisSensorHandle m_Handle;
    nn::hid::SixAxisSensorState  m_State;
    const nn::hid::NpadIdType*   m_pId;
};

}; // namespace


void SixAxisSensorDemo::Initialize() NN_NOEXCEPT
{
    for(int i = 0; i < NN_ARRAY_SIZE(NpadIds1to4); i++)
    {
        CreateSixAxisSensor(NpadIds1to4[i]);
    }
}

void SixAxisSensorDemo::Finalize() NN_NOEXCEPT
{
}

void SixAxisSensorDemo::Draw() NN_NOEXCEPT
{
    static int64_t s_Frame = 0;
    MakeCommand(s_Frame, m_Name.c_str());
    s_Frame++;
}

bool SixAxisSensorDemo::CreateSixAxisSensor(const nn::hid::NpadIdType& id) NN_NOEXCEPT
{
    // 現在有効な操作形態(NpadStyleSet)を取得
    nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(id);

    NpadPluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
    if(pNpad == NULL)
    {
        return false;
    }

    // 各種操作形態を管理対象に追加
    NpadSixAxisSensor* pSixAxisSensor = new NpadSixAxisSensor(id);

    pSixAxisSensor->Initialize(pNpad->GetNpadStyleSet());
    m_NpadStyleSixAxisSensors.push_back(pSixAxisSensor);

    return true;
}

void  SixAxisSensorDemo::MakeCommand(int64_t frame, const char* pName) NN_NOEXCEPT
{
    NN_UNUSED(frame);

    nn::gfx::CommandBuffer& commandBuffer = m_pGraphicsSystem->GetCommandBuffer();
    nn::gfx::util::DebugFontTextWriter& debugFontTextWriter = m_pGraphicsSystem->GetDebugFont();

    for (std::vector<void*>::iterator it = m_NpadStyleSixAxisSensors.begin();
         it != m_NpadStyleSixAxisSensors.end();
        ++it)
    {
        NpadSixAxisSensor* pSixAxis = reinterpret_cast<NpadSixAxisSensor*>(*it);
        if (!pSixAxis->IsConnected())
        {
            continue;
        }
        pSixAxis->Update();
        pSixAxis->Draw(&debugFontTextWriter);
        pSixAxis->CanPrint();
    }

    // 再検出処理
    for(int i = 0; i < NN_ARRAY_SIZE(NpadIds1to4); i++)
    {
        bool hasCreated;
        hasCreated = false;

        for (std::vector<void*>::iterator it = m_NpadStyleSixAxisSensors.begin();
             it != m_NpadStyleSixAxisSensors.end();
            ++it)
        {
            NpadSixAxisSensor* pSixAxis = reinterpret_cast<NpadSixAxisSensor*>(*it);
            if(pSixAxis->GetNpadIdType() == NpadIds1to4[i])
            {
                hasCreated = true;
                break;
            }
        }
        if(hasCreated == true)
        {
            continue;
        }

        // NpadIdに対応するインスタンスが未作成の場合はインスタンスを作成
        CreateSixAxisSensor(NpadIds1to4[i]);
    }

    // 共通操作説明を描画します。
    WriteCommonGuide(&debugFontTextWriter, pName);

    // 負荷メータを表示する
    DrawLoadMeter();

    // テキストを描画
    debugFontTextWriter.Draw(&commandBuffer);
}
