﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/gfx.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/util/util_Color.h>

#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeshRes.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>

#include "Demo1.h"
#include "Demo1Npad.h"
#include "Demo1PluginManager.h"

void NpadDemo::Initialize() NN_NOEXCEPT
{
    nn::hid::InitializeNpad();

    nns::hid::util::SetControllerManagerWithDefault(&GetPluginManager().GetControllerManager());

    // 使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(GetPluginManager().GetSupportedNpadStyleSet());

    // 使用する Npad を設定
    nn::hid::SetSupportedNpadIdType(NpadIds, NN_ARRAY_SIZE(NpadIds));

    m_pThreadStack = reinterpret_cast<char*>(
        GetStandardAllocator().Allocate(
            ThreadStackSize,
            nn::os::ThreadStackAlignment
        )
        );

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &m_NpadThread, NpadThread, this, m_pThreadStack,
        ThreadStackSize, nn::os::DefaultThreadPriority));
}

NpadDemo::~NpadDemo() NN_NOEXCEPT
{
}

void NpadDemo::Start() NN_NOEXCEPT
{
    nn::os::StartThread(&m_NpadThread);
}

void NpadDemo::Finalize() NN_NOEXCEPT
{
    nn::os::DestroyThread(&m_NpadThread);

    GetStandardAllocator().Free(m_pThreadStack);
}

void NpadDemo::End() NN_NOEXCEPT
{
}

// Npad のスティックの状態
void NpadDemo::SetNpadStickState(const NpadCommonState state) NN_NOEXCEPT
{
    char sticks[18] = {};
    m_CurrentStick += " L";
    nn::util::SNPrintf(sticks, NN_ARRAY_SIZE(sticks), "(%6d, %6d)",
                       state.analogStickL.x, state.analogStickL.y);
    m_CurrentStick += sticks;
    m_CurrentStick += " R";
    nn::util::SNPrintf(sticks, NN_ARRAY_SIZE(sticks), "(%6d, %6d)",
                       state.analogStickR.x, state.analogStickR.y);
    m_CurrentStick += sticks;
}

void NpadDemo::SetWriter(
    nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const float offsetX,
    const float offsetY,
    size_t stepCounter,
    const size_t npadNum) NN_NOEXCEPT
{
    if (npadNum == 0)
    {
        stepCounter += 1;
        pTextWriter->SetCursor(offsetX, offsetY * stepCounter);
        pTextWriter->Print(m_OutStick[0].c_str());
    }
    else if (npadNum == 1)
    {
        stepCounter += 2;
        pTextWriter->SetCursor(offsetX, offsetY * stepCounter);
        pTextWriter->Print(m_OutStick[0].c_str());
    }
    else if (npadNum == 2)
    {
        stepCounter += 3;
        pTextWriter->SetCursor(offsetX, offsetY * stepCounter);
        pTextWriter->Print(m_OutStick[0].c_str());
    }
    else if (npadNum == 3)
    {
        stepCounter += 4;
        pTextWriter->SetCursor(offsetX, offsetY * stepCounter);
        pTextWriter->Print(m_OutStick[0].c_str());
    }
    else if (npadNum == 4)
    {
        stepCounter += 5;
        pTextWriter->SetCursor(offsetX, offsetY * stepCounter);
        pTextWriter->Print(m_OutStick[0].c_str());
    }
    else if (npadNum == 5)
    {
        stepCounter += 6;
        pTextWriter->SetCursor(offsetX, offsetY * stepCounter);
        pTextWriter->Print(m_OutStick[0].c_str());
    }
    else if (npadNum == 6)
    {
        stepCounter += 7;
        pTextWriter->SetCursor(offsetX, offsetY * stepCounter);
        pTextWriter->Print(m_OutStick[0].c_str());
    }
    else if (npadNum == 7)
    {
        stepCounter += 8;
        pTextWriter->SetCursor(offsetX, offsetY * stepCounter);
        pTextWriter->Print(m_OutStick[0].c_str());
    }
}

// Npad の状態を更新
void NpadDemo::WriteNpadStickState(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    size_t stepCounter = 12;
    float offsetX = 650.0f;
    float offsetY = 30.0f;

    pTextWriter->SetTextColor(Color::Yellow);
    pTextWriter->SetScale(NormalFontScaleX, NormalFontScaleY);
    pTextWriter->SetCursor(offsetX, offsetY * static_cast<float>(stepCounter));
    pTextWriter->Print("Npad Stick");

    pTextWriter->SetScale(NormalFontScaleX, NormalFontScaleY);

    if (m_OutStick.size() > 10)
    {
        m_OutStick.erase(m_OutStick.begin() + 10);
    }

    char styleStr[14] = {};
    int column = 0;
    for (int i = 0; i < NN_ARRAY_SIZE(NpadIds); i++)
    {
        //現在有効な操作形態(NpadStyleSet)を取得
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadIds[i]);

        NpadPluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
        if(pNpad == NULL)
        {
            continue;
        }

        // ボタンの入力状態を取得
        NpadCommonState state = pNpad->GetNpadButtonState(NpadIds[i]);

        pTextWriter->SetTextColor(pNpad->GetMenuTextColor());
        nn::util::SNPrintf(styleStr, NN_ARRAY_SIZE(styleStr), "%s (%d)  ",
                           pNpad->GetName().c_str(), i);
        m_CurrentStick = styleStr;
        SetNpadStickState(state);
        m_OutStick.insert(m_OutStick.begin(), m_CurrentStick);

        SetWriter(pTextWriter, offsetX, offsetY, stepCounter, column++);
    }
    ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
}

// Npad の状態を更新
void NpadDemo::UpdateNpadState() NN_NOEXCEPT
{
    char styleStr[14] = {};
    for (int i = 0; i < NN_ARRAY_SIZE(NpadIds); i++)
    {
        static nn::hid::NpadButtonSet s_OldButtons[NN_ARRAY_SIZE(NpadIds)] = {};

        //現在有効な操作形態(NpadStyleSet)を取得
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadIds[i]);

        NpadPluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
        if(pNpad == NULL)
        {
            continue;
        }

        // ボタンの入力状態を取得
        NpadCommonState state = pNpad->GetNpadButtonState(NpadIds[i]);

        //ボタンが押された
        if ((state.buttons ^ s_OldButtons[i] & state.buttons).IsAnyOn())
        {
            // Npad の入力状態を表示
            nn::util::SNPrintf(styleStr, NN_ARRAY_SIZE(styleStr), "%s (%d)  ",
                               pNpad->GetName().c_str(), i);
            m_CurrentString = styleStr;
            m_CurrentString += pNpad->GetNpadButtonStateString(state.buttons);
            m_OutString.insert(m_OutString.begin(), m_CurrentString);
        }
        s_OldButtons[i] = state.buttons;
    }
}

// Npad の状態を描画する
void NpadDemo::DrawNpadStates() NN_NOEXCEPT
{
    if (m_OutString.size() > 10)
    {
        m_OutString.erase(m_OutString.begin() + 10);
    }

    UpdateNpadState();

    WriteNpadStickState(&GetDebugFont());

    GetDebugFont().SetScale(NormalFontScaleX, NormalFontScaleY);

    if (m_OutString.size() > 0)
    {
        size_t stepCounter = 1;
        for (size_t i = 0; i < m_OutString.size(); i++)
        {
            if (m_OutString[i].substr(0, 1) == "F")
            {
                GetDebugFont().SetTextColor(nn::util::Color4u8::Magenta());
            }
            else if (m_OutString[i].substr(0, 1) == "J")
            {
                GetDebugFont().SetTextColor(nn::util::Color4u8::Cyan());
            }
            else
            {
                GetDebugFont().SetTextColor(nn::util::Color4u8::Green());
            }

            if (stepCounter > 10)
            {
                break;
            }
            GetDebugFont().SetCursor(650.0f, 30.0f * static_cast<float>(stepCounter++));
            GetDebugFont().Print(m_OutString[i].c_str());
        }
    }
}

void NpadDemo::MakeCommand(int64_t frame, const char* pName) NN_NOEXCEPT
{
    NN_UNUSED(frame);

    nn::gfx::CommandBuffer& commandBuffer = m_pGraphicsSystem->GetCommandBuffer();
    nn::gfx::util::DebugFontTextWriter& debugFontTextWriter = m_pGraphicsSystem->GetDebugFont();

    // テスト用にテキスト出力
    GetPluginManager().GetControllerManager().Update();

    //!< 共通操作説明を描画します。
    WriteCommonGuide( &debugFontTextWriter, pName);

    // 負荷メータを表示する
    DrawLoadMeter();

    // Npad の状態を描画する
    DrawNpadStates();

    // テキストを描画
    debugFontTextWriter.Draw(&commandBuffer);
}

void NpadDemo::Draw() NN_NOEXCEPT
{
    static int64_t s_Frame = 0;
    MakeCommand(
        s_Frame,
        m_Name.c_str());
    s_Frame++;
}

void NpadDemo::Wait() NN_NOEXCEPT
{
    nn::os::WaitThread(&m_NpadThread);
}

int NpadDemo::GetSceneChangeCount() NN_NOEXCEPT
{
    m_MutexIndex.Lock();
    int drawIndex = m_SceneChangeCount;
    m_SceneChangeCount = 0;
    m_MutexIndex.Unlock();

    return drawIndex;
}

bool NpadDemo::EndTrigger() NN_NOEXCEPT
{
    m_MutexState.Lock();
    for (int i = 0; i < NN_ARRAY_SIZE(NpadIds); i++)
    {
        // 現在有効な操作形態(NpadStyleSet)を取得
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadIds[i]);

        NpadPluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
        if(pNpad == NULL)
        {
            continue;
        }

        // ボタンの入力状態を取得
        NpadCommonState state = pNpad->GetNpadButtonState(NpadIds[i]);

        // ＋ と - ボタンを同時に押されたら終了
        if (state.buttons.Test<nn::hid::NpadButton::Plus>() &&
                state.buttons.Test<nn::hid::NpadButton::Minus>())
        {
            nn::os::SignalEvent(&GetGlobalEvent());
        }
    }
    m_MutexState.Unlock();
    return (nn::os::TryWaitEvent(&GetGlobalEvent()));
}

void NpadDemo::Update() NN_NOEXCEPT
{
    m_MutexState.Lock();
    bool isPressedMinus = false;
    bool isPressedPlus = false;
    for (int i = 0; i < NN_ARRAY_SIZE(NpadIds); i++)
    {
        // 現在有効な操作形態(NpadStyleSet)を取得
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadIds[i]);

        // ボタンの状態を更新
        NpadPluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
        if(pNpad == NULL)
        {
            continue;
        }

        // ボタンの入力状態を取得
        NpadCommonState state = pNpad->GetNpadButtonState(NpadIds[i]);

        if (state.buttons.Test<nn::hid::NpadButton::Minus>())
        {
            isPressedMinus = true;
        }
        else if (state.buttons.Test<nn::hid::NpadButton::Plus>())
        {
            isPressedPlus = true;
        }
    }
    m_MutexState.Unlock();

    m_MutexIndex.Lock();
    if (isPressedMinus && !m_IsPrePressedMinus)
    {
        // 前に Minus が押されていないかつ現在押されている場合
        m_SceneChangeCount = -1;
    }
    m_IsPrePressedMinus = isPressedMinus;

    if (isPressedPlus && !m_IsPrePressedPlus)
    {
        // 前に Plus が押されていないかつ現在押されている場合
        m_SceneChangeCount = 1;
    }
    m_IsPrePressedPlus = isPressedPlus;
    m_MutexIndex.Unlock();
}

void NpadDemo::NpadThread(void* pArg) NN_NOEXCEPT
{
    NpadDemo * pNpadDemo = reinterpret_cast<NpadDemo*>(pArg);
    pNpadDemo->NpadThreadImpl();
}

void NpadDemo::NpadThreadImpl() NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        Update();

        if (EndTrigger())
        {
            break;
        }
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
    }
}
