﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <sstream>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "Demo1ClusteringModeState.h"
#include "Demo1Color.h"
#include "Demo1HandAnalysisModeState.h"
#include "Demo1ImageTransferModeState.h"
#include "Demo1IrSensorDemo.h"
#include "Demo1MomentModeState.h"
#include "Demo1PluginManager.h"

namespace
{

void WriteIrCameraTitle(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    int index) NN_NOEXCEPT
{
    pTextWriter->SetScale(NormalFontScaleX, NormalFontScaleY);
    pTextWriter->SetTextColor(Color::White);

    const float posX = (index == 0) ? 695.0f : 995.0f;
    const float posY = 40.0f;
    pTextWriter->SetCursor(posX, posY);
    if (index == 0)
    {
        pTextWriter->Print("1st IrCamera Status");
    }
    else
    {
        pTextWriter->Print("2nd IrCamera Status");
    }
}


void WriteIrCameraStatus(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    nn::irsensor::IrCameraStatus irCameraStatus,
    int index) NN_NOEXCEPT
{
    pTextWriter->SetScale(NormalFontScaleX, NormalFontScaleY);
    pTextWriter->SetTextColor(Color::Red);

    const float posX = (index == 0) ? 695.0f : 995.0f;
    const float posY = 70.0f;
    pTextWriter->SetCursor(posX, posY);
    if (irCameraStatus == nn::irsensor::IrCameraStatus_Unsupported)
    {
        pTextWriter->Print("IrCameraStatus_Unsupported");
    }
    else if (irCameraStatus == nn::irsensor::IrCameraStatus_Unconnected)
    {
        pTextWriter->Print("IrCameraStatus_Unconnected");
    }
}


void DrawMenu(const std::vector<MenuItem>& menu,
    const float offsetX, const float offsetY,
    int selectIndex,
    nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
{

    NN_ASSERT_NOT_NULL(pTextWriter);

    pTextWriter->SetScale(NormalFontScaleX, NormalFontScaleY);

    for (size_t index = 0; index < menu.size(); index++)
    {
        pTextWriter->SetTextColor(
            (static_cast<size_t>(selectIndex)  == index) ? Color::Red : Color::White
        );
        pTextWriter->SetCursor(offsetX, offsetY + index * 20.0f);

        std::stringstream text;
        text << menu[index].GetName() << ": ";
        menu[index].Read(text);

        pTextWriter->Print(text.str().c_str());
    }
}

void WriteProcessorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    pTextWriter->SetScale(NormalFontScaleX, NormalFontScaleY);

    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetCursor(offsetX + 330, offsetY + 210);
    pTextWriter->Print("Press X / B to change component");
    pTextWriter->SetCursor(offsetX + 330, offsetY + 240);
    pTextWriter->Print("Press Y / A to change value");
    pTextWriter->SetCursor(offsetX + 330, offsetY + 270);
    pTextWriter->Print("Press R to reflect user setting");
}

}

void CreateSampler(GraphicsSystem* pGraphicsSystem, Sampler* pSampler) NN_NOEXCEPT
{
    nn::gfx::Sampler::InfoType samplerInfo;
    samplerInfo.SetDefault();
    samplerInfo.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
    samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
    samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
    samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_Mirror);
    pSampler->GetSampler().Initialize(&pGraphicsSystem->GetDevice(), samplerInfo);

    pGraphicsSystem->RegisterSamplerSlot(&pSampler->GetDescriptorSlot(), pSampler->GetSampler());
}

void CreateTexture(GraphicsSystem* pGraphicsSystem,
    int width, int height,
    nn::gfx::ImageFormat format,
    Texture* pTexture) NN_NOEXCEPT
{
    {
        nn::gfx::Texture::InfoType textureInfo;

        textureInfo.SetDefault();
        textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
        textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        textureInfo.SetMipCount(1);
        textureInfo.SetTileMode(nn::gfx::TileMode_Linear);
        textureInfo.SetWidth(width);
        textureInfo.SetHeight(height);
        textureInfo.SetImageFormat(format);

        pGraphicsSystem->AllocateTexture(&pTexture->GetTexture(), &textureInfo);
    }

    {
        nn::gfx::TextureView::InfoType viewInfo;
        viewInfo.SetDefault();
        viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        viewInfo.SetImageFormat(format);
        viewInfo.SetTexturePtr(&pTexture->GetTexture());
        viewInfo.SetChannelMapping(nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red,
            nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red);

        pTexture->GetTextureView().Initialize(&pGraphicsSystem->GetDevice(), viewInfo);
    }

    {
        pGraphicsSystem->RegisterTextureViewSlot(
            &pTexture->GetDescriptorSlot(),
            pTexture->GetTextureView());
    }
}

void IrSensorDemo::Initialize(
    nn::hid::NpadIdType id,
    GraphicsSystem* pGraphicsSystem,
    void* pImageTransferWorkeMemory
)
{
    m_NpadId = id;

    //IR カメラのハンドルの取得
    m_IrCameraHandle = nn::irsensor::GetIrCameraHandle(m_NpadId);

    //IR カメラの初期化
    nn::irsensor::Initialize(m_IrCameraHandle);

    // IrSensor を Moment として初期化
    m_pModeStates[IrSensorMode_Moment] = new MomentModeState(
        &m_NextProcessor, &m_MenuSelection, m_IrCameraHandle);
    m_pModeStates[IrSensorMode_Clustering] = new ClusteringModeState(
        &m_NextProcessor, &m_MenuSelection, m_IrCameraHandle);

    m_pModeStates[IrSensorMode_ImageTransfer] = new ImageTransferModeState(
        &m_NextProcessor, &m_MenuSelection, m_IrCameraHandle,
        pImageTransferWorkeMemory, pGraphicsSystem);
    m_pModeStates[IrSensorMode_HandAnalysis] = new HandAnalysisModeState(
        &m_NextProcessor, &m_MenuSelection, m_IrCameraHandle,
        pGraphicsSystem);

    m_CurrentProcessor = m_NextProcessor = IrSensorMode_Moment;
    m_pModeStates[m_CurrentProcessor]->Start();
}

void IrSensorDemo::Finalize() NN_NOEXCEPT
{
    m_pModeStates[m_CurrentProcessor]->Stop();

    delete m_pModeStates[IrSensorMode_Moment];
    delete m_pModeStates[IrSensorMode_Clustering];
    delete m_pModeStates[IrSensorMode_ImageTransfer];
    delete m_pModeStates[IrSensorMode_HandAnalysis];

    nn::irsensor::Finalize(m_IrCameraHandle);
}

bool IrSensorDemo::Run(int i, GraphicsSystem* pGraphicsSystem,
    nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
{
    //IR カメラの状態の取得

    const nn::irsensor::IrCameraStatus IrCameraStatus = nn::irsensor::GetIrCameraStatus(
        m_IrCameraHandle
    );

    WriteIrCameraTitle(pTextWriter, i);
    if (nn::irsensor::IrCameraStatus_Available == IrCameraStatus)
    {
        if (!ProcessGamePad())
        {
            return false;
        }

        m_pModeStates[m_CurrentProcessor]->Update();

        {
            const std::vector<MenuItem>& rwMenu
                = m_pModeStates[m_NextProcessor]->GetReadWriteMenu();
            const std::vector<MenuItem>& rMenu
                = m_pModeStates[m_NextProcessor]->GetReadOnlyMenu();

            int selectIndex = GetMenuIndex();

            WriteProcessorState(pTextWriter, 360 + 300.0f * i, 420);

            DrawMenu(rwMenu, 695.0f + 300.0f * i, 400,
                selectIndex, pTextWriter);
            DrawMenu(rMenu, 695.0f + 300.0f * i, 300,
                -1, pTextWriter);

        }

        m_pModeStates[m_CurrentProcessor]->Render(
            &pGraphicsSystem->GetPrimitiveRenderer(),
            &pGraphicsSystem->GetCommandBuffer(),
            i);
        pGraphicsSystem->GetPrimitiveRenderer().SetColor(Color::White);
    }
    else
    {
        WriteIrCameraStatus(pTextWriter, IrCameraStatus, i);
    }

    return true;
}

bool IrSensorDemo::ProcessGamePad() NN_NOEXCEPT
{
    GetKeyState();

    if (IsButtonPressed<nn::hid::NpadButton::B>())
    {
        m_MenuSelection += 1;
    }
    if (IsButtonPressed<nn::hid::NpadButton::X>())
    {
        m_MenuSelection -= 1;
    }

    // メニューアクション
    if (IsButtonPressed<nn::hid::NpadButton::Y>() || IsButtonPressed<nn::hid::NpadButton::A>())
    {
        int8_t delta = IsButtonPressed<nn::hid::NpadButton::A>() ? 1 : -1;

        const std::vector<MenuItem>& menu
            = this->m_pModeStates[m_NextProcessor]->GetReadWriteMenu();
        const int MenuIndex = GetMenuIndex();
        if (MenuIndex >= 0)
        {
            menu[MenuIndex].Increment(delta);
        }
    }

    // イベントトリガー
    if (IsButtonPressed<nn::hid::NpadButton::R>())
    {
        m_pModeStates[m_CurrentProcessor]->Stop();
        m_pModeStates[m_NextProcessor]->Start();
        m_CurrentProcessor = m_NextProcessor;
    }

    return true;
}

int IrSensorDemo::GetMenuIndex() const NN_NOEXCEPT
{
    const std::vector<MenuItem>& Menu = this->m_pModeStates[m_NextProcessor]->GetReadWriteMenu();
    int size = static_cast<int>(Menu.size());
    int menuIndex = ((m_MenuSelection % size) + size) % size;

    NN_ASSERT(menuIndex >= 0);
    NN_ASSERT(menuIndex < size);

    return menuIndex;
}

// キーの状態を取得する
void IrSensorDemo::GetKeyState() NN_NOEXCEPT
{
    // 操作形態(NpadStyleSet)を取得
    nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(m_NpadId);

    NpadPluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
    if(pNpad == NULL)
    {
        return;
    }

    m_OldNpadButtonState = m_CurrentNpadButtonState;
    m_CurrentNpadButtonState = pNpad->GetNpadButtonState(m_NpadId);
}

// ボタンが押されたか判断する
template<typename BUTTON>
bool IrSensorDemo::IsButtonPressed() const NN_NOEXCEPT
{
    bool isButtonPressed = false;

    // 操作形態(NpadStyleSet)を取得
    nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(m_NpadId);

    NpadPluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
    if(pNpad == NULL)
    {
        return isButtonPressed;
    }

    if (m_CurrentNpadButtonState.buttons.Test<BUTTON>() &&
        !m_OldNpadButtonState.buttons.Test<BUTTON>())
    {
        isButtonPressed = true;
    }

    return isButtonPressed;
}
