﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/vi.h>

#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Rom.h>
#include <nn/init/init_Malloc.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include <nns/gfx/gfx_PrimitiveRenderer.h>

#include "Demo1HandAnalysisUtil.h"

class Float2
{
public:
    Float2() NN_NOEXCEPT {}
    Float2(float newX, float newY) NN_NOEXCEPT : m_X(newX), m_Y(newY) {}
    NN_IMPLICIT Float2(float val) NN_NOEXCEPT : m_X(val), m_Y(val) {}
    NN_IMPLICIT Float2(nn::util::Float2 val) NN_NOEXCEPT : m_X(val.x), m_Y(val.y) {}

    float GetX() const NN_NOEXCEPT
    {
        return m_X;
    }

    float GetY() const NN_NOEXCEPT
    {
        return m_Y;
    }

    void AddX(float x) NN_NOEXCEPT
    {
        m_X += x;
    }

    void AddY(float y) NN_NOEXCEPT
    {
        m_Y += y;
    }

    inline Float2 operator + (const Float2& value) const NN_NOEXCEPT
    {
        Float2 result;
        result.m_X = this->m_X + value.m_X;
        result.m_Y = this->m_Y + value.m_Y;
        return result;
    };
    inline Float2 operator - (const Float2& value) const NN_NOEXCEPT{
        Float2 result;
        result.m_X = this->m_X - value.m_X;
        result.m_Y = this->m_Y - value.m_Y;
        return result;
    };
    inline friend Float2 operator * (float scalar, const Float2& vector) NN_NOEXCEPT
    {
        Float2 result;
        result.m_X = vector.m_X * scalar;
        result.m_Y = vector.m_Y * scalar;
        return result;
    }
    Float2& operator = (nn::util::Float2& p) NN_NOEXCEPT
    {
        this->m_X = p.x;
        this->m_Y = p.y;
        return *this;
    }

private:
    float m_X;
    float m_Y;
};

class CompareAngles
{
public:
    NN_IMPLICIT CompareAngles(std::vector<Float2> directionsToCurrentPoint) NN_NOEXCEPT
        : m_DirectionsToCurrentPoint(directionsToCurrentPoint)
    {
    }
    bool operator()(size_t a, size_t b) NN_NOEXCEPT
    {
        const float Axby =
            m_DirectionsToCurrentPoint[a].GetX() * m_DirectionsToCurrentPoint[b].GetY();
        const float Aybx =
            m_DirectionsToCurrentPoint[a].GetY() * m_DirectionsToCurrentPoint[b].GetX();
        if(Aybx < Axby)
        {
            return true;
        }
        else if(Aybx > Axby)
        {
            return false;
        }
        return m_DirectionsToCurrentPoint[a].GetX() * m_DirectionsToCurrentPoint[a].GetX() +
            m_DirectionsToCurrentPoint[a].GetY() * m_DirectionsToCurrentPoint[a].GetY() <
        m_DirectionsToCurrentPoint[b].GetX() * m_DirectionsToCurrentPoint[b].GetX() +
            m_DirectionsToCurrentPoint[b].GetY() * m_DirectionsToCurrentPoint[b].GetY();
    }

private:
    std::vector<Float2> m_DirectionsToCurrentPoint;
};

void ComputePolygonProperties(
    PolygonProperties* pPolygonProperties,
    const nn::util::Float2* pPolygonPoints,
    size_t polygonPointCount) NN_NOEXCEPT
{
    nn::util::Float2 Zero2f = {};

    if (polygonPointCount < 3)
    {
        pPolygonProperties->perimeter = 0;
        pPolygonProperties->signedArea = 0;
        pPolygonProperties->areaCentroid = Zero2f;
        pPolygonProperties->perimeterCentroid = Zero2f;
        return;
    }
    float sumLength = 0.0f;
    float sumPreArea = 0.0f;
    Float2 sumBorderPreCentroid = Float2(0.0f);
    Float2 sumAreaPreCentroid = Float2(0.0f);
    const Float2 Center = Float2(0.0f);
    Float2 A = Float2(pPolygonPoints[polygonPointCount - 1]) - Center;
    const size_t Size = polygonPointCount - 1;
    for (size_t i = 0; i < Size; ++i)
    {
        const Float2 B = Float2(pPolygonPoints[i]) - Center;
        const Float2 AB = B - A;
        const float Length = sqrtf( AB.GetX() * AB.GetX() + AB.GetY() * AB.GetY());
        const float PreArea = A.GetX() * B.GetY() - A.GetY() * B.GetX();
        const Float2 AreaPreCentroid = PreArea * (A + B);
        sumLength += Length;
        sumPreArea += PreArea;
        sumBorderPreCentroid.AddX(Length * 0.5f * (A.GetX() + B.GetX()));
        sumBorderPreCentroid.AddY(Length * 0.5f * (A.GetY() + B.GetY()));
        sumAreaPreCentroid.AddX(AreaPreCentroid.GetX());
        sumAreaPreCentroid.AddY(AreaPreCentroid.GetY());
        A = B;
    }
    NN_ASSERT(sumLength > 0);
    pPolygonProperties->perimeter = sumLength;
    pPolygonProperties->perimeterCentroid.x = (1.f / sumLength) *
        sumBorderPreCentroid.GetX() + Center.GetX();

    pPolygonProperties->perimeterCentroid.y = (1.f / sumLength) *
        sumBorderPreCentroid.GetY() + Center.GetY();
    if (sumPreArea > 0)
    {
        pPolygonProperties->signedArea = 0.5f * sumPreArea;
        pPolygonProperties->areaCentroid.x = sumAreaPreCentroid.GetX() *
            (1.f / (3 * sumPreArea)) + Center.GetX();
        pPolygonProperties->areaCentroid.y = sumAreaPreCentroid.GetY() *
            (1.f / (3 * sumPreArea)) + Center.GetY();
    }
    else
    {
        pPolygonProperties->signedArea = 0.f;
        pPolygonProperties->areaCentroid = pPolygonProperties->perimeterCentroid;
    }
}

void ExtractConvexHull(
    std::vector<size_t>& convexHullIndices,
    const std::vector<nn::util::Float2>& oneTurnSilhouette ) NN_NOEXCEPT
{
    NN_ASSERT(convexHullIndices.empty());
    const size_t MaxPointCount = 2048;
    NN_ASSERT(oneTurnSilhouette.size() <= MaxPointCount);
    const size_t Size = std::min(oneTurnSilhouette.size(), MaxPointCount);

    if (Size < 4)
    {
        for (size_t i = 0; i < Size; ++i)
        {
            convexHullIndices.push_back(i);
        }
        return;
    }
    size_t firstIndex = std::numeric_limits<size_t>::max();
    Float2 firstPoint = Float2(std::numeric_limits<float>::max());
    for (size_t i = 0; i < Size; ++i)
    {
        const Float2 P = oneTurnSilhouette[i];
        if (P.GetX() < firstPoint.GetX() ||
            (P.GetX() == firstPoint.GetX() && P.GetY() <= firstPoint.GetY()))
        {
            firstPoint = P;
            firstIndex = i;
        }
    }
    std::vector<size_t> sortedIndices;
    {
        std::vector<Float2> directionsToCurrentPoint;
        for (size_t i = 0; i < Size; ++i)
        {
            if (firstIndex == i)
            {
                directionsToCurrentPoint.push_back(Float2(0.f));
                continue;
            }
            Float2 DirectionToCurrentPoint = Float2(oneTurnSilhouette[i]) - firstPoint;
            NN_ASSERT(0 <= DirectionToCurrentPoint.GetX());
            directionsToCurrentPoint.push_back(DirectionToCurrentPoint);
            sortedIndices.push_back(i);
        }

        CompareAngles compareAngles(directionsToCurrentPoint);
        std::sort(sortedIndices.begin(), sortedIndices.end(), compareAngles);
    }
    std::vector<size_t> tmpConvexHullIndices;
    tmpConvexHullIndices.push_back(firstIndex);
    size_t sortSize = sortedIndices.size();
    for (size_t i = 0; i < sortSize; ++i)
    {
        const size_t Index = sortedIndices[i];
        const Float2 Point = oneTurnSilhouette[Index];
        while (tmpConvexHullIndices.size() >= 2)
        {
            const size_t PrevIndex = tmpConvexHullIndices.back();
            const Float2 PrevPoint = oneTurnSilhouette[PrevIndex];
            const size_t M2Index = *(tmpConvexHullIndices.rbegin() + 1);
            const Float2 M2Point = oneTurnSilhouette[M2Index];
            const Float2 PrevSegment = PrevPoint - M2Point;
            const Float2 Segment = Point - PrevPoint;
            const float D = PrevSegment.GetX() * Segment.GetY() -
                PrevSegment.GetY() * Segment.GetX();
            if (D > 0)
            {
                break;
            }
            tmpConvexHullIndices.pop_back();
        }
        tmpConvexHullIndices.push_back(Index);
    }
    const size_t ConvexHullSize = tmpConvexHullIndices.size();
    for (size_t i = 0; i < ConvexHullSize; ++i)
    {
        convexHullIndices.push_back(tmpConvexHullIndices[i]);
    }
    convexHullIndices.push_back(tmpConvexHullIndices[0]);
}

