﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{CodecOpusMultiStreamDecoder.cpp,PageSampleCodecOpusMultiStreamDecoder}
 *
 * @brief
 * Opus マルチストリームデコーダのサンプルプログラム
 */

/**
 * @page PageSampleCodecOpusMultiStreamDecoder Opus マルチストリームデコーダ
 * @tableofcontents
 *
 * @brief
 * Opus マルチストリームデコーダのサンプルプログラムの解説です。
 *
 * @section PageSampleCodecOpusMultiStreamDecoder_SectionBrief 概要
 * Opus マルチストリームデコーダを用いて、Opus データをデコードしながらオーディオ再生を行うサンプルです。
 * オーディオ再生にはオーディオレンダラを利用します。
 *
 * @section PageSampleCodecOpusMultiStreamDecoder_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/CodecOpusMultiStreamDecoder
 * Samples/Sources/Applications/CodecOpusMultiStreamDecoder @endlink 以下にあります。
 *
 * @section PageSampleCodecOpusMultiStreamDecoder_SectionNecessaryEnvironment 必要な環境
 * オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleCodecOpusMultiStreamDecoder_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、一定の時間 BGM を再生し、自動的に終了します。
 *
 * @section PageSampleCodecOpusMultiStreamDecoder_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleCodecOpusMultiStreamDecoder_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleCodecOpusMultiStreamDecoder_SectionDetail 解説
 *
 * @subsection PageSampleCodecOpusMultiStreamDecoder_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  CodecOpusMultiStreamDecoder.cpp
 *  @includelineno CodecOpusMultiStreamDecoder.cpp
 *
 * @subsection PageSampleCodecOpusMultiStreamDecoder_SectionSampleDetail サンプルプログラムの解説
 *
 * このサンプルプログラムは、あらかじめエンコードされた Opus データをデコードしながら、オーディオ再生を行うものです。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - ファイルシステムの初期化
 * - Opus データの読み込み
 * - Opus マルチストリームデコーダの初期化
 * - オーディオレンダラの初期化、および開始
 * - Opus マルチストリームデコーダの実行と、デコード結果の再生
 * - オーディオレンダラの終了
 * - Opus マルチストリームデコーダの終了
 *
 * サンプルプログラムでは、デフォルトで Opus マルチストリームデコーダに nn::codec::OpusMultiStreamDecoder が使用されます。
 * USE_HARDWARE_DECODER マクロを定義してビルドすると nn::codec::HardwareOpusMultiStreamDecoder が使用されます。
 * 以下の説明は nn::codec::OpusMultiStreamDecoder を代表例としています。
 * nn::codec::HardwareOpusMultiStreamDecoder を使用する場合は、OpusMultiStreamDecoder を HardwareOpusMultiStreamDecoder に置き換えてお読み下さい。
 *
 * 最初にファイルシステムの初期化を行います。
 * リソースファイルはビルド時にコンバートされるものを利用します。
 * リソースディレクトリの場所は適宜変更して利用してください。
 *
 * Opus データをファイルから読み込みます。
 *
 * ヘッダ情報をもとに Opus マルチストリームデコーダを初期化します。
 *
 * 初期化関数 nn::codec::OpusMultiStreamDecoder::Initialize() にはサンプルレートとチャンネル数、さらにそれらによってサイズが決まるワークバッファを与える必要があります。
 *
 * ワークバッファのサイズは nn::codec::OpusMultiStreamDecoder::GetWorkBufferSize() により取得可能です。
 *
 * オーディオレンダラを準備します。
 * オーディオレンダラの詳細に関しては @link ../../../Samples/Sources/Applications/AudioRenderer
 * Samples/Sources/Applications/AudioRenderer @endlink サンプルおよび関連する API リファレンスを参照してください。
 *
 * メインループでは、ストリーム再生のような処理を行います。
 * 複数の PCM バッファを用意し、 nn::audio::VoiceType 経由で再生しながら、
 * 再生が終わったバッファに対して順次 Opus データのデコードを行います。
 *
 * Opus データをすべてデコードし、再生を終えると、オーディオレンダラを停止し、
 * オーディオレンダラおよび Opus マルチストリームデコーダを終了し、メモリの破棄を行います。
 */

#include <cstdlib>  // std::malloc, std::free
#include <new>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/audio.h>
#include <nn/codec.h>
#include <nn/fs.h>
#include <nn/mem.h>
#include <nn/os.h>

#include <nns/nns_Log.h>

//==============================================================================
// USE_HARDWARE_DECODER を有効にすると、HardwareOpusDecoder に切り替わります。
// #define USE_HARDWARE_DECODER
//==============================================================================

namespace {

char g_HeapBuffer[8 * 1024 * 1024];

uint8_t g_OpusData[1024 * 1024];

char* g_MountRomCacheBuffer = NULL;

const int BufferCount = 4;
const int OpusSampleCount = 960;
const int OpusChannelCount = 6;

NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t g_PcmBuffer[1024 * 1024];
NN_STATIC_ASSERT(BufferCount * OpusSampleCount * OpusChannelCount * sizeof(int16_t) <= sizeof(g_PcmBuffer));

void InitializeFileSystem()
{
    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

}

extern "C" void nnMain()
{
#ifdef USE_HARDWARE_DECODER
    NNS_LOG("CodecOpusDecoder sample (using nn::codec::HardwareOpusDecoder)\n");
#else  // USE_HARDWARE_DECODER
    NNS_LOG("CodecOpusDecoder sample (using nn::codec::OpusDecoder)\n");
#endif  // USE_HARDWARE_DECODER

    nn::mem::StandardAllocator allocator(g_HeapBuffer, sizeof(g_HeapBuffer));
    nn::mem::StandardAllocator pcmBufferAllocator(g_PcmBuffer, sizeof(g_PcmBuffer));

    // ファイルシステムを初期化します
    InitializeFileSystem();

    // Opus データを読み込みます
    const char Filename[] = "asset:/SampleBgm0-6ch.opus";
    nn::fs::FileHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, Filename, nn::fs::OpenMode_Read));
    int64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&size, handle));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, g_OpusData, static_cast<std::size_t>(size)));
    nn::fs::CloseFile(handle);

    // ヘッダより、サンプルレートとチャンネル数を取得します
    nn::codec::OpusBasicInfo opusBasicInfo;
    std::memcpy(&opusBasicInfo, g_OpusData, sizeof(opusBasicInfo));
    NN_ABORT_UNLESS(opusBasicInfo.header.type == nn::codec::OpusInfoType_BasicInfo);
    const auto sampleRate = opusBasicInfo.sampleRate;
    const auto channelCount = opusBasicInfo.channelCount;
    // バッファ確保のために宣言している OpusChannelCount と、*.opus のチャンネル数が同じ想定
    NN_ASSERT_EQUAL(channelCount, OpusChannelCount);

    // ヘッダより、マルチストリーム情報を取得します
    const auto pOpusMultiStreamInfo = reinterpret_cast<nn::codec::OpusMultiStreamInfo*>(g_OpusData + sizeof(opusBasicInfo));
    NN_ABORT_UNLESS(pOpusMultiStreamInfo->header.type == nn::codec::OpusInfoType_MultiStreamInfo);
    const auto totalStreamCount = pOpusMultiStreamInfo->totalStreamCount;
    const auto stereoStreamCount = pOpusMultiStreamInfo->stereoStreamCount;

    // Opus マルチストリームデコーダを初期化します
#ifndef USE_HARDWARE_DECODER
    nn::codec::OpusMultiStreamDecoder decoder;
#else  // #ifndef USE_HARDWARE_DECODER
    const int32_t noOption = 0; // 現在、指定可能なオプションはありません
    nn::codec::HardwareOpusMultiStreamDecoder decoder(noOption);
#endif // #ifndef USE_HARDWARE_DECODER
    void* workBuffer = std::malloc(decoder.GetWorkBufferSize(sampleRate, channelCount, totalStreamCount, stereoStreamCount));
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);
    nn::codec::OpusResult opusResult = decoder.Initialize(
        sampleRate,
        channelCount,
        totalStreamCount,
        stereoStreamCount,
        pOpusMultiStreamInfo->channelMapping,
        workBuffer,
        decoder.GetWorkBufferSize(sampleRate, channelCount, totalStreamCount, stereoStreamCount));
    NN_ABORT_UNLESS(opusResult == nn::codec::OpusResult_Success);

    //オーディオレンダラを構築します
    nn::audio::AudioRendererParameter audioRendererParameter;
    nn::audio::InitializeAudioRendererParameter(&audioRendererParameter);
    audioRendererParameter.sampleRate = 48000;
    audioRendererParameter.sampleCount = 240;
    audioRendererParameter.mixBufferCount = 6;
    audioRendererParameter.voiceCount = 6;
    audioRendererParameter.sinkCount = 1;
    NN_ABORT_UNLESS(nn::audio::IsValidAudioRendererParameter(audioRendererParameter));

    std::size_t audioRendererWorkBufferSize = nn::audio::GetAudioRendererWorkBufferSize(audioRendererParameter);
    void* audioRendererWorkBuffer = allocator.Allocate(audioRendererWorkBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(audioRendererWorkBuffer);

    nn::audio::AudioRendererHandle audioRendererHandle;
    nn::os::SystemEvent systemEvent;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&audioRendererHandle, &systemEvent, audioRendererParameter, audioRendererWorkBuffer, audioRendererWorkBufferSize));

    std::size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(audioRendererParameter);
    void* configBuffer = allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, audioRendererParameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 6);

    nn::audio::DeviceSinkType sink;
    int8_t bus[6] = { 0, 1, 2, 3, 4, 5 };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddDeviceSink(&config, &sink, &finalMix, bus, sizeof(bus) / sizeof(bus[0]), "MainAudioOut"));

    // WaveBuffer に追加するオーディオデータを保持するためのメモリプールを準備します。
    nn::audio::MemoryPoolType pcmBufferMemoryPool;
    NN_ABORT_UNLESS(nn::audio::AcquireMemoryPool(&config, &pcmBufferMemoryPool, g_PcmBuffer, sizeof(g_PcmBuffer)));
    NN_ABORT_UNLESS(nn::audio::RequestAttachMemoryPool(&pcmBufferMemoryPool));

    nn::audio::VoiceType voice;
    NN_ABORT_UNLESS(nn::audio::AcquireVoiceSlot(&config, &voice, sampleRate, channelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));

    nn::audio::SetVoiceDestination(&config, &voice, &finalMix);
    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);
    nn::audio::SetVoiceMixVolume(&voice, &finalMix, .1f, 0, bus[nn::audio::ChannelMapping_FrontLeft]);
    nn::audio::SetVoiceMixVolume(&voice, &finalMix, .1f, 1, bus[nn::audio::ChannelMapping_FrontRight]);
    nn::audio::SetVoiceMixVolume(&voice, &finalMix, .1f, 2, bus[nn::audio::ChannelMapping_FrontCenter]);
    nn::audio::SetVoiceMixVolume(&voice, &finalMix, .1f, 3, bus[nn::audio::ChannelMapping_LowFrequency]);
    nn::audio::SetVoiceMixVolume(&voice, &finalMix, .1f, 4, bus[nn::audio::ChannelMapping_RearLeft]);
    nn::audio::SetVoiceMixVolume(&voice, &finalMix, .1f, 5, bus[nn::audio::ChannelMapping_RearRight]);

    // ストリーム再生用のバッファを用意します
    // 最初は 0 で初期化したものを渡し、この後のメインループで Opus データのデコード結果を与えています
    nn::audio::WaveBuffer waveBuffer[BufferCount];
    void* pcmBuffer[BufferCount];
    for (int i = 0; i < BufferCount; ++i)
    {
        const std::size_t pcmBufferSize = OpusSampleCount * channelCount * sizeof(int16_t);
        pcmBuffer[i] = pcmBufferAllocator.Allocate(pcmBufferSize, nn::audio::BufferAlignSize);

        std::memset(pcmBuffer[i], 0, pcmBufferSize);

        waveBuffer[i].buffer = pcmBuffer[i];
        waveBuffer[i].size = pcmBufferSize;
        waveBuffer[i].startSampleOffset = 0;
        waveBuffer[i].endSampleOffset = static_cast<int32_t>(pcmBufferSize / sizeof(int16_t) / channelCount);
        waveBuffer[i].loop = false;
        waveBuffer[i].isEndOfStream = false;
        waveBuffer[i].pContext = nullptr;
        waveBuffer[i].contextSize = 0;

        nn::audio::AppendWaveBuffer(&voice, &waveBuffer[i]);
    }

    // オーディオレンダラを開始します
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioRenderer(audioRendererHandle));

    // Opus データ本体と、そのサイズを取得します
    const unsigned char* p = reinterpret_cast<unsigned char*>(g_OpusData + opusBasicInfo.dataInfoOffset);
    nn::codec::OpusInfoHeader dataInfoHeader;
    std::memcpy(&dataInfoHeader, p, sizeof(dataInfoHeader));
    NN_ABORT_UNLESS(dataInfoHeader.type == nn::codec::OpusInfoType_DataInfo);
    std::size_t remainedCount = dataInfoHeader.size;
    p += sizeof(dataInfoHeader);

    int bufferCount = BufferCount;
    for (;;)
    {
        systemEvent.Wait();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(audioRendererHandle, &config));

        // 再生が完了したバッファがあれば、Opus デコードを実行します
        if (auto pReleasedWaveBuffer = nn::audio::GetReleasedWaveBuffer(&voice))
        {
            nn::audio::WaveBuffer* pWaveBuffer = nullptr;
            int16_t* pPcmBuffer = nullptr;
            for (int i = 0; i < BufferCount; ++i)
            {
                if (pReleasedWaveBuffer == &waveBuffer[i])
                {
                    pWaveBuffer = &waveBuffer[i];
                    pPcmBuffer = static_cast<int16_t*>(pcmBuffer[i]);
                    break;
                }
            }

            // データが残っていればデコードを実行します
            if (remainedCount > 0)
            {
                std::size_t consumed;
                int sampleCount;
                opusResult = decoder.DecodeInterleaved(&consumed, &sampleCount, pPcmBuffer, OpusSampleCount * channelCount * sizeof(int16_t), p, remainedCount);
                NN_ABORT_UNLESS(opusResult == nn::codec::OpusResult_Success);
                p += consumed;
                remainedCount -= consumed;

                pWaveBuffer->size = sampleCount * sizeof(int16_t) * channelCount;
                pWaveBuffer->endSampleOffset = sampleCount;
                pWaveBuffer->loop = false;
                pWaveBuffer->isEndOfStream = false;

                nn::audio::AppendWaveBuffer(&voice, pWaveBuffer);
            }
            // データが残っていなければ、全てのバッファの再生が完了するまで待ちます
            else if (--bufferCount == 0)
            {
                break;
            }
        }
    }

    // 利用していたメモリプールにデタッチ要求をします。
    nn::audio::RequestDetachMemoryPool(&pcmBufferMemoryPool);
    // メモリプールがデタッチされるのを待ちます。
    while (nn::audio::IsMemoryPoolAttached(&pcmBufferMemoryPool))
    {
        nn::audio::GetReleasedWaveBuffer(&voice);

        systemEvent.Wait();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(audioRendererHandle, &config));
    }
    // MemoryPool を解放する。
    nn::audio::ReleaseMemoryPool(&config, &pcmBufferMemoryPool);
    // バッファを解放する。
    for (int i = 0; i < BufferCount; ++i)
    {
        pcmBufferAllocator.Free(pcmBuffer[i]);
    }

    // オーディオレンダラを終了し、メモリを破棄します
    nn::audio::StopAudioRenderer(audioRendererHandle);
    nn::audio::CloseAudioRenderer(audioRendererHandle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());
    allocator.Free(audioRendererWorkBuffer);
    allocator.Free(configBuffer);

    // Opus マルチストリームデコーダを終了し、メモリを破棄します
    decoder.Finalize();
    std::free(workBuffer);

    // ファイルシステムの終了処理を行います
    FinalizeFileSystem();

}  // NOLINT(readability/fn_size)
