﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @page BtmSystemSettings Bluetooth 機内モードの設定変更とペアリング情報の消去
    @tableofcontents

    @brief
    機内モードの設定変更とペアリング情報を消去する方法を示すサンプルプログラムの解説です。

    @section BtmSystemSettings_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。
    - 機内モードの ON/OFF 状態取得
    - 機内モードへの遷移
    - 機内モードの解除
    - ペアリング情報の削除
 */

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_TickTypes.h>
#include <nn/settings/system/settings_BluetoothDevices.h>

#include <nn/btm/btm.h>
#include <nn/btm/system/btm_SystemApi.h>

namespace
{
    const char ProgramName[] = "BtmSystemSettings Sample Application";
} // anonymous-namespace


extern "C" void nnMain()
{
    NN_LOG("%s Start.\n", ProgramName);

    // 機内モードの状態取得をします
    bool isRadioOn = nn::btm::system::IsRadioEnabled();
    NN_LOG("Bluetooth Airplane mode: %s\n", isRadioOn ? "OFF" : "ON");

    // 機内モードへの遷移を行います
    NN_LOG("@@@ Set Radio off (Enable Airplane mode)\n");
    nn::btm::system::EnableRadio(false);
    // 遷移の完了を待ちます
    for (int i = 0; i < 10; ++i)
    {
        isRadioOn = nn::btm::system::IsRadioEnabled();
        if (isRadioOn == false)
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
    NN_SDK_ASSERT(isRadioOn == false);
    NN_LOG("Bluetooth Airplane mode: %s\n", isRadioOn ? "OFF" : "ON");

    NN_LOG("@@@ SetRadio on (Disable Airplane mode)\n");
    nn::btm::system::EnableRadio(true);
    // 遷移の完了を待ちます
    for (int i = 0; i < 10; ++i)
    {
        isRadioOn = nn::btm::system::IsRadioEnabled();
        if (isRadioOn)
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
    NN_SDK_ASSERT(isRadioOn == true);
    NN_LOG("Bluetooth Airplane mode: %s\n", isRadioOn ? "OFF" : "ON");

    // ペアリング情報削除の動作確認のためにペアリング済みデバイスの数を取得しておきます
    auto count = nn::btm::system::GetPairedGamepadCount();
    NN_LOG("%d devices paired\n", count);

    nn::btm::system::ClearGamepadPairingDatabase();
    // 処理の完了を待ちます
    for (int i = 0; i < 20; ++i)
    {
        count = nn::btm::system::GetPairedGamepadCount();
        if (count == 0)
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
    NN_SDK_ASSERT_EQUAL(count, 0);

    NN_LOG("%s Done\n", ProgramName);
}

