﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "bluetooth_Client.h"
#include <cstdlib>

enum {DISCOVERY_INTERVAL_MS = 20000}; // 0 for continuous discovery

class myBtClient : public bluetoothClient
{
public:

    int  goalTsi = 9;
    uint8_t numPending = 0;
    nn::os::EventType numPendingEvent;
    nn::os::EventType disconnectEvent;

private:

    void EventFromDiscoveryStateChangedCallback( const nn::bluetooth::InfoFromDiscoveryStateChangedCallback* pInfo)
    {
        bluetoothClient::EventFromDiscoveryStateChangedCallback(pInfo);
        if (DISCOVERY_INTERVAL_MS == 0 && pInfo->state == nn::bluetooth::BT_DISCOVERY_STOPPED)
        {
            NN_LOG("  [StartDiscovery]\n");
            nn::bluetooth::StartDiscovery();
        }
    }

    void EventFromConnectionStateCallback(const nn::bluetooth::InfoFromConnectionStateCallback* pInfo)
    {
        bluetoothClient::EventFromConnectionStateCallback(pInfo);

        if(pInfo->state==nn::bluetooth::BTHH_CONN_STATE_CONNECTED)
        {
            setTsi(pInfo->bluetoothAddress, goalTsi);
        }

        if(pInfo->state==nn::bluetooth::BTHH_CONN_STATE_DISCONNECTED)
        {
            nn::os::SignalEvent(&disconnectEvent);
        }

    }

    void EventFromExtensionCallbacks(const nn::bluetooth::InfoFromExtensionCallbacks* pInfo)
    {
        bluetoothClient::EventFromExtensionCallbacks(pInfo);

        if (pInfo->eventType == nn::bluetooth::EventFromGetPendingConnectionsCallback)
        {
            numPending = pInfo->connections.numPending;
            nn::os::SignalEvent(&numPendingEvent);
        }
    }
};


static myBtClient client;



//-----------------------------------------------------------------------------
extern "C" void nnMain()
{
    NN_LOG("myBtClient NOW RUNNING \n");

    int tries;
    const int MAX_TRIES = 50;
    nn::Result result;

    if (client.startBluetooth() < 0)
    {
        NN_LOG("Couldn't start bluetooth!\n");
        exit(-1);
    }

    int64_t t1, t2;


    nn::os::InitializeEvent( &client.numPendingEvent, false, nn::os::EventClearMode_AutoClear );
    nn::os::InitializeEvent( &client.disconnectEvent, false, nn::os::EventClearMode_AutoClear );


    for (int iteration = 1; ; iteration++)
    {
        NN_LOG("\n-------------------------  TEST %d ----------------------------------\n", iteration);

        NN_LOG("[Allow connections for 3sec].");
        result = nn::bluetooth::ExtSetVisibility(false,true);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        NN_LOG(".");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        NN_LOG(".");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        NN_LOG("\n[Getting ready for sleep]\n");
        result = nn::bluetooth::ExtSetVisibility(false,false);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);



        // WAIT FOR NUM PENDING CONNECTIONS TO BE ZERO
        t1 = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
        for (tries = 0; tries < MAX_TRIES; tries++)
        {
            result = nn::bluetooth::ExtGetPendingConnections();
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            bool success = nn::os::TimedWaitEvent(&client.numPendingEvent, nn::TimeSpan::FromSeconds(1));
            if (!success) {
                NN_ABORT("  Failed to get numPendingEvent\n");
            }

            if (client.numPending == 0) {
                break; // finished properly
            }

            if (tries == MAX_TRIES - 1) {
                NN_ABORT("  Failed to finish with pending connections\n");
            }

            // wait a little before trying again
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

        }
        t2 = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
        NN_LOG("  Time for get zero GetPending = %5lld ms\n", t2 - t1);



        // DISCONNECT ANY CONNECTED CONTROLLERS
        t1 = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
        int disconnectCount = 0;
        nn::os::ClearEvent(&client.disconnectEvent);
        for (int ctrl=0; ctrl<bluetoothClient::MAX_CTRL; ctrl++)
        {
            if (client.controllerInfo[ctrl].connectState != nn::bluetooth::BTHH_CONN_STATE_DISCONNECTED
                    && client.controllerInfo[ctrl].valid)
            {
                result = nn::bluetooth::HidDisconnect(&client.controllerInfo[ctrl].bdAddr);
                disconnectCount++;

                if (!result.IsSuccess()) {
                    NN_LOG("HidDisconnect returned error\n");
                }
            }
        }
        // wait for disconnection events
        for (int i = 0; i<disconnectCount; i++)
        {
            bool success = nn::os::TimedWaitEvent(&client.disconnectEvent, nn::TimeSpan::FromMilliSeconds(10000));
            if (!success) {
                NN_ABORT("  Failed to get disconnectEvent\n");
            }
        }
        // check to ensure all are really disconnected
        for (int ctrl=0; ctrl<bluetoothClient::MAX_CTRL; ctrl++)
        {
            if (client.controllerInfo[ctrl].connectState != nn::bluetooth::BTHH_CONN_STATE_DISCONNECTED
                    && client.controllerInfo[ctrl].valid)
            {
                NN_LOG("  Not all disconnected! %d %d %02x:%02x %d\n", ctrl,
                        client.controllerInfo[ctrl].connectState,
                        client.controllerInfo[ctrl].bdAddr.address[4],
                        client.controllerInfo[ctrl].bdAddr.address[5],
                        client.controllerInfo[ctrl].valid);

            }
        }
        t2 = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
        NN_LOG("  Time to disconnect controllers = %5lld ms\n", t2 - t1);



        // ENTER AND EXIT LLR SCAN
        NN_LOG("[Starting and Stopping LLR scan]\n");
        result = nn::bluetooth::ExtStartLlrMode();
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        result = nn::bluetooth::ExtExitLlrMode();
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    NN_LOG("  [Exiting...]\n");
    client.finishBluetooth();
} // NOLINT(impl/function_size)



