﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/audio.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/nn_TimeSpan.h>
#include <nn/dd/dd_DeviceAddressSpace.h>
#include <nn/audio/audio_FinalOutputRecorderApi.h>
#include <nn/audio/audio_FinalOutputRecorderTypes.h>

namespace nn{
namespace audio{

namespace
{
char g_HeapBuffer[2048 * 2048];

void GenerateSquareWaveInt16(void* buffer, int channelCount, int sampleRate, int sampleCount, int amplitude)
{
    static int s_TotalSampleCount = 0;

    const int WAVE_SAMPLE_RATE = 440;
    const int WAVE_LENGTH = sampleRate / WAVE_SAMPLE_RATE;

    int16_t* buf = reinterpret_cast<int16_t*>(buffer);
    for (int sample = 0; sample < sampleCount; sample++)
    {
        int16_t value = static_cast<int16_t>(s_TotalSampleCount < (WAVE_LENGTH / 2) ? amplitude : -amplitude);
        for (int ch = 0; ch < channelCount; ch++)
        {
            buf[sample*channelCount + ch] = value;
        }
        s_TotalSampleCount++;
        if (s_TotalSampleCount == WAVE_LENGTH)
        {
            s_TotalSampleCount = 0;
        }
    }
}
}

extern "C" void nnMain()
{
    nn::mem::StandardAllocator allocator(g_HeapBuffer, sizeof(g_HeapBuffer));

    nn::audio::AudioOut audioOut;
    nn::audio::FinalOutputRecorder gameRecord;

    nn::os::SystemEvent audioOutBufferEvent;
    nn::os::SystemEvent gameRecordBufferEvent;
    nn::audio::AudioOutParameter paramAudioOut { 0 };
    nn::audio::FinalOutputRecorderParameter paramGameRecord { 0 };
    nn::audio::InitializeAudioOutParameter(&paramAudioOut);
    nn::audio::InitializeFinalOutputRecorderParameter(&paramGameRecord);
    NN_LOG("Opening AudioOut\n");
    if (nn::audio::OpenDefaultAudioOut(&audioOut, &audioOutBufferEvent, paramAudioOut).IsFailure())
    {
        paramAudioOut.sampleRate = 0;
        NN_ABORT_UNLESS(
            nn::audio::OpenDefaultAudioOut(&audioOut, &audioOutBufferEvent, paramAudioOut).IsSuccess(),
            "Failed to open AudioOut."
        );
    }

    NN_LOG("Opening GameRecord\n");
    if (nn::audio::OpenFinalOutputRecorder(&gameRecord, &gameRecordBufferEvent, paramGameRecord).IsFailure())
    {
        paramGameRecord.sampleRate = 0;
        NN_ABORT_UNLESS(
            nn::audio::OpenFinalOutputRecorder(&gameRecord, &gameRecordBufferEvent, paramGameRecord).IsSuccess(),
            "Failed to open GameRecord."
         );
     }

    int channelCount = nn::audio::GetAudioOutChannelCount(&audioOut);
    int sampleRate = nn::audio::GetAudioOutSampleRate(&audioOut);
    nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioOutSampleFormat(&audioOut);
    const int frameRate = 20;
    const int frameSampleCount = sampleRate / frameRate;
    const size_t dataSize = frameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat);
    const size_t bufferSize = nn::util::align_up(dataSize, nn::audio::FinalOutputRecorderBuffer::SizeGranularity);

    const int outBufferCount = 50;
    const int processingBufferCount = 16;
    const int outRecordFromBufferCount = 2;
    const int amplitude = std::numeric_limits<int16_t>::max() / 16;
    nn::audio::FinalOutputRecorderBuffer audioInBuffer[processingBufferCount];
    nn::audio::AudioOutBuffer audioOutBuffer[outBufferCount];
    nn::audio::AudioOutBuffer audioOutRecFromBuffer[outRecordFromBufferCount];
    void* inBuffer[processingBufferCount];
    void* outBuffer[outBufferCount];
    void* outRecBuffer[outRecordFromBufferCount];
    for (int i = 0; i < outBufferCount; ++i) //all out buffers set to 0
    {
        outBuffer[i] = allocator.Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
        NN_ASSERT(outBuffer[i]);
        memset(outBuffer[i], 0, bufferSize);
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], outBuffer[i], bufferSize, dataSize);
    }

    int currentBuffer = 0;
    for(int i = 0; i < processingBufferCount; ++i)
    {
        inBuffer[i] = allocator.Allocate(bufferSize, nn::audio::FinalOutputRecorderBuffer::AddressAlignment);
        NN_ASSERT(inBuffer[i]);

        nn::audio::SetFinalOutputRecorderBufferInfo(&audioInBuffer[i], inBuffer[i], bufferSize, dataSize);

        outBuffer[i] = allocator.Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
        outRecBuffer[i] = allocator.Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
        NN_ASSERT(outRecBuffer[i]);
        GenerateSquareWaveInt16(outRecBuffer[i], channelCount, sampleRate, frameSampleCount, amplitude);
        nn::audio::SetAudioOutBufferInfo(&audioOutRecFromBuffer[i], outRecBuffer[i], bufferSize, dataSize);
        nn::audio::AppendFinalOutputRecorderBuffer(&gameRecord, &audioInBuffer[i]);
        nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutRecFromBuffer[i]);
        ++currentBuffer;
    }

    NN_ABORT_UNLESS(
        nn::audio::StartFinalOutputRecorder(&gameRecord).IsSuccess(),
        "Failed to start recording."
    );

    NN_ABORT_UNLESS(
        nn::audio::StartAudioOut(&audioOut).IsSuccess(),
        "Failed to start playback."
    );

    NN_LOG("Game Record Start\n");

    for(int i = currentBuffer; i < outBufferCount; ++i)
    {
        gameRecordBufferEvent.Wait();
        nn::audio::FinalOutputRecorderBuffer* pAudioInBuffer = nullptr;
        pAudioInBuffer = GetReleasedFinalOutputRecorderBuffer(&gameRecord);

        audioOutBufferEvent.Wait();
        nn::audio::AudioOutBuffer* pAudioOutBuffer = &audioOutBuffer[currentBuffer - 2];
        nn::audio::AudioOutBuffer* pReleasedAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);

        void* pInBuffer = nn::audio::GetFinalOutputRecorderBufferDataPointer(pAudioInBuffer);
        size_t inSize = nn::audio::GetFinalOutputRecorderBufferDataSize(pAudioInBuffer);
        void* pOutBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
        size_t outSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);
        NN_ASSERT(inSize == outSize);
        std::memcpy(pOutBuffer, pInBuffer, inSize);

        void* pReleasedOutBuffer = nn::audio::GetAudioOutBufferDataPointer(pReleasedAudioOutBuffer);
        size_t releasedSize = outSize;
        NN_ASSERT(releasedSize == frameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat));
        GenerateSquareWaveInt16(pReleasedOutBuffer, channelCount, sampleRate, frameSampleCount, amplitude);
        nn::audio::AppendFinalOutputRecorderBuffer(&gameRecord, pAudioInBuffer);
        nn::audio::AppendAudioOutBuffer(&audioOut, pReleasedAudioOutBuffer);
        ++currentBuffer;
    }
    //wait 1 second
    const int pause = 1;
    NN_LOG("Sleeping\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(pause));
    NN_LOG("Replaying.\n");
    int playbackBuffer = 0; //first two buffers are used for recording, and will cause distortion if played
    nn::audio::AudioOutBuffer* pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
    while (pAudioOutBuffer)
    {
        nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[playbackBuffer++]);
        pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
    }
    for(int i = playbackBuffer; i < currentBuffer; ++i)
    {
        audioOutBufferEvent.Wait();
        nn::audio::AudioOutBuffer* pAudioOutBuffer = nullptr;
        pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
        nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
    }

    NN_LOG("Stop game record\n");
    nn::audio::StopAudioOut(&audioOut);
    nn::audio::StopFinalOutputRecorder(&gameRecord);
    nn::audio::CloseAudioOut(&audioOut);
    nn::audio::CloseFinalOutputRecorder(&gameRecord);
    for (int i = 0; i < outBufferCount; ++i)
    {
        allocator.Free(outBuffer[i]);
    }
    for(int i = 0; i < processingBufferCount; ++i)
    {
        allocator.Free(inBuffer[i]);
    }
    for(int i = 0; i < outRecordFromBufferCount; ++i)
    {
        allocator.Free(outRecBuffer[i]);
    }
    return;
}//NOLINT(impl/function_size)
}
}

