﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AtkSimple.cpp,PageSampleAtkSimple}
 *
 * @brief
 *  nn::atk ライブラリの基本機能の使い方を示すサンプルプログラム
 */

/**
 * @page PageSampleAtkSimple シンプルな再生
 * @tableofcontents
 *
 * @brief
 *  nn::atk ライブラリの基本機能のサンプルプログラムの解説です。
 *
 * @section PageSampleAtkSimple_SectionBrief 概要
 *  Atk ライブラリを使ってサウンドの再生を行うためのもっともシンプルなサンプルです。
 *
 * @section PageSampleAtkSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/AtkSimple Samples/Sources/Applications/AtkSimple @endlink 以下にあります。
 *
 * @section PageSampleAtkSimple_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleAtkSimple_SectionHowToOperate 操作方法
 *  コンソールに操作方法が表示されます。
 *
 * @section PageSampleAtkSimple_SectionPrecaution 注意事項
 *  実行中のログが、画面上とコンソールに表示され、
 *  起動直後のログとして操作方法が表示されます。
 *
 *  PC 版ではウィンドウは立ち上がらず、コンソール表示のみになる点にご注意ください。
 *
 * @section PageSampleAtkSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAtkSimple_SectionDetail 解説
 *
 * @subsection PageSampleAtkSimple_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  AtkSimple.cpp
 *  @includelineno AtkSimple.cpp
 *
 * @subsection PageSampleAtkSimple_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの Atk に関連している処理の流れは以下の通りです。
 *
 *  - アロケータの初期化
 *    - 内部で２種類（メモリプール管理対象用とそれ以外）のアロケータを初期化。
 *  - Atk ライブラリの初期化 (InitializeAtk())
 *    - SoundSystem の初期化
 *    - SoundHeap の初期化
 *    - SoundArchive をオープンし、パラメータ情報をロード
 *    - SoundDataManager の初期化
 *    - SoundArchivePlayer で用いるストリームバッファをメモリプール管理対象用アロケータから確保し初期化
 *    - メモリプール管理対象のメモリをメモリプールにアタッチ
 *    - SoundArchivePlayer の初期化
 *  - SoundHeap にウェーブサウンドとシーケンスサウンドの再生に必要な情報をロード (LoadData())
 *  - メインループでの処理
 *    - キー操作があれば再生・停止処理を実行
 *    - SoundArchivePlayer の更新
 *  - Atk ライブラリの終了処理 (FinalizeAtk())
 *    - SoundArchivePlayer の 終了処理
 *    - メモリプール管理対象のメモリをメモリプールからデタッチ
 *    - その他の終了処理・メモリ解放
 *  - アロケータの終了処理
 *
 *
 *  キー操作を行うことで、ウェーブサウンド、シーケンスサウンド、ストリームサウンドを再生します。
 *  このサンプルプログラムでは再生用のAPIとして、StartSound と HoldSound を使用しています。
 *  StartSoundは一度実行すると停止処理を行うか、サウンドの再生が終わるまで再生を続けます。
 *  HoldSoundは HoldSound を呼び続けている間だけ再生します。
 *
 *  メモリプール管理対象のメモリのアタッチ・デタッチに関しては以下の点にご注意ください。
 *  - メモリプールのアタッチ・デタッチは即座に反映されないため、 SoundSystem が初期化された状態で、
 *    アタッチ・デタッチのリクエストを送り、しばらく待つ必要がある
 *  - 再生中にはメモリプールをデタッチできないため、明示的に停止させるか SoundArchivePlayer::Finalize()
 *    を実行したあとにメモリプールをデタッチする必要がある
 */

#include "Common.fsid"

#include <nn/atk.h>

#include <nns/atk/atk_SampleCommon.h>

#include <nns/nns_Log.h>

namespace
{
    const char Title[] = "AtkSimple";
    const char ArchiveRelativePath[] = "Common.bfsar";

    const int SoundHeapSize = 4 * 1024 * 1024;

    nn::atk::SoundHeap          g_SoundHeap;
    nn::atk::FsSoundArchive     g_SoundArchive;
    nn::atk::SoundArchivePlayer g_SoundArchivePlayer;
    nn::atk::SoundDataManager   g_SoundDataManager;

    nn::audio::MemoryPoolType   g_MemoryPool;

    void* g_pMemoryForSoundSystem;
    void* g_pMemoryForSoundHeap;
    void* g_pMemoryForInfoBlock;
    void* g_pMemoryForSoundDataManager;
    void* g_pMemoryForSoundArchivePlayer;
    void* g_pMemoryForStreamBuffer;

    nn::atk::SoundHandle        g_SoundHandle;
    nn::atk::SoundHandle        g_SoundHandleHold;
}

void Initialize()
{
    nns::atk::InitializeHeap();
    nns::atk::InitializeFileSystem();
    nns::atk::InitializeHidDevices();
}

void Finalize()
{
    nns::atk::FinalizeHidDevices();
    nns::atk::FinalizeFileSystem();
    nns::atk::FinalizeHeap();
}

void InitializeAtk()
{
    bool isSuccess = true;

    // SoundSystem の初期化
    nn::atk::SoundSystem::SoundSystemParam param;
    std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );
    g_pMemoryForSoundSystem = nns::atk::Allocate( memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );
    isSuccess = nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>( g_pMemoryForSoundSystem ),
        memSizeForSoundSystem );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundSystem" );

    // SoundHeap の初期化
    g_pMemoryForSoundHeap = nns::atk::Allocate( SoundHeapSize );
    isSuccess = g_SoundHeap.Create( g_pMemoryForSoundHeap, SoundHeapSize );
    NN_ABORT_UNLESS( isSuccess, "cannot create SoundHeap" );

    // SoundArchive の初期化
    const char* archiveAbsolutePath = nns::atk::GetAbsolutePath(ArchiveRelativePath);
    isSuccess = g_SoundArchive.Open(archiveAbsolutePath);
    NN_ABORT_UNLESS( isSuccess, "cannot open SoundArchive(%s)\n", archiveAbsolutePath );

    // SoundArchive のパラメータ情報をメモリにロード
    std::size_t infoBlockSize = g_SoundArchive.GetHeaderSize();
    g_pMemoryForInfoBlock = nns::atk::Allocate(  infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
    isSuccess = g_SoundArchive.LoadHeader( g_pMemoryForInfoBlock, infoBlockSize );
    NN_ABORT_UNLESS( isSuccess, "cannot load InfoBlock" );

    // SoundDataManager の初期化
    std::size_t memSizeForSoundDataManager = g_SoundDataManager.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundDataManager = nns::atk::Allocate( memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize );
    isSuccess = g_SoundDataManager.Initialize(
        &g_SoundArchive,
        g_pMemoryForSoundDataManager,
        memSizeForSoundDataManager );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundDataManager" );

    // SoundArchivePlayer で用いるストリームバッファの初期化
    // ストリームバッファはメモリプール管理されているヒープから確保する必要があります。
    std::size_t memSizeForStreamBuffer = g_SoundArchivePlayer.GetRequiredStreamBufferSize( &g_SoundArchive );
    g_pMemoryForStreamBuffer = nns::atk::AllocateForMemoryPool(memSizeForStreamBuffer);

    // 専用のヒープをメモリプールにアタッチ
    nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPool, nns::atk::GetPoolHeapAddress(), nns::atk::GetPoolHeapSize());

    // SoundArchivePlayer の初期化
    std::size_t memSizeForSoundArchivePlayer = g_SoundArchivePlayer.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundArchivePlayer = nns::atk::Allocate( memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize);
    isSuccess = g_SoundArchivePlayer.Initialize(
        &g_SoundArchive,
        &g_SoundDataManager,
        g_pMemoryForSoundArchivePlayer, memSizeForSoundArchivePlayer,
        g_pMemoryForStreamBuffer, memSizeForStreamBuffer );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundArchivePlayer" );
}

void FinalizeAtk()
{
    g_SoundArchivePlayer.Finalize();

    // 専用のヒープをメモリプールからデタッチ
    nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPool);

    g_SoundDataManager.Finalize();
    g_SoundArchive.Close();
    g_SoundHeap.Destroy();
    nn::atk::SoundSystem::Finalize();

    nns::atk::FreeForMemoryPool(g_pMemoryForStreamBuffer);
    nns::atk::Free(g_pMemoryForSoundArchivePlayer);
    nns::atk::Free(g_pMemoryForSoundDataManager);
    nns::atk::Free(g_pMemoryForInfoBlock);
    nns::atk::Free(g_pMemoryForSoundHeap);
    nns::atk::Free(g_pMemoryForSoundSystem);
}

void LoadData()
{
    bool isSuccess = true;

    isSuccess = g_SoundDataManager.LoadData( SEQ_MARIOKART, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SEQ_MARIOKART) failed." );

    isSuccess = g_SoundDataManager.LoadData( SE_YOSHI, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_YOSHI) failed." );
}

void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName)
{
    g_SoundHandle.Stop( 0 );

    bool result = g_SoundArchivePlayer.StartSound( &g_SoundHandle, soundId ).IsSuccess();
    NNS_LOG("StartSound(%s) ... (%d)\n", debugLabelName, result);
}

void PlayWithHoldSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName)
{
    bool result = g_SoundArchivePlayer.HoldSound( &g_SoundHandleHold, soundId ).IsSuccess();
    NNS_LOG("HoldSound(%s) ... (%d)\n", debugLabelName, result);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("[A]              StartSound SEQ  (SEQ_MARIOKART)\n");
    NNS_LOG("[X]              StartSound WSD  (SE_YOSHI)\n");
    NNS_LOG("[Y]              StartSound STRM (STRM_MARIOKART)\n");
    NNS_LOG("[B]              Stop Sound\n");
    NNS_LOG("[RIGHT]          HoldSound SEQ  (SEQ_MARIOKART)\n");
    NNS_LOG("[UP]             HoldSound WSD  (SE_YOSHI)\n");
    NNS_LOG("[LEFT]           HoldSound STRM (STRM_MARIOKART)\n");
    NNS_LOG("[L]              Print Usage\n");
    NNS_LOG("[+/Start][Space] Exit Application\n");
    NNS_LOG("--------------------------------------------------\n");
}

bool UpdateAtk()
{
    // StartSound / Stop
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >() )
    {
        PlayWithStartSound(SEQ_MARIOKART, "SEQ_MARIOKART");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >() )
    {
        PlayWithStartSound(SE_YOSHI, "SE_YOSHI");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >() )
    {
        PlayWithStartSound(STRM_MARIOKART, "STRM_MARIOKART");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B >() )
    {
        g_SoundHandle.Stop( 0 );
    }

    // HoldSound
    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Right >() )
    {
        PlayWithHoldSound(SEQ_MARIOKART, "SEQ_MARIOKART");
    }
    else if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Up >() )
    {
        PlayWithHoldSound(SE_YOSHI, "SE_YOSHI");
    }
    else if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Left >() )
    {
        PlayWithHoldSound(STRM_MARIOKART, "STRM_MARIOKART");
    }

    // Print Usage
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::L >() )
    {
        PrintUsage();
    }

    // Exit
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Start >() )
    {
        return false;
    }

    g_SoundArchivePlayer.Update();

    return true;
}

extern "C" void nnMain()
{
    Initialize();
    InitializeAtk();

    LoadData();

    PrintUsage();

    for ( ;; )
    {
        nns::atk::UpdateHidDevices();

        if ( !UpdateAtk() )
        {
            break;
        }

        // Vsync の代わり
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    FinalizeAtk();
    Finalize();
}

