﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if  defined( NN_ATK_ENABLE_GFX_VIEWING )

#include <cstdlib>
#include <nn/mem/mem_StandardAllocator.h>
#include <nv/nv_MemoryManagement.h>
#include <nvn/nvn_FuncPtrInline.h>
#include "GfxContext.h"
#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <Windows.h>
#undef      max
#endif

namespace
{
    //  描画用のバッファをいくつ持つか
    const int MultiBufferingCount = 2;

    //  LineMesh の初期化
    void InitializeLineMesh(nn::gfx::Device* pDevice, nns::gfx::PrimitiveRenderer::PrimitiveMesh* pLineMesh, int lineMeshCount, nns::gfx::GpuBuffer* pLineMeshBuffer, nn::gfx::MemoryPool* pMemoryPool, nn::mem::StandardAllocator* pAllocator, int vertexCount) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL( pDevice );
        NN_ABORT_UNLESS_NOT_NULL( pLineMesh );
        NN_ABORT_UNLESS_NOT_NULL( pLineMeshBuffer );
        NN_ABORT_UNLESS_NOT_NULL( pMemoryPool );
        NN_ABORT_UNLESS_NOT_NULL( pAllocator );

        //  GpuBuffer とその MemoryPool を用意します
        const size_t bufferSizePerLineMeshVertex =
            sizeof(uint32_t) +          //  idx
            sizeof(nn::util::Float3) +  //  Pos
            sizeof(nn::util::Float4) +  //  Color
            sizeof(nn::util::Float2) +  //  Uv
            sizeof(nn::util::Float3);   //  Normal
        const size_t bufferSizePerLineMesh = bufferSizePerLineMeshVertex * vertexCount;
        nns::gfx::GpuBuffer::InitializeArg bufferArg;
        bufferArg.SetBufferCount( lineMeshCount );
        bufferArg.SetBufferSize( bufferSizePerLineMesh );
        bufferArg.SetGpuAccessFlag( nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer );
        const size_t bufferAlign = nns::gfx::GpuBuffer::GetGpuBufferAlignement( pDevice, bufferArg );

        nn::gfx::MemoryPool::InfoType poolInfo;
        poolInfo.SetDefault();
        poolInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );
        const size_t poolAlign = nn::gfx::MemoryPool::GetPoolMemoryAlignment( pDevice, poolInfo );
        const size_t poolGranularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( pDevice, poolInfo );

        const size_t align = std::max( bufferAlign, poolAlign );
        const size_t memSize = nn::util::align_up( lineMeshCount * bufferSizePerLineMesh, poolGranularity );
        void* pMemory = pAllocator->Allocate( memSize, align );
        NN_ABORT_UNLESS_NOT_NULL( pMemory );

        poolInfo.SetPoolMemory( pMemory, memSize );
        pMemoryPool->Initialize( pDevice, poolInfo );
        pLineMeshBuffer->Initialize( pDevice, bufferArg, pMemoryPool, 0 );

        //  PrimitiveMesh の初期化
        for( int line = 0; line < lineMeshCount; line++ )
        {
            pLineMeshBuffer->Map( line );
            const bool isSuccess = pLineMesh[line].Initialize( pLineMeshBuffer, vertexCount, vertexCount );
            NN_UNUSED( isSuccess );
            NN_SDK_ASSERT( isSuccess );

            uint32_t* pIndex = pLineMesh[line].GetIndexBufferCpuAddress();
            for( int i = 0; i< vertexCount; i++ )
            {
                pIndex[i] = i;
            }

            nn::util::Float3* pPos = static_cast<nn::util::Float3*>( pLineMesh[line].GetVertexBufferCpuAddress( nns::gfx::PrimitiveRenderer::VertexAttribute_Pos ) );
            nn::util::Float2* pUv = static_cast<nn::util::Float2*>( pLineMesh[line].GetVertexBufferCpuAddress( nns::gfx::PrimitiveRenderer::VertexAttribute_Uv ) );
            nn::util::Float4* pColor = static_cast<nn::util::Float4*>( pLineMesh[line].GetVertexBufferCpuAddress( nns::gfx::PrimitiveRenderer::VertexAttribute_Color ) );

            for( int i = 0; i< vertexCount; i++ )
            {
                pPos[i].x = 0.0f;
                pPos[i].y = 0.0f;
                pPos[i].z = 0.0f;

                pUv[i].x = 0.0f;
                pUv[i].y = 0.0f;

                pColor[i].x = 1.0f;
                pColor[i].y = 1.0f;
                pColor[i].z = 1.0f;
                pColor[i].w = 1.0f;
            }

            pLineMeshBuffer->Unmap();
        }
    }
    //  変換行列の初期化
    void InitializeMatrix(nn::util::Matrix4x3fType* pViewMatrix, nn::util::Matrix4x4fType* pProjectionMatrix, nn::util::Matrix4x3fType* pModelMatrix, int displayWidth, int displayHeight) NN_NOEXCEPT
    {
        //  View 行列には左上原点で X 軸の正方向が右, Y 軸の正方向が下になるように設定
        const float halfWidth = 0.5f * displayWidth;
        const float halfHeight = 0.5f * displayHeight;
        nn::util::MatrixSet( pViewMatrix,
            1,  0, 0,
            0, -1, 0,
            0,  0, 1,
            -halfWidth, halfHeight, 0
        );

        //  Projection 行列には幅 DisplayWidth, 高さ DisplayHeight になるように設定
        const float nearZ = 0.0f;
        const float farZ = 32.0f;
        nn::util::MatrixOrthographicOffCenterRightHanded( pProjectionMatrix, -halfWidth, halfWidth, -halfHeight, halfHeight, nearZ, farZ );

        //  Model 行列は単位行列
        nn::util::MatrixIdentity( pModelMatrix );
    }
    //  uint32_t で指定された色から Uint8x4 の色に変換します
    nn::util::Uint8x4 TransColor(uint32_t color) NN_NOEXCEPT
    {
        nn::util::Uint8x4 result;
        result.v[0] = ( color >> 16 ) & 0xff;
        result.v[1] = ( color >> 8 ) & 0xff;
        result.v[2] = ( color >> 0 ) & 0xff;

        const uint8_t alpha = ( color >> 24 ) & 0xff;
        if( alpha == 0 )
        {
            //  α 値が 0 のときは特別に 255 として扱います
            result.v[3] = 255;
        }
        else
        {
            result.v[3] = alpha;
        }
        return result;
    }
}

NN_DEFINE_STATIC_CONSTANT( const int GfxContext::DrawalbeLinkedPointCountMax );
NN_DEFINE_STATIC_CONSTANT( const int GfxContext::DrawLinkedPointFunctionCallCount );

//  初期化します
void GfxContext::Initialize(void* pMemory, size_t memorySize, int displayWidth, int displayHeight) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( pMemory );
    m_Allocator.Initialize( pMemory, memorySize );

    //  GraphicsFramework の初期化
    {
        //  GraphicsFramework が使用するメモリ領域は現時点ではデフォルトのアロケータから確保されるようです。
        //  将来的にデフォルトのアロケータからメモリを確保することが問題になるようでしたら、
        //  初期化の引数に関数ポインタを渡すことでアロケータを設定できます。
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nns::gfx::GraphicsFramework::InitializeGraphicsSystem( GraphicsSystemMemorySize );

        nns::gfx::GraphicsFramework::FrameworkInfo info;
        info.SetDefault();
        info.SetBufferCount( MultiBufferingCount );
        m_GraphicsFramework.Initialize( info );
    }

    //  プリミティブレンダラの作成
    auto pDevice = m_GraphicsFramework.GetDevice();
    {
        const int DrawCallCount = 768;
        const int ViewFunctionCallCount = 768;

        nns::gfx::PrimitiveRenderer::RendererInfo info;
        info.SetDefault();
        info.SetDrawCallCountMax( DrawCallCount );
        info.SetViewFunctionCallCountMax( ViewFunctionCallCount );
        info.SetMultiBufferQuantity( MultiBufferingCount );
        info.SetAllocator( AllocateForPrimitiveRenderer, this );

        m_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer( pDevice, info );
        NN_ABORT_UNLESS_NOT_NULL( m_pPrimitiveRenderer );
        m_pPrimitiveRenderer->SetScreenWidth( displayWidth );
        m_pPrimitiveRenderer->SetScreenHeight( displayHeight );
    }

    //  LineMesh の初期化
    InitializeLineMesh( pDevice, m_LineMesh, DrawLinkedPointFunctionCallCount, &m_LineMeshBuffer, &m_LineMeshMemoryPool, &m_Allocator, DrawalbeLinkedPointCountMax );

    //  変換行列の初期化
    InitializeMatrix( &m_ViewMatrix, &m_ProjectionMatrix, &m_ModelMatrix, displayWidth, displayHeight );

    //  フォントレンダラの初期化
    {
        const size_t memSize = m_FontRenderer.GetRequiredMemorySize( pDevice );
        void* pFontMemory = m_Allocator.Allocate( memSize );
        NN_ABORT_UNLESS_NOT_NULL( pFontMemory );

        const int samplerSlot = m_GraphicsFramework.AllocateDescriptorSlot( nn::gfx::DescriptorPoolType_Sampler, 1 );
        const int textureSlot = m_GraphicsFramework.AllocateDescriptorSlot( nn::gfx::DescriptorPoolType_TextureView, 1 );
        auto pSamplerPool = m_GraphicsFramework.GetDescriptorPool( nn::gfx::DescriptorPoolType_Sampler );
        auto pTexturePool = m_GraphicsFramework.GetDescriptorPool( nn::gfx::DescriptorPoolType_TextureView );
        m_FontRenderer.Initialize( pDevice, pFontMemory, memSize, displayWidth, displayHeight, pSamplerPool, samplerSlot, pTexturePool, textureSlot );
    }

    m_CurrentBufferIndex = 0;
}
//  終了処理をします
void GfxContext::Finalize() NN_NOEXCEPT
{
    auto pDevice = m_GraphicsFramework.GetDevice();

    m_FontRenderer.Finalize();
    m_LineMeshBuffer.Finalize( pDevice );
    m_LineMeshMemoryPool.Finalize( pDevice );
    nns::gfx::PrimitiveRenderer::DestroyRenderer( m_pPrimitiveRenderer, pDevice, FreeForPrimitiveRenderer, this );
    m_GraphicsFramework.Finalize();
}

//  描画を開始します
void GfxContext::Begin() NN_NOEXCEPT
{
    //  現在のバッファを次へ移します
    m_CurrentBufferIndex++;
    if( m_CurrentBufferIndex >= MultiBufferingCount )
    {
        m_CurrentBufferIndex = 0;
    }

    // テクスチャ取得と vsync 待ち
    m_GraphicsFramework.AcquireTexture( m_CurrentBufferIndex );
    m_GraphicsFramework.WaitDisplaySync( m_CurrentBufferIndex, nn::TimeSpan::FromSeconds(2) );

    //  設定を反映
    m_GraphicsFramework.BeginFrame( m_CurrentBufferIndex );
    auto pCommandBuffer = m_GraphicsFramework.GetRootCommandBuffer( m_CurrentBufferIndex );
    auto pTargetView = m_GraphicsFramework.GetColorTargetView();

    NN_ABORT_UNLESS_NOT_NULL( pCommandBuffer );
    pCommandBuffer->ClearColor( pTargetView, 0.25f, 0.25f, 0.25f, 1.0f, nullptr );
    pCommandBuffer->SetRenderTargets( 1, &pTargetView, nullptr );
    pCommandBuffer->SetViewportScissorState( m_GraphicsFramework.GetViewportScissorState() );

    //  PrimitiveRenderer を準備
    m_pPrimitiveRenderer->Update( m_CurrentBufferIndex );
    m_pPrimitiveRenderer->SetViewMatrix( &m_ViewMatrix );
    m_pPrimitiveRenderer->SetProjectionMatrix( &m_ProjectionMatrix );
    m_pPrimitiveRenderer->SetModelMatrix( &m_ModelMatrix );
    m_LineMeshIndex = 0;
}
//  描画を終了します
void GfxContext::End() NN_NOEXCEPT
{
    m_FontRenderer.FlushCommand( m_GraphicsFramework.GetRootCommandBuffer( m_CurrentBufferIndex ) );
    m_GraphicsFramework.EndFrame( m_CurrentBufferIndex );
    m_GraphicsFramework.ExecuteCommand( m_CurrentBufferIndex );
    m_GraphicsFramework.QueuePresentTexture( 1 );
    m_GraphicsFramework.WaitGpuSync( m_CurrentBufferIndex, nn::TimeSpan::FromSeconds(2) );
}

//  四角形を描画します
void GfxContext::DrawQuad(float left, float top, float right, float bottom, uint32_t color) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( m_pPrimitiveRenderer );
    DrawQuad( left, top, right, bottom, TransColor( color ) );
}
//  四角形を描画します
void GfxContext::DrawQuad(float left, float top, float right, float bottom, const nn::util::Uint8x4& color) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( m_pPrimitiveRenderer );
    const nn::util::Vector3f position( left, top, 0.0f );
    const nn::util::Vector3f size( right - left, bottom - top, 0.0f );

    m_pPrimitiveRenderer->SetColor( color );
    m_pPrimitiveRenderer->DrawQuad( m_GraphicsFramework.GetRootCommandBuffer( m_CurrentBufferIndex ), position, size );
}
//  線分を描画します
void GfxContext::DrawLine(float x1, float y1, float x2, float y2, uint32_t color) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( m_pPrimitiveRenderer );
    const nn::util::Vector3f begin( x1, y1, 0.0f );
    const nn::util::Vector3f end( x2, y2, 0.0f );

    m_pPrimitiveRenderer->SetColor( TransColor( color ) );
    m_pPrimitiveRenderer->DrawLine( m_GraphicsFramework.GetRootCommandBuffer( m_CurrentBufferIndex ), begin, end );
}
//  円を描画します
void GfxContext::DrawCircle(float x, float y, float r, uint32_t color) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( m_pPrimitiveRenderer );
    const nn::util::Vector3f center( x, y, 0.0f );

    m_pPrimitiveRenderer->SetColor( TransColor( color ) );
    m_pPrimitiveRenderer->DrawSphere( m_GraphicsFramework.GetRootCommandBuffer( m_CurrentBufferIndex ), nns::gfx::PrimitiveRenderer::Surface_Solid, nns::gfx::PrimitiveRenderer::Subdiv_Normal, center, r * 2 );
}
//  点を結ぶように連続して線分を描画します
void GfxContext::DrawLinkedPoint(const nn::util::Float3* pPoints, int pointCount, const nn::util::Uint8x4& color) NN_NOEXCEPT
{
    if( pointCount == 0 )
    {
        return ;
    }

    NN_ABORT_UNLESS_NOT_NULL( m_pPrimitiveRenderer );
    NN_ABORT_UNLESS_NOT_NULL( pPoints );
    NN_ABORT_UNLESS_LESS_EQUAL( pointCount, DrawalbeLinkedPointCountMax );

    const int lineMeshIndex = m_LineMeshIndex++;
    NN_ABORT_UNLESS_LESS( lineMeshIndex, DrawLinkedPointFunctionCallCount );
    m_LineMeshBuffer.Map( lineMeshIndex );

    nn::util::Float3* pPos = static_cast<nn::util::Float3*>( m_LineMesh[lineMeshIndex].GetVertexBufferCpuAddress( nns::gfx::PrimitiveRenderer::VertexAttribute_Pos ) );

    for( int i = 0; i < pointCount; i++ )
    {
        pPos[i] = pPoints[i];
    }
    for( int i = pointCount; i < DrawalbeLinkedPointCountMax; i++ )
    {
        pPos[i] = pPoints[pointCount - 1];
    }

    m_LineMeshBuffer.Unmap();

    m_pPrimitiveRenderer->SetColor( color );
    m_pPrimitiveRenderer->DrawUserMesh( m_GraphicsFramework.GetRootCommandBuffer( m_CurrentBufferIndex ), nn::gfx::PrimitiveTopology_LineStrip, &m_LineMesh[lineMeshIndex] );
}

//  FontRenderer を取得します
FontRenderer& GfxContext::GetFontRenderer() NN_NOEXCEPT
{
    return m_FontRenderer;
}

//  PrimitiveRenderer に渡す Allocate 関数
void* GfxContext::AllocateForPrimitiveRenderer(size_t size, size_t alignment, void* pUserData) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( pUserData );
    GfxContext* pGfxContext = reinterpret_cast<GfxContext*>( pUserData );

    void* buffer = pGfxContext->m_Allocator.Allocate( size, alignment );
    NN_ABORT_UNLESS_NOT_NULL( buffer );
    return buffer;
}
//  PrimitiveRenderer に渡す Free 関数
void GfxContext::FreeForPrimitiveRenderer(void* ptr, void* pUserData) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( pUserData );
    GfxContext* pGfxContext = reinterpret_cast<GfxContext*>( pUserData );

    pGfxContext->m_Allocator.Free( ptr );
}


#endif  //  NN_ATK_ENABLE_GFX_VIEWING
