﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "GfxContext.h"
#include "FontRenderer.h"

#if defined( NN_ATK_ENABLE_GFX_VIEWING )

#include <cstdarg>
#include <map>
#include <nn/os.h>


//  図形の描画を担当します
class DebugPrimitiveViewer
{
public:
    DebugPrimitiveViewer() NN_NOEXCEPT;
    void Initialize() NN_NOEXCEPT;
    void Erase(const char* tag) NN_NOEXCEPT;
    void Reset() NN_NOEXCEPT;
    void Draw(GfxContext& gfxContext) NN_NOEXCEPT;

    void DrawQuad(const char* tag, float left, float top, float right, float bottom, uint32_t color) NN_NOEXCEPT;
    void DrawLine(const char* tag, float x1, float y1, float x2, float y2, uint32_t color) NN_NOEXCEPT;
    void DrawCircle(const char* tag, float x, float y, float r, uint32_t color) NN_NOEXCEPT;

private:
    enum class PrimitiveType
    {
        PrimitiveType_Quad,
        PrimitiveType_Line,
        PrimitiveType_Circle,
    };
    struct QuadData
    {
        float left, top, right, bottom;
    };
    struct LineData
    {
        float x1, y1, x2, y2;
    };
    struct CircleData
    {
        float x, y, r;
    };
    struct PrimitiveData
    {
        PrimitiveType type;
        uint32_t color;
        std::string tag;
        union
        {
            QuadData quad;
            LineData line;
            CircleData circle;
        };
    };
    //  key: タグ, val: タグに対する図形
    std::map<std::string,PrimitiveData> m_PrimitiveData;
    nn::os::Mutex m_DataChangeMutex;
};

//  文字列の描画を担当します
class DebugPrintViewer
{
public:
    DebugPrintViewer() NN_NOEXCEPT;
    void Initialize() NN_NOEXCEPT;
    void Reset() NN_NOEXCEPT;
    void Draw(GfxContext& gfxContext) NN_NOEXCEPT;

    void Print(const char* tag, const char* fmt, va_list args) NN_NOEXCEPT;
    void Print(const char* tag, float x, float y, const char* fmt, va_list args) NN_NOEXCEPT;
    void Print(const char* tag, float x, float y, float scaleX, float scaleY, const char* fmt, va_list args) NN_NOEXCEPT;
    void Print(const char* tag, float x, float y, float scaleX, float scaleY, const nn::util::Unorm8x4& fontColor, const nn::util::Uint8x4& backColor, const char* fmt, va_list args) NN_NOEXCEPT;

private:
    struct PrintData
    {
        float x;
        float y;
        float scaleX;
        float scaleY;
        nn::util::Unorm8x4 fontColor;
        nn::util::Uint8x4 backColor;
        std::string str;
    };
    //  key: タグ, val: タグに対する文字列
    std::map<std::string,PrintData> m_PrintData;
    float m_PrintPositionX, m_PrintPositionY;
    nn::os::Mutex m_DataChangeMutex;
};

//  Gfx を用いた Debug 用表示を行います
class DebugViewer
{
public:
    // デフォルトの文字列描画倍率
    static const float DefaultPrintScaleX;
    static const float DefaultPrintScaleY;
    // デフォルトのバー描画サイズ
    static const int DefaultDrawBarWidth = 200;
    static const int DefaultDrawBarHeight = 10;

    //  初期化します
    void Initialize() NN_NOEXCEPT;
    //  終了処理します
    void Finalize() NN_NOEXCEPT;
    //  プリミティブを削除します。
    //  tag には表示する文字列のタグを指定してください。
    void ErasePrimitive(const char* tag) NN_NOEXCEPT;
    //  情報をリセットします
    void Reset() NN_NOEXCEPT;
    //  更新します
    void Update() NN_NOEXCEPT;
    //  ディスプレイに描画します
    void Present() NN_NOEXCEPT;

    //  文字列を描画します。
    //  tag には表示する文字列のタグを指定してください。
    //  同じ tag が指定された場合は、以前の内容を上書きします。
    void Print(const char* tag, const char* fmt, ...) NN_NOEXCEPT;
    //  文字列を ( x, y ) の位置に描画します。
    //  tag には表示する文字列のタグを指定してください。
    //  同じ tag が指定された場合は、以前の内容を上書きします。
    void Print(const char* tag, int x, int y, const char* fmt, ...) NN_NOEXCEPT;
    //  文字列を ( x, y ) の位置に ( scaleX, scaleY ) の倍率で描画します。
    //  tag には表示する文字列のタグを指定してください。
    //  同じ tag が指定された場合は、以前の内容を上書きします。
    void Print(const char* tag, int x, int y, float scaleX, float scaleY, const char* fmt, ...) NN_NOEXCEPT;
    //  文字列を ( x, y ) の位置に ( scaleX, scaleY ) の倍率で ( fontColor, backColor ) の色で描画します。
    //  tag には表示する文字列のタグを指定してください。
    //  同じ tag が指定された場合は、以前の内容を上書きします。
    void Print(const char* tag, int x, int y, float scaleX, float scaleY, const nn::util::Unorm8x4& fontColor, const nn::util::Uint8x4& backColor, const char* fmt, ...) NN_NOEXCEPT;
    //  文字列を描画します。
    //  tag には表示する文字列のタグを指定してください。
    //  同じ tag が指定された場合は、以前の内容を上書きします。
    void PrintImpl(const char* tag, const char* fmt, va_list args) NN_NOEXCEPT;
    //  文字列を ( x, y ) の位置に描画します。
    //  tag には表示する文字列のタグを指定してください。
    //  同じ tag が指定された場合は、以前の内容を上書きします。
    void PrintImpl(const char* tag, int x, int y, const char* fmt, va_list args) NN_NOEXCEPT;
    //  文字列を ( x, y ) の位置に ( scaleX, scaleY ) の倍率で描画します。
    //  tag には表示する文字列のタグを指定してください。
    //  同じ tag が指定された場合は、以前の内容を上書きします。
    void PrintImpl(const char* tag, int x, int y, float scaleX, float scaleY, const char* fmt, va_list args) NN_NOEXCEPT;
    //  文字列を ( x, y ) の位置に ( scaleX, scaleY ) の倍率で ( fontColor, backColor ) の色で描画します。
    //  tag には表示する文字列のタグを指定してください。
    //  同じ tag が指定された場合は、以前の内容を上書きします。
    void PrintImpl(const char* tag, int x, int y, float scaleX, float scaleY, const nn::util::Unorm8x4& fontColor, const nn::util::Uint8x4& backColor, const char* fmt, va_list args) NN_NOEXCEPT;

    //  四角形を描画します。
    //  tag には表示する図形のタグを指定していください。
    //  同じ tag が指定された場合は、以前の情報を上書きします。
    //  color には表示する図形の色を 0xAARRGGBB または 0xRRGGBB の形か GetDebugColor を用いて指定してください。
    void DrawQuad(const char* tag, int left, int top, int right, int bottom, uint32_t color) NN_NOEXCEPT;
    //  線分を描画します。
    //  tag には表示する図形のタグを指定していください。
    //  同じ tag が指定された場合は、以前の情報を上書きします。
    //  color には表示する図形の色を 0xAARRGGBB または 0xRRGGBB の形か GetDebugColor を用いて指定してください。
    void DrawLine(const char* tag, int x1, int y1, int x2, int y2, uint32_t color) NN_NOEXCEPT;
    //  円を描画します。
    //  tag には表示する図形のタグを指定していください。
    //  同じ tag が指定された場合は、以前の情報を上書きします。
    //  color には表示する図形の色を 0xAARRGGBB または 0xRRGGBB の形か GetDebugColor を用いて指定してください。
    void DrawCircle(const char* tag, int x, int y, int r, uint32_t color) NN_NOEXCEPT;

    //  Singleton インスタンスを取得します
    static DebugViewer& GetInstance() NN_NOEXCEPT;

private:
    DebugViewer() NN_NOEXCEPT;
    DebugViewer(const DebugViewer&) = delete;
    DebugViewer(DebugViewer&&)= delete;
    const DebugViewer& operator =(const DebugViewer&) = delete;
    const DebugViewer& operator =(DebugViewer&&) = delete;

private:
    GfxContext m_GfxContext;
    DebugPrimitiveViewer m_PrimitiveViewer;
    DebugPrintViewer m_PrintViewer;
};


#else

//  Gfx を用いない場合のダミークラス
class DebugViewer
{
public:
    void Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    void Reset() NN_NOEXCEPT;
    void Update() NN_NOEXCEPT;
    void Present() NN_NOEXCEPT;
    static DebugViewer& GetInstance() NN_NOEXCEPT;
};

#endif  //  NN_ATK_ENABLE_GFX_VIEWING


//  DebugPrimitive で色を取得します
extern uint32_t GetDebugColor(uint8_t r, uint8_t g, uint8_t b) NN_NOEXCEPT;
//  DebugPrimitive で色を取得します
extern uint32_t GetDebugColor(uint8_t r, uint8_t g, uint8_t b, uint8_t a) NN_NOEXCEPT;

//  DebugViewer を用いて文字列を描画します
extern void DebugPrint(const char* tag, const char* fmt, ...) NN_NOEXCEPT;
//  DebugViewer を用いて ( x, y ) の位置に文字列を描画します
extern void DebugPrint(const char* tag, int x, int y, const char* fmt, ...) NN_NOEXCEPT;
//  DebugViewer を用いて ( x, y ) の位置に文字列を ( scaleX, scaleY ) の倍率で描画します
extern void DebugPrint(const char* tag, int x, int y, float scaleX, float scaleY, const char* fmt, ...) NN_NOEXCEPT;
//  DebugViewer を用いて ( x, y ) の位置に文字列を ( scaleX, scaleY ) の倍率で ( fontColor, backColor ) の色で描画します
extern void DebugPrint(const char* tag, int x, int y, float scaleX, float scaleY, const nn::util::Unorm8x4& fontColor, const nn::util::Uint8x4& backColor, const char* fmt, ...) NN_NOEXCEPT;

//  DebugViewer を用いて四角形を描画します
extern void DebugDrawQuad(const char* tag, int left, int top, int right, int bottom, uint32_t color) NN_NOEXCEPT;
//  DebugViewer を用いて線分を描画します
extern void DebugDrawLine(const char* tag, int x1, int y1, int x2, int y2, uint32_t color) NN_NOEXCEPT;
//  DebugViewer を用いて円を描画します
extern void DebugDrawCircle(const char* tag, int x, int y, int r, uint32_t color) NN_NOEXCEPT;
