﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "StreamSound.h"

#include "../FlagList.h"
#include "../GfxCode/DebugViewer.h"

#include <atomic>

namespace
{
    int64_t LoadFile(void** pFile, const char* fileName)
    {
        NN_ABORT_UNLESS(*pFile == nullptr);

        nn::fs::FileHandle handle;
        nn::Result fsResult;

        fsResult = nn::fs::OpenFile(&handle, fileName, nn::fs::OpenMode_Read);
        NN_ABORT_UNLESS(fsResult.IsSuccess(), "nn::fs::OpenFile is failed.");

        int64_t size;
        fsResult = nn::fs::GetFileSize(&size, handle);
        NN_ABORT_UNLESS(fsResult.IsSuccess(), "nn::fs::GetFileSize is failed.");

        *pFile = nns::atk::Allocate(static_cast<size_t>(size), nn::audio::BufferAlignSize);
        fsResult = nn::fs::ReadFile(handle, 0 , *pFile, static_cast<size_t>(size));
        NN_ABORT_UNLESS(fsResult.IsSuccess(), "nn::fs::ReadFile is failed.");

        nn::fs::CloseFile(handle);

        return size;
    }

    void UnloadFile(void** pFile)
    {
        nns::atk::Free(*pFile);
        *pFile = nullptr;
    }

    void* g_OpusWorkBuffer;
    void* g_HardwareOpusWorkBuffer;
    void* g_pExStreamSoundFile;

    int64_t g_ExStreamSoundFileSize;

    enum FlagType
    {
        FlagType_EnableExternalPrefetchStreamSound,    //< 外部再生版のプリフェッチストリームを有効化
        FlagType_EnableLessBlockPrefetch,              //< ストリームバッファのブロック数が少ないモードを有効化
        FlagType_EnableStreamJump,                     //< ストリームジャンプを有効化
        FlagType_EnableStreamJumpNamedRegion,          //< リージョン名使用版のストリームジャンプを有効化
        FlagType_Num
    };

    FlagElement g_LocalElements[] =
    {
        { "EnableExternalPrefetchStreamSound", false },
        { "EnableLessBlockPrefetch", false },
        { "EnableStreamJump", false },
        { "EnableStreamJumpNamedRegion", false },
    };

    FlagList g_LocalFlagList(g_LocalElements, sizeof(g_LocalElements) / sizeof(g_LocalElements[0]));

    enum StreamRegion {
        StreamRegionIntro = 0,
        StreamRegionA = 1
    };
    uint32_t g_RegionList[] = {
        StreamRegionIntro,
        StreamRegionA,
        StreamRegionA
    };
    uint32_t g_RegionListIndex = 0;
    std::atomic<bool> g_IsRegionEndUpdated;
    std::atomic<int64_t> g_CurrentRegionEnd;

    nn::atk::StreamRegionCallbackResult StreamRegionCallback(
        nn::atk::StreamRegionCallbackParam* param, void* /*arg*/ )
    {
        nn::atk::detail::StreamSoundFile::RegionInfo info;
        NN_ASSERT(param->pRegionInfoReader->ReadRegionInfo(&info, param->regionNo));
        g_CurrentRegionEnd = static_cast<int64_t>(info.end);
        g_IsRegionEndUpdated = true;

        if ( g_RegionListIndex >= sizeof(g_RegionList) / sizeof(g_RegionList[0]) )
        {
            NN_LOG("[%s] g_RegionListIndex(%d) => FINISH\n", __FUNCTION__, g_RegionListIndex);
            return nn::atk::StreamRegionCallbackResult_Finish;
        }

        param->regionNo = g_RegionList[g_RegionListIndex];
        NN_ASSERT_LESS(param->regionNo, param->regionCount);
        NN_LOG("[%s] g_RegionListIndex(%d) => CONTINUE\n", __FUNCTION__, g_RegionListIndex);
        NN_ASSERT(param->pRegionInfoReader->ReadRegionInfo(&info, param->regionNo));
        NN_LOG("[NextRegion] idx:%4d start:%8u end:%8u len:%8u\n", param->regionNo, info.start, info.end, info.end - info.start);
        g_RegionListIndex++;
        return nn::atk::StreamRegionCallbackResult_Continue;
    }

    int g_RegionNameListIndex = -1;
    const char* RegionNameList[] =
    {
        "RegionA",
        "RegionB",
        "RegionA",
        "RegionA",
        "RegionB",
    };

    nn::atk::StreamRegionCallbackResult StreamNamedRegionCallback(
        nn::atk::StreamRegionCallbackParam* param, void* /*arg*/ )
    {
        if (g_RegionNameListIndex < 0)
        {
            param->regionNo = 0;
            g_RegionNameListIndex = 0;
            NN_LOG("[%s] %d => CONTINUE\n", __FUNCTION__, param->regionNo);
            return nn::atk::StreamRegionCallbackResult_Continue;
        }
        else
        {
            if (g_RegionNameListIndex >= sizeof(RegionNameList) / sizeof(RegionNameList[0]))
            {
                NN_LOG("[%s] %s => FINISH\n", __FUNCTION__, param->regionName);
                return nn::atk::StreamRegionCallbackResult_Finish;
            }
        }

        param->isRegionNameEnabled = true;
        std::strncpy(param->regionName, RegionNameList[g_RegionNameListIndex], nn::atk::RegionNameLengthMax + 1);
        NN_LOG("[%s] %s => CONTINUE\n", __FUNCTION__, param->regionName);

        g_RegionNameListIndex++;

        return nn::atk::StreamRegionCallbackResult_Continue;

    }
}

void StreamSoundCheckModule::OnInitializeAtk() NN_NOEXCEPT
{
    m_IsPause = false;

    if ( GetLocalFlagList().IsFlagEnabled(FlagType_EnableLessBlockPrefetch) )
    {
        m_CommonObject.GetSoundArchivePlayer().SetStreamBlockCount(2); // ストリームバッファブロックを減らしプリフェッチデータのサイズを減少させる機能の評価
    }

    int opusDecoderCount = 4;
    size_t workBufferSize = nn::atk::GetRequiredOpusDecoderBufferSize(opusDecoderCount);
    g_OpusWorkBuffer = nns::atk::Allocate(workBufferSize);
    nn::atk::InitializeOpusDecoder(g_OpusWorkBuffer, workBufferSize, opusDecoderCount);

    int hardwareOpusDecoderCount = 4;
    size_t hardwareOpusWorkBufferSize = nn::atk::GetRequiredHardwareOpusDecoderBufferSize(hardwareOpusDecoderCount);
    g_HardwareOpusWorkBuffer = nns::atk::Allocate(hardwareOpusWorkBufferSize);
    nn::atk::InitializeHardwareOpusDecoder(g_HardwareOpusWorkBuffer, hardwareOpusWorkBufferSize, hardwareOpusDecoderCount);

    // プリフェッチストリームサウンドをメモリにロード
    if ( GetLocalFlagList().IsFlagEnabled(FlagType_EnableExternalPrefetchStreamSound) )
    {
        LoadPrefetchStreamData();
    }

    m_CommonObject.Initialize();

    g_ExStreamSoundFileSize = LoadFile(&g_pExStreamSoundFile, "content:/PIANO16.bfstm");

    m_EstimatedRegionEndTick = nn::os::Tick(0);
    g_IsRegionEndUpdated = false;
    g_CurrentRegionEnd = 0;
}

void StreamSoundCheckModule::OnFinalizeAtk() NN_NOEXCEPT
{
    UnloadFile(&g_pExStreamSoundFile);
    g_ExStreamSoundFileSize = 0;

    m_CommonObject.Finalize();

    if ( GetLocalFlagList().IsFlagEnabled(FlagType_EnableExternalPrefetchStreamSound) )
    {
        nns::atk::FreeForMemoryPool( m_pMemoryForBfstpFile );
    }

    nn::atk::FinalizeHardwareOpusDecoder();
    nn::atk::FinalizeOpusDecoder();
    nns::atk::Free(g_HardwareOpusWorkBuffer);
    nns::atk::Free(g_OpusWorkBuffer);
}

void StreamSoundCheckModule::OnLoadData() NN_NOEXCEPT
{
    bool isSuccess;
    nn::atk::SoundDataManager& soundDataManager = m_CommonObject.GetSoundDataManager();
    nn::atk::SoundHeap& soundHeap = m_CommonObject.GetSoundHeap();

    isSuccess = soundDataManager.LoadData( STRM_MARIOKART, &soundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(STRM_MARIOKART) failed." );
}

void StreamSoundCheckModule::OnPrintUsage() NN_NOEXCEPT
{
    if ( GetLocalFlagList().IsFlagEnabled(FlagType_EnableStreamJump) )
    {
        if ( GetLocalFlagList().IsFlagEnabled(FlagType_EnableStreamJumpNamedRegion) )
        {
            NN_LOG("[Y]            StartSound STRM (STRM_NAMEDREGION_TEST)\n");
        }
        else
        {
            NN_LOG("[Y]            StartSound STRM (STRM_REGION_JUMP)\n");
        }
    }
    else
    {
        NN_LOG("[Y]            StartSound STRM (STRM_PIANO16_PCM16)\n");
    }

    if ( GetLocalFlagList().IsFlagEnabled(FlagType_EnableExternalPrefetchStreamSound) )
    {
        NN_LOG("[R + Y]        StartSound Prefetch EXSTRM (STRM_PIANO16_ADPCM)\n");
    }
    else
    {
        NN_LOG("[R + Y]        StartSound EXSTRM (STRM_PIANO16_ADPCM)\n");
    }
    NN_LOG("[X]            StartSound OPUS With OpusDecoder (1ch) \n");
    NN_LOG("[R + X]        StartSound OPUS With HardwareOpusDecoder (2ch) \n");
    NN_LOG("[A]            StartSound STRM on Memory (STRM_PIANO16_PCM16)\n");
    NN_LOG("[R + A]        StartSound STRM (STRM_MARIOKART_OPUS)\n");
    NN_LOG("[Down]         StartSound Prefetch STRM (STRM_MARIOKART)\n");
    NN_LOG("[B]            Stop Sound\n");
    NN_LOG("[R + B]        Pause Sound\n");
    m_CommonObject.GetAtkProfiler().PrintUsage();
}

void StreamSoundCheckModule::OnUpdateInput() NN_NOEXCEPT
{
    int processCount = m_CommonObject.GetAtkProfiler().UpdateInput();

    if (processCount == 0)
    {
        // StartSound / StopSound
        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >())
        {
            if (nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
            {
                m_CommonObject.PlayWithStartSound(STRM_MARIOKART_OPUS, "STRM_MARIOKART_OPUS");
            }
            else
            {
                PlayExOnMemStrmWithStartSound(STRM_PIANO16_PCM16, "STRM_PIANO16_PCM16");
            }
        }

        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >())
        {
            if (nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
            {
                PlayExStrmWithStartSound(STRM_PIANO16_ADPCM, "STRM_PIANO16_ADPCM");
            }
            else
            {
                if ( GetLocalFlagList().IsFlagEnabled(FlagType_EnableStreamJump) )
                {
                    if ( GetLocalFlagList().IsFlagEnabled(FlagType_EnableStreamJumpNamedRegion) )
                    {
                        m_CommonObject.GetSoundHandle().Stop(0);

                        g_RegionNameListIndex = -1;

                        nn::atk::SoundStartable::StartInfo info;
                        info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;
                        info.streamSoundInfo.regionCallback = StreamNamedRegionCallback;
                        info.streamSoundInfo.regionCallbackArg = nullptr;

                        CommonObject::StartParam startParam;
                        startParam.pStartInfo = &info;

                        m_CommonObject.PlayWithStartSound(STRM_NAMEDREGION_TEST, "STRM_NAMEDREGION_TEST", startParam);
                    }
                    else
                    {
                        m_CommonObject.GetSoundHandle().Stop(0);

                        g_RegionListIndex = 0;

                        nn::atk::SoundStartable::StartInfo info;
                        info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;
                        info.streamSoundInfo.regionCallback = StreamRegionCallback;
                        info.streamSoundInfo.regionCallbackArg = nullptr;

                        CommonObject::StartParam startParam;
                        startParam.pStartInfo = &info;

                        m_CommonObject.PlayWithStartSound(STRM_REGION_JUMP, "STRM_REGION_JUMP", startParam);
                    }
                }
                else
                {
                    m_CommonObject.PlayWithStartSound(STRM_PIANO16_PCM16, "STRM_PIANO16_PCM16");
                }
            }
        }

        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >())
        {
            if (nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
            {
                PlayOpusStrmWithStartSound(STRM_MARIOKART, "Play opus file with STRM_MARIOKART label");
            }
            else
            {
                PlayOpusStrmWithStartSound(STRM_PIANO16_ADPCM, "Play opus file with STRM_PIANO16_ADPCM label");
            }
        }

        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B>())
        {
            if (nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
            {
                m_IsPause = !m_IsPause;
                m_CommonObject.GetSoundHandle().Pause(m_IsPause, 0);
            }
            else
            {
                m_CommonObject.GetSoundHandle().Stop(0);
            }
        }

        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >())
        {
            m_CommonObject.PlayWithStartSound( STRM_MARIOKART, "Prefetch STRM_MARIOKART" );
        }
    }

    m_CommonObject.GetAtkProfiler().Show(m_CommonObject.GetSoundHandle());
}

void StreamSoundCheckModule::OnUpdateAtk() NN_NOEXCEPT
{
    m_CommonObject.Update();
    UpdateEstimateRegionEndTick();
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void StreamSoundCheckModule::OnUpdateDraw() NN_NOEXCEPT
{
    m_CommonObject.UpdateDraw(GetModuleName());
    DrawRestRegionTime();
}
#endif

FlagList& StreamSoundCheckModule::GetLocalFlagList() NN_NOEXCEPT
{
    return g_LocalFlagList;
}

void StreamSoundCheckModule::LoadPrefetchStreamData() NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    nn::Result fsResult;

    const char* prefetchFileName = nullptr;
    if ( GetLocalFlagList().IsFlagEnabled(FlagType_EnableLessBlockPrefetch) )
    {
        prefetchFileName = "content:/PIANO16.lessBlock.bfstp";
    }
    else
    {
        prefetchFileName = "content:/PIANO16.bfstp";
    }

    fsResult = nn::fs::OpenFile(&handle, prefetchFileName, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(fsResult.IsSuccess(), "nn::fs::OpenFile is failed.");

    int64_t size;
    fsResult = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(fsResult.IsSuccess(), "nn::fs::GetFileSize is failed.");

    m_pMemoryForBfstpFile = nns::atk::AllocateForMemoryPool(static_cast<size_t>(size), nn::audio::BufferAlignSize);
    fsResult = nn::fs::ReadFile(handle, 0 , m_pMemoryForBfstpFile, static_cast<size_t>(size));
    NN_ABORT_UNLESS(fsResult.IsSuccess(), "nn::fs::ReadFile is failed.");

    NN_LOG("%s is loaded\n", prefetchFileName);

    nn::fs::CloseFile(handle);
}

// 特殊なストリームサウンドを再生します
void StreamSoundCheckModule::PlayExStrmWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT
{
    m_CommonObject.GetSoundHandle().Stop( 0 );
    nn::atk::SoundStartable::StartInfo info;
    info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;
    // 外部ファイルのパスを与えてストリームサウンドを再生
    info.streamSoundInfo.externalPath = "content:/PIANO16.bfstm";
    if ( GetLocalFlagList().IsFlagEnabled(FlagType_EnableExternalPrefetchStreamSound) )
    {
        info.streamSoundInfo.prefetchData = m_pMemoryForBfstpFile;
        NN_LOG("Use External PrefetchData!\n");
    }
    bool result = m_CommonObject.GetSoundArchivePlayer().StartSound( &m_CommonObject.GetSoundHandle(), soundId, &info ).IsSuccess();
    NN_LOG("StartSound(%s) ... (%d)\n", debugLabelName, result);
}

// 特殊なストリームサウンドを再生します
void StreamSoundCheckModule::PlayExOnMemStrmWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT
{
    m_CommonObject.GetSoundHandle().Stop( 0 );
    nn::atk::SoundStartable::StartInfo info;
    info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;

    // 外部ファイルのメモリを与えてストリームサウンドを再生
    info.streamSoundInfo.pExternalData = g_pExStreamSoundFile;
    info.streamSoundInfo.externalDataSize = static_cast<size_t>(g_ExStreamSoundFileSize);

    bool result = m_CommonObject.GetSoundArchivePlayer().StartSound( &m_CommonObject.GetSoundHandle(), soundId, &info ).IsSuccess();
    NN_LOG("StartSound(%s on Memory, ptr = %p, size = %zu)  ... (%d)\n",
        debugLabelName,
        info.streamSoundInfo.pExternalData,
        info.streamSoundInfo.externalDataSize,
        result);
}

// Opus形式のストリームサウンドを再生します
void StreamSoundCheckModule::PlayOpusStrmWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT
{
    m_CommonObject.GetSoundHandle().Stop( 0 );
    nn::atk::SoundStartable::StartInfo info;
    info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StartOffset;
    info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;
    info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundMetaInfo;
    info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundMetaInfo2;

    info.startOffsetType = nn::atk::SoundStartable::StartInfo::StartOffsetType_MilliSeconds;
    info.startOffset = 5000;

    const nn::atk::SoundArchive& player = m_CommonObject.GetSoundArchivePlayer().GetSoundArchive();

    // 外部ファイルのパスとループ情報を与えてストリームサウンドを再生
    if ( soundId == STRM_PIANO16_ADPCM )
    {
        info.streamSoundInfo.externalPath = "content:/PIANO16.opus";

        info.streamSoundMetaInfo2.loopStartFrame = 71845;
        info.streamSoundMetaInfo2.loopEndFrame = 355677;

        // 既存のパラメータをメタ情報として流用
        player.ReadStreamSoundInfo(&info.streamSoundMetaInfo, STRM_PIANO16_PCM16);

        // ファイル形式を Opus ( CPU 再生) に変更
        info.streamSoundMetaInfo.streamFileType = nn::atk::SoundArchive::StreamFileType_Opus;
        info.streamSoundMetaInfo.decodeMode = nn::atk::SoundArchive::DecodeMode_Cpu;
    }
    else if ( soundId == STRM_MARIOKART )
    {
        info.streamSoundInfo.externalPath = "content:/kart_title.24.opus";

        info.streamSoundMetaInfo2.loopStartFrame = 276098;
        info.streamSoundMetaInfo2.loopEndFrame = 2217999;

        // 既存のパラメータをメタ情報として流用
        player.ReadStreamSoundInfo(&info.streamSoundMetaInfo, STRM_MARIOKART);

        // ファイル形式を Opus (ハードウェア再生) に変更
        info.streamSoundMetaInfo.streamFileType = nn::atk::SoundArchive::StreamFileType_Opus;
        info.streamSoundMetaInfo.decodeMode = nn::atk::SoundArchive::DecodeMode_Accelerator;
    }
    else
    {
        NN_ABORT("Unexpected case.");
    }
    info.streamSoundMetaInfo2.isLoop = true;


    bool result = m_CommonObject.GetSoundArchivePlayer().StartSound( &m_CommonObject.GetSoundHandle(), soundId, &info ).IsSuccess();
    NN_LOG("StartSound(%s) ... (%d)\n", debugLabelName, result);
}

void StreamSoundCheckModule::UpdateEstimateRegionEndTick() NN_NOEXCEPT
{
    if (g_IsRegionEndUpdated)
    {
        // リージョン終端への到達予想時刻を計算
        nn::atk::StreamSoundHandle streamSoundHandle(&m_CommonObject.GetSoundHandle());
        if (streamSoundHandle.IsAttachedSound())
        {
            nn::atk::StreamSoundDataInfo info;
            if (streamSoundHandle.ReadStreamSoundDataInfo(&info))
            {
                int64_t restRegionSampleCount = g_CurrentRegionEnd - streamSoundHandle.GetPlaySamplePosition();
                nn::TimeSpan restRegionTime = nn::TimeSpan::FromMilliSeconds(static_cast<int64_t>(restRegionSampleCount / (info.sampleRate / 1000.f)));
                m_EstimatedRegionEndTick = nn::os::GetSystemTick() + nn::os::Tick(restRegionTime);
                g_IsRegionEndUpdated = false;
            }
        }
    }
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void StreamSoundCheckModule::DrawRestRegionTime() NN_NOEXCEPT
{
    if (m_EstimatedRegionEndTick != nn::os::Tick(0))
    {
        float restRegionTime = (m_EstimatedRegionEndTick - nn::os::GetSystemTick()).ToTimeSpan().GetMilliSeconds() / 1000.f;
        if (restRegionTime < 0.f)
        {
            restRegionTime = 0.f;
            m_EstimatedRegionEndTick = nn::os::Tick(0);
        }
        DebugPrint("RestRegionTime", 0, 0, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY, "%f", restRegionTime);
    }
}
#endif
