﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "CustomSubMix.h"

#include "../FlagList.h"
#include "../GfxCode/DebugViewer.h"

namespace
{
    enum FlagType
    {
        FlagType_NoSubMix,  //< SubMix を使用しないで FinalMix のみで再生する
        FlagType_UseDefaultOutputReceiver, //< StartInfo による OutputReceiver の設定を行わずに再生する
        FlagType_Num
    };
    FlagElement g_LocalElements[] =
    {
        { "NoSubMix", false },
        { "UseDefaultOutputReceiver", false },
    };
    FlagList g_LocalFlagList(g_LocalElements, sizeof(g_LocalElements) / sizeof(g_LocalElements[0]));

    const int AppendReverbBusIndex = 2;
}

NN_DEFINE_STATIC_CONSTANT( const int CustomSubMixCheckModule::CommonObjectForCustomSubMix::SubMixCount );

void CustomSubMixCheckModule::CommonObjectForCustomSubMix::Initialize(const CommonObject::InitializeParam& param) NN_NOEXCEPT
{
    m_IsNoSubMix = param.GetFlagList()->IsFlagEnabled( FlagType_NoSubMix );
    m_IsUseDefaultOutputReceiver = param.GetFlagList()->IsFlagEnabled( FlagType_UseDefaultOutputReceiver );
    const int channelCount = nn::atk::detail::driver::HardwareManager::GetInstance().GetChannelCountMax();

    CommonObject::InitializeParam commonObjParam;
    auto& soundSystemParam = commonObjParam.GetSoundSystemParam();

    //  SubMix それぞれのバス数
    const int SubMixBusCount[]= { 1, 3, 1 };

    if( m_IsNoSubMix )
    {
        soundSystemParam.enableCustomSubMix = true;
        soundSystemParam.subMixCount = 0;
        soundSystemParam.mixBufferCount = 0;
    }
    else
    {
        soundSystemParam.enableCustomSubMix = true;
        soundSystemParam.subMixCount = SubMixCount;

        int busCount = 0;
        for(auto x : SubMixBusCount)
        {
            busCount += x;
        }

        soundSystemParam.mixBufferCount = busCount * channelCount;
    }

    CommonObject::Initialize( commonObjParam );
    m_pFinalMix = &nn::atk::SoundSystem::GetFinalMix();

    if( !m_IsNoSubMix )
    {
        //  SubMix[0] ---> SubMix[1] ---
        //                               \
        //                                ---> FinalMix
        //                               /
        //  SubMix[2] ------------------
        //
        //  SubMix[0] : バス数 1, チャンネル数 channelCount (= 6)
        //  SubMix[1] : バス数 3, チャンネル数 channelCount (= 6)
        //              2 番目のバスには Reverb を Append している
        //  SubMix[2] : バス数 1, チャンネル数 channelCount (= 6)
        //
        //  というサブミックスを構成します


        //  処理の簡素化のために FinalMix を指定する FinalMixIndex を用意します
        const int FinalMixIndex = SubMixCount;
        //  DestinationIndex[i] は SubMix[i] の出力先 SubMix の index を指定します
        const int DestinationIndex[] =
        {
            1, FinalMixIndex, FinalMixIndex
        };

        //  接続先バス数を変数に記憶します
        int busCount[ FinalMixIndex + 1 ];
        for(int i = 0; i < SubMixCount; i++)
        {
            busCount[i] = SubMixBusCount[i];
        }
        busCount[ FinalMixIndex ] = m_pFinalMix->GetBusCount();

        //  SubMix の初期化
        for(int i = 0; i < SubMixCount; i++)
        {
            const int srcBus = busCount[i];
            const int dstBus = busCount[ DestinationIndex[i] ];

            const size_t size = nn::atk::SubMix::GetRequiredMemorySize( srcBus, channelCount, dstBus, channelCount );
            void* buffer = nns::atk::Allocate( size );
            const bool result = m_SubMix[i].Initialize( srcBus, channelCount, dstBus, channelCount, buffer, size );
            NN_SDK_ASSERT( result );
            NN_UNUSED( result );
            m_SubMixBuffer[i] = buffer;
        }

        //  SubMix の接続
        for(int i = 0; i< SubMixCount; i++)
        {
            nn::atk::OutputReceiver* pReceiver;
            if( DestinationIndex[i] == FinalMixIndex )
            {
                pReceiver = m_pFinalMix;
            }
            else
            {
                pReceiver = &m_SubMix[ DestinationIndex[i] ];
            }
            m_SubMix[i].SetDestination( pReceiver );
        }

        //  SubMix[1] の各バスの出力先を FinalMix のバス 0 に設定
        for(int bus = 0; bus < m_SubMix[1].GetBusCount(); bus++)
        {
            m_SubMix[1].SetSend( bus, 0, 1.0f );
        }

        m_SubMixDestinationBusIndex = 0;    //  SubMix[0] の出力先バスはひとまず 0 番目
        m_SubMixDestinationSubMixIndex = DestinationIndex[0];
        SetTargetToSubMix( 0 );             //  デフォルトの出力先は SubMix[0]
    }
    else
    {
        SetTargetToFinalMix();
    }

    //  Reverb の準備
    if( !m_IsNoSubMix )
    {
        const size_t size = m_Reverb.GetRequiredMemSize();
        m_ReverbBuffer = nns::atk::AllocateForMemoryPool( size, nn::audio::BufferAlignSize );
        const bool result = m_SubMix[1].AppendEffect( &m_Reverb, AppendReverbBusIndex, m_ReverbBuffer, size );
        NN_SDK_ASSERT( result );
        NN_UNUSED( result );
        m_Reverb.SetEnabled( true );
    }

    // デフォルトの出力先の設定
    if ( !m_IsNoSubMix )
    {
        if (m_IsUseDefaultOutputReceiver)
        {
            CommonObject::GetSoundArchivePlayer().SetDefaultOutputReceiver(&m_SubMix[0]);
        }
    }
}

void CustomSubMixCheckModule::CommonObjectForCustomSubMix::Finalize() NN_NOEXCEPT
{
    //  Effect の削除
    if( !m_IsNoSubMix )
    {
        const auto WaitTime = nn::TimeSpan::FromMilliSeconds( 5 );

        m_Reverb.SetEnabled( false );
        while( !m_Reverb.IsRemovable() )
        {
            nn::os::SleepThread( WaitTime );
        }
        m_SubMix[1].RemoveEffect( &m_Reverb, AppendReverbBusIndex );
        nns::atk::FreeForMemoryPool( m_ReverbBuffer );
    }

    CommonObject::Finalize();
}

void CustomSubMixCheckModule::CommonObjectForCustomSubMix::OnPreFinalizeSoundSystem() NN_NOEXCEPT
{
    if( !m_IsNoSubMix ){
        //  SubMix の削除
        for(int i = 0; i < SubMixCount; i++)
        {
            m_SubMix[i].Finalize();
            nns::atk::Free( m_SubMixBuffer[i] );
        }
    }
}

void CustomSubMixCheckModule::CommonObjectForCustomSubMix::PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT
{
    nn::atk::SoundStartable::StartInfo info;

    if ( m_IsUseDefaultOutputReceiver )
    {
        // デフォルト出力先を使用する場合、startInfo の outputReceiver 設定をスキップする
        CommonObject::PlayWithStartSound( soundId, debugLabelName );
        return;
    }

    //  SubMix を使う場合は、ターゲット設定されている SubMix/FinalMix に出力するようにします
    if( !m_IsNoSubMix )
    {
        info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_OutputReceiver;
        if( m_IsTargetToFinalMix )
        {
            info.pOutputReceiver = m_pFinalMix;
        }
        else
        {
            info.pOutputReceiver = &m_SubMix[m_TargetSubMixIndex];
        }
    }

    CommonObject::StartParam param;
    param.pStartInfo = &info;
    CommonObject::PlayWithStartSound( soundId, debugLabelName, param );
}

void CustomSubMixCheckModule::CommonObjectForCustomSubMix::SetTargetToSubMix(int index) NN_NOEXCEPT
{
    if( m_IsNoSubMix )
    {
        //  SubMix を未使用の場合は FinalMix しか選べません
        m_IsTargetToFinalMix = true;
    }
    else
    {
        NN_SDK_ASSERT_RANGE( index, 0, SubMixCount );
        m_TargetSubMixIndex = index;
        m_IsTargetToFinalMix = false;
    }
}

void CustomSubMixCheckModule::CommonObjectForCustomSubMix::SetTargetToFinalMix() NN_NOEXCEPT
{
    m_IsTargetToFinalMix = true;
}

void CustomSubMixCheckModule::CommonObjectForCustomSubMix::PrintState(int cursorPosition) const NN_NOEXCEPT
{
    if( m_IsTargetToFinalMix )
    {
        NN_LOG("[Target SubMix]: FinalMix\n");
        NN_LOG("> No Parameter...\n");
    }
    else
    {
        const nn::atk::SubMix* pSubMix = &m_SubMix[m_TargetSubMixIndex];
        int parameterIndex = 0;
        const char* SelectPrefix = "> ";
        const char* BlankPrefix =  "  ";
        const char* prefix;

        NN_LOG("[Target SubMix]: SubMix[%d]\n", m_TargetSubMixIndex);

        prefix = parameterIndex== cursorPosition ? SelectPrefix : BlankPrefix;
        NN_LOG("%s SubMix Vol : %f\n", prefix, pSubMix->GetSubMixVolume() );
        parameterIndex++;

        prefix = parameterIndex== cursorPosition ? SelectPrefix : BlankPrefix;
        if( m_TargetSubMixIndex == 0 )
        {
            //  現状 m_SubMix[0] だけ接続先のバスを変更できます
            NN_LOG("%s DestinationBusIndex : %d\n", prefix, m_SubMixDestinationBusIndex );
        }
        else
        {
            NN_LOG("%s DestinationBusIndex : %d\n", prefix, m_SubMixDestinationBusIndex );
        }
        parameterIndex++;

        for(int bus = 0; bus < pSubMix->GetBusCount(); bus++)
        {
            prefix = parameterIndex== cursorPosition ? SelectPrefix : BlankPrefix;
            NN_LOG("%s Bus[%d] Vol : %f\n", prefix, bus, pSubMix->GetBusVolume( bus ) );
            parameterIndex++;
        }

        for(int channel = 0; channel < pSubMix->GetChannelCount(); channel++)
        {
            prefix = parameterIndex== cursorPosition ? SelectPrefix : BlankPrefix;
            NN_LOG("%s Channel[%d] Vol : %f\n", prefix, channel, pSubMix->GetChannelVolume( channel ) );
            parameterIndex++;
        }
    }
}

int CustomSubMixCheckModule::CommonObjectForCustomSubMix::GetParameterCount() const NN_NOEXCEPT
{
    if( m_IsTargetToFinalMix )
    {
        return 1;
    }
    else
    {
        //  固定のパラメータ数です。現状 "SubMix Vol", "DestinationBusIndex" の 2 つのみです
        const int FixedParameterCount = 2;

        const nn::atk::SubMix* pSubMix = &m_SubMix[m_TargetSubMixIndex];
        return FixedParameterCount + pSubMix->GetBusCount() + pSubMix->GetChannelCount();
    }
}

void CustomSubMixCheckModule::CommonObjectForCustomSubMix::ChangeParameter(int cursorPosition, int difference) NN_NOEXCEPT
{
    if( m_IsTargetToFinalMix )
    {
        return ;
    }

    nn::atk::SubMix* pSubMix = &m_SubMix[m_TargetSubMixIndex];
    int parameterIndex = 0;    //  カーソルの位置のパラメータを変更するために数えます
    const int FadeFrame = 3;
    const float VolumeUnit = 0.1f;  //  Volume の変更単位です

    //  SubMix Vol
    if( parameterIndex++ == cursorPosition )
    {
        const float current = pSubMix->GetSubMixVolume();
        pSubMix->SetSubMixVolume( current + difference * VolumeUnit, FadeFrame );
    }

    //  DestinationBusIndex
    if( parameterIndex++ == cursorPosition )
    {
        if( m_TargetSubMixIndex == 0 )
        {
            //  現状 m_SubMix[0] だけ接続先のバスを変更できます
            const int dstBusCount = m_SubMix[ m_SubMixDestinationSubMixIndex ].GetBusCount();
            const int nextBusIndex = ( m_SubMixDestinationBusIndex + difference + dstBusCount ) % dstBusCount;

            const int srcBusIndex = 0;
            m_SubMix[0].SetSend( srcBusIndex, m_SubMixDestinationBusIndex, 0.0f );
            m_SubMix[0].SetSend( srcBusIndex, nextBusIndex, 1.0f );
            m_SubMixDestinationBusIndex = nextBusIndex;
        }
    }

    //  Bus Vol
    for(int bus = 0; bus < pSubMix->GetBusCount(); bus++)
    {
        if( parameterIndex++ == cursorPosition )
        {
            const float current = pSubMix->GetBusVolume( bus );
            pSubMix->SetBusVolume( bus, current + difference * VolumeUnit, FadeFrame );
        }
    }

    //  Channel Vol
    for(int channel = 0; channel < pSubMix->GetChannelCount(); channel++)
    {
        if( parameterIndex++ == cursorPosition )
        {
            const float current = pSubMix->GetChannelVolume( channel );
            pSubMix->SetChannelVolume( channel, current + difference * VolumeUnit, FadeFrame );
        }
    }
}

void CustomSubMixCheckModule::OnInitializeAtk() NN_NOEXCEPT
{
    CommonObject::InitializeParam param;
    param.SetFlagList( &GetLocalFlagList() );
    m_CommonObject.Initialize( param );

    m_CursorPosition = 0;
    m_TargetIndex = 0;
}

void CustomSubMixCheckModule::OnFinalizeAtk() NN_NOEXCEPT
{
    m_CommonObject.Finalize();
}

void CustomSubMixCheckModule::OnLoadData() NN_NOEXCEPT
{
    m_CommonObject.LoadData();
}

void CustomSubMixCheckModule::OnPrintUsage() NN_NOEXCEPT
{
    m_CommonObject.PrintUsage();

    NN_LOG( "[Up]           Select SubMix Parameter\n" );
    NN_LOG( "[Down]         Select SubMix Parameter\n" );
    NN_LOG( "[Left]         Change SubMix Parameter\n" );
    NN_LOG( "[Right]        Change SubMix Parameter\n" );
    NN_LOG( "[R+Left]       Change Target SubMix\n" );
    NN_LOG( "[R+Right]      Change Target SubMix\n" );
}

void CustomSubMixCheckModule::OnUpdateInput() NN_NOEXCEPT
{
    bool showState = false;
    if( m_CommonObject.UpdateInput() > 0 )
    {
        showState = true;
    }

    if( nns::atk::IsTrigger( nns::atk::SamplePad::Button_Up ) )
    {
        const int count = m_CommonObject.GetParameterCount();
        if( m_CursorPosition == 0 )
        {
            m_CursorPosition = count - 1;
        }
        else
        {
            m_CursorPosition--;
        }
        showState = true;
    }
    else if( nns::atk::IsTrigger( nns::atk::SamplePad::Button_Down ) )
    {
        const int count = m_CommonObject.GetParameterCount();
        if( m_CursorPosition == count - 1 )
        {
            m_CursorPosition = 0;
        }
        else
        {
            m_CursorPosition++;
        }
        showState = true;
    }

    if( nns::atk::IsHold( nns::atk::SamplePad::Button_R ) )
    {
        //  処理の簡素化のために FinalMix を指定する定数を用意します
        const int FinalMixIndex = CustomSubMixCheckModule::CommonObjectForCustomSubMix::SubMixCount;
        const int currentIndex = m_TargetIndex;

        if( nns::atk::IsTrigger( nns::atk::SamplePad::Button_Right ) )
        {
            m_TargetIndex++;
            if( m_TargetIndex > FinalMixIndex )
            {
                m_TargetIndex = 0;
            }
        }
        else if( nns::atk::IsTrigger( nns::atk::SamplePad::Button_Left ) )
        {
            m_TargetIndex--;
            if( m_TargetIndex < 0 )
            {
                m_TargetIndex = FinalMixIndex;
            }
        }

        if( currentIndex != m_TargetIndex )
        {
            if( m_TargetIndex == FinalMixIndex )
            {
                m_CommonObject.SetTargetToFinalMix();
            }
            else
            {
                m_CommonObject.SetTargetToSubMix( m_TargetIndex );
            }

            //  カーソルがパラメータ数を超えていた場合はクランプします
            const int count = m_CommonObject.GetParameterCount();
            if( m_CursorPosition >= count )
            {
                m_CursorPosition= count - 1;
            }
            showState = true;
        }
    }
    else
    {
        if( nns::atk::IsTrigger( nns::atk::SamplePad::Button_Right ) )
        {
            m_CommonObject.ChangeParameter( m_CursorPosition, 1 );
            showState = true;
        }
        else if( nns::atk::IsTrigger( nns::atk::SamplePad::Button_Left ) )
        {
            m_CommonObject.ChangeParameter( m_CursorPosition, -1 );
            showState = true;
        }
    }

    if( showState )
    {
        m_CommonObject.PrintState( m_CursorPosition );
    }
}

void CustomSubMixCheckModule::OnUpdateAtk() NN_NOEXCEPT
{
    m_CommonObject.Update();
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void CustomSubMixCheckModule::OnUpdateDraw() NN_NOEXCEPT
{
    m_CommonObject.UpdateDraw(GetModuleName());
}
#endif

FlagList& CustomSubMixCheckModule::GetLocalFlagList() NN_NOEXCEPT
{
    return g_LocalFlagList;
}
