﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "MultiAudioEngine.h"

#include "../FlagList.h"
#include "../GfxCode/DebugViewer.h"

namespace
{
    FlagList g_LocalFlagList(nullptr, 0);

    // メモリプール用ヒープ
    const int MemoryPoolHeapSize = 32 * 1024 * 1024;
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_PoolHeapMemory[MemoryPoolHeapSize];
    nn::mem::StandardAllocator g_PoolHeapAllocator;
    nn::audio::MemoryPoolType g_MemoryPool;
}

void MultiAudioEngineCheckModule::OnInitializeAtk() NN_NOEXCEPT
{
    g_PoolHeapAllocator.Initialize(g_PoolHeapMemory, sizeof(g_PoolHeapMemory));

    // AudioEngine の初期化
    nn::atk2::AudioEngine::InitArg initArg;
    nn::atk2::AudioEngine::PlatformInitArg platformInitArg;
    m_AudioEngine.GetDefaultInitArg(initArg, platformInitArg);

    const size_t bufferSizeForAudioEngine = m_AudioEngine.GetRequiredBufferSize(initArg, platformInitArg);
    m_BufferForAudioEngine = nns::atk::Allocate( bufferSizeForAudioEngine, nn::os::MemoryPageSize );
    initArg.workBuffer = m_BufferForAudioEngine;
    initArg.workBufferSize = bufferSizeForAudioEngine;

    const size_t platformBufferSizeForAudioEngine = m_AudioEngine.GetRequiredPlatformBufferSize(initArg, platformInitArg);
    m_PlatformBufferForAudioEngine = nns::atk::Allocate( platformBufferSizeForAudioEngine );
    platformInitArg.workBuffer = m_PlatformBufferForAudioEngine;
    platformInitArg.workBufferSize = platformBufferSizeForAudioEngine;

    const size_t bufferSizeMemoryPoolForAudioEngine = m_AudioEngine.GetRequiredBufferSizeForMemoryPool(initArg, platformInitArg);
    m_BufferMemoryPoolForAudioEngine = nns::atk::Allocate( bufferSizeMemoryPoolForAudioEngine, nn::audio::MemoryPoolType::AddressAlignment );
    platformInitArg.workBufferForMemoryPool = m_BufferMemoryPoolForAudioEngine;
    platformInitArg.workBufferSizeForMemoryPool = bufferSizeMemoryPoolForAudioEngine;

    nn::atk2::AudioEngine::Result result = m_AudioEngine.Initialize(initArg, platformInitArg);
    if (result != nn::atk2::AudioEngine::Result_Success)
    {
        NN_ABORT_UNLESS( false, "Cannot initialize AudioEngine" );
    }

    // 専用のヒープをメモリプールにアタッチ
    nn::audio::AcquireMemoryPool( &m_AudioEngine.GetRendererManager().GetAudioRendererConfig(), &g_MemoryPool, g_PoolHeapMemory, MemoryPoolHeapSize );
    nn::audio::RequestAttachMemoryPool(&g_MemoryPool);

    m_CommonObject.Initialize();
}

void MultiAudioEngineCheckModule::OnFinalizeAtk() NN_NOEXCEPT
{
    m_CommonObject.Finalize();
    m_AudioEngine.Finalize();
    g_PoolHeapAllocator.Finalize();
}

void MultiAudioEngineCheckModule::OnLoadData() NN_NOEXCEPT
{
    m_CommonObject.LoadData();
}

void MultiAudioEngineCheckModule::OnPrintUsage() NN_NOEXCEPT
{
    m_CommonObject.PrintUsage();

    NN_LOG("[Up]           Play Voice\n");
    NN_LOG("[Down]         Stop Voice\n");
    NN_LOG("[Left]         Free Voice\n");
    NN_LOG("[Right]        Alloc Voice\n");
}

void MultiAudioEngineCheckModule::OnUpdateInput() NN_NOEXCEPT
{
    m_CommonObject.UpdateInput();

    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >() )
    {
        if (m_pRendererVoice != nullptr)
        {
            m_pRendererVoice->SetVoiceState(nn::atk2::detail::VoiceState_Stop);
            NN_LOG("Set stop state[0x%08x].\n", m_pRendererVoice);
        }
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >() )
    {
        if (m_pRendererVoice != nullptr)
        {
            m_AudioEngine.GetRendererManager().Lock();
            m_AudioEngine.GetRendererVoiceManager().Lock();
            m_pRendererVoice->AppendWaveBufferList(&m_WaveBuffer);
            m_pRendererVoice->SetVoiceState(nn::atk2::detail::VoiceState_Play);
            m_AudioEngine.GetRendererVoiceManager().Unlock();
            m_AudioEngine.GetRendererManager().Unlock();
            NN_LOG("Set play state[0x%08x].\n", m_pRendererVoice);
        }
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
    {
        if (m_pRendererVoice != nullptr)
        {
            m_AudioEngine.GetRendererVoiceManager().FreeVoice(m_pRendererVoice);
            NN_LOG("Free renderer voice.\n");
        }
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
    {
        if (m_pRendererVoice == nullptr)
        {
            m_AudioEngine.GetRendererManager().Lock();
            m_AudioEngine.GetRendererVoiceManager().Lock();

            m_pRendererVoice = m_AudioEngine.GetRendererVoiceManager().AllocVoice();
            m_pRendererVoice->SetSampleRate(48000);
            m_pRendererVoice->SetSampleFormat(nn::atk2::detail::SampleFormat_PcmS16);
            m_pRendererVoice->AcquireVoiceSlot();
            m_pRendererVoice->SetVoicePriority(0);
            m_pRendererVoice->SetVoiceVolume(0.3f);
            m_pRendererVoice->SetVoicePitch(1.0f);
            m_pRendererVoice->SetVoiceBiquadFilterParameter(0);
            m_pRendererVoice->SetOutputReceiver(&m_CommonObject.GetAudioEngine().GetRendererManager().GetFinalMix());
            m_pRendererVoice->SetVoiceMixVolume(0.5f, 0, 0);
            m_pRendererVoice->SetVoiceMixVolume(0.5f, 0, 1);

            int sampleCount = 48000;
            size_t bufferSize = sampleCount * sizeof(int16_t);
            void* buffer = g_PoolHeapAllocator.Allocate(bufferSize, nn::atk2::detail::RendererManager::BufferAlignSize);
            int16_t* sampleArray = reinterpret_cast<int16_t*>(buffer);
            const float Pi = 3.1415926535897932384626433f;
            for (auto i = 0; i < sampleCount; ++i)
            {
                sampleArray[i] = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(2 * Pi * 440 * i / 48000));
            }

            m_WaveBuffer.bufferAddress = buffer;
            m_WaveBuffer.bufferSize = bufferSize;
            m_WaveBuffer.loopFlag = true;
            m_WaveBuffer.sampleLength = sampleCount;
            m_WaveBuffer.sampleOffset = 0;

            NN_LOG("Alloc renderer voice[0x%08x].\n", m_pRendererVoice);

            m_AudioEngine.GetRendererVoiceManager().Unlock();
            m_AudioEngine.GetRendererManager().Unlock();
        }
    }
}

void MultiAudioEngineCheckModule::OnUpdateAtk() NN_NOEXCEPT
{
    nn::atk2::AudioEngine::Result result = m_AudioEngine.Update();
    if (result != nn::atk2::AudioEngine::Result_Success)
    {
        NN_ASSERT(false, "Failed to update Audio Engine.");
    }
    m_CommonObject.Update();
}

void MultiAudioEngineCheckModule::OnUpdateDraw() NN_NOEXCEPT
{
    m_CommonObject.UpdateDraw(GetModuleName());
}

const char* MultiAudioEngineCheckModule::GetModuleName() const NN_NOEXCEPT
{
    return "MultiAudioEngine";
}

FlagList& MultiAudioEngineCheckModule::GetLocalFlagList() NN_NOEXCEPT
{
    return g_LocalFlagList;
}
