﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk.h>
#include <nn/htcs.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/settings/settings_DebugPad.h>

#include <nn/atk/viewer/atk_SoundEdit.h>

#include "nns/atk/atk_SampleCommon.h"
#include "ModuleSelector.h"
#include "CheckCode/ICheckModule.h"
#include "FlagList.h"

#include "CheckCode/Simple.h"
#include "CheckCode/Voice.h"
#include "CheckCode/RendererVoice.h"
#include "CheckCode/MultiAudioEngine.h"
#include "CheckCode/CustomSubmix.h"
#include "CheckCode/EditGlobalFlag.h"

#include "GfxCode/DebugViewer.h"

namespace
{
    const char Title[] = "Atk2Sandbox";

    ModuleSelector g_ModuleSelector;

    SimpleCheckModule g_SimpleCheckModule;
    VoiceCheckModule g_VoiceCheckModule;
    RendererVoiceCheckModule g_RendererVoiceCheckModule;
    MultiAudioEngineCheckModule g_MultiAudioEngineCheckModule;
    CustomSubmixCheckModule g_CustomSubmixCheckModule;
    EditGlobalFlagModule g_EditGlobalFlagModule;

    ICheckModule* g_pCurrentCheckModule;

    ICheckModule* g_pModuleList[] =
    {
        &g_SimpleCheckModule,
        &g_VoiceCheckModule,
        &g_RendererVoiceCheckModule,
        &g_MultiAudioEngineCheckModule,
        &g_CustomSubmixCheckModule,
        &g_EditGlobalFlagModule,
    };

    const int ModuleListCount = sizeof(g_pModuleList) / sizeof(g_pModuleList[0]);

    // ここを変更した場合には、FlagList.h に定義されている。GlobalFlagType も同時に修正してください。
    FlagElement g_GlobalElements[] =
    {
        { "DisableAudioThread", false },
        { "ManualAudioFrameUpdate", false },
    };

    FlagList g_GlobalFlagList(g_GlobalElements, sizeof(g_GlobalElements) / sizeof(g_GlobalElements[0]));

    // 操作説明のデフォルトの文字列描画位置
    const int DefaultPrintUsagePosX = 800;
    const int DefaultPrintUsagePosY = 150;
}

void InitializeHid() NN_NOEXCEPT
{
    nns::atk::InitializeHidDevices();

    // 追加の必要なキーボードのキー入力を DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonZR = nn::hid::KeyboardKey::Z::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void FinalizeHid() NN_NOEXCEPT
{
}

void Initialize() NN_NOEXCEPT
{
    nns::atk::InitializeHeap();
    nns::atk::InitializeFileSystem();

    InitializeHid();

    const nn::Result result = nn::fs::MountHostRoot();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::htcs::Initialize(nns::atk::Allocate, nns::atk::Free);

    AttachGlobalFlagList(&g_GlobalFlagList);

    DebugViewer::GetInstance().Initialize();
}

void Finalize() NN_NOEXCEPT
{
    DebugViewer::GetInstance().Finalize();

    DetachGlobalFlagList();

    nn::htcs::Finalize();
    nn::fs::UnmountHostRoot();

    FinalizeHid();

    nns::atk::FinalizeFileSystem();
    nns::atk::FinalizeHeap();
}

void InitializeAtk() NN_NOEXCEPT
{
    g_pCurrentCheckModule->OnInitializeAtk();
}

void FinalizeAtk() NN_NOEXCEPT
{
    g_pCurrentCheckModule->OnFinalizeAtk();
}

void LoadData() NN_NOEXCEPT
{
    g_pCurrentCheckModule->OnLoadData();
}


void PrintUsage() NN_NOEXCEPT
{
    const char* moduleName = g_pCurrentCheckModule->GetModuleName();
    NN_LOG("---------------------------------------------------\n");
    NN_LOG("%s(%s)\n", Title, moduleName);
    NN_LOG("---------------------------------------------------\n");
    g_pCurrentCheckModule->OnPrintUsage();
    NN_LOG("[L]            Print Usage\n");
    NN_LOG("[R] + [Start][Space] Exit Application\n");
    NN_LOG("---------------------------------------------------\n");

    g_pCurrentCheckModule->OnPostPrintUsage();
}

bool UpdateInput() NN_NOEXCEPT
{
    g_pCurrentCheckModule->OnUpdateInput();

    // PrintUsage
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::L >() )
    {
        PrintUsage();
    }

    // Exit
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Start >() && nns::atk::IsHold< ::nn::hid::DebugPadButton::R >() )
    {
        return false;
    }

    return true;
}

void UpdateAtk() NN_NOEXCEPT
{
    g_pCurrentCheckModule->OnUpdateAtk();
}

void UpdateDraw() NN_NOEXCEPT
{
    g_pCurrentCheckModule->OnUpdateDraw();

    DebugPrint("Sandbox_Usage", DefaultPrintUsagePosX, DefaultPrintUsagePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY,
        "\n"
        "[L]          :Print Usage\n"
        "[Start/Space]:Exit Application\n"
    );
}

extern "C" void nnMain() NN_NOEXCEPT
{
    Initialize();

    for (;;)
    {
        g_pCurrentCheckModule = g_ModuleSelector.Select(g_pModuleList, ModuleListCount);
        if (g_pCurrentCheckModule == nullptr)
        {
            break;
        }

        DebugViewer::GetInstance().Reset();

        InitializeAtk();

        LoadData();

        PrintUsage();

        for (;;)
        {
            nns::atk::UpdateHidDevices();

            if ( !UpdateInput() )
            {
                break;
            }
            UpdateAtk();
            UpdateDraw();

            DebugViewer::GetInstance().Update();
            DebugViewer::GetInstance().Present();

            // Vsync の代わり
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }

        FinalizeAtk();
        DebugViewer::GetInstance().Reset();
    }

    Finalize();
}

