﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{AccountApplicationAuthorization.cpp,PageSampleAccountApplicationAuthorization}

    @brief @copybrief PageSampleAccountApplicationAuthorization
 */

/**
    @page PageSampleAccountApplicationAuthorization ニンテンドーアカウントとのOAuth 2.0/OpenID Connectのための認可取得機能のアプリケーションでの利用

    @tableofcontents

    @brief アカウントライブラリを使用して、ニンテンドーアカウントサーバーとのOAuth 2.0/OpenID Connectのための認可コードを取得する方法を示します。

    @section PageSampleAccountApplicationAuthorization_SectionBrief 概要
    アカウントライブラリを使用して、ニンテンドーアカウントサーバーとのOAuth 2.0/OpenID Connectのための認可コードを取得する方法を示します。

    このサンプルプログラムで示す処理は、ニンテンドーアカウントサーバーとOAuth 2.0/OpenID Connectを行う場合にのみ必要です。
    アプリケーションがネットワークサービスアカウントの利用に限る場合、本プログラムを参照する必要はありません。

    @section PageSampleAccountApplicationAuthorization_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/AccountApplicationAuthorization
    Samples/Sources/Applications/AccountApplicationAuthorization @endlink 以下にあります。

    @section PageSampleAccountApplicationAuthorization_SectionNecessaryEnvironment 必要な環境
    本サンプルプログラムは開発機向けにのみビルドし実行することができます。
    Windows 環境をサポートしていないことに注意してください。

    @section PageSampleAccountApplicationAuthorization_SectionHowToOperate 操作方法
    一般的なサンプルプログラムと同様に本プログラムをビルドし、実行してください。
    このプログラムの実行の経過は画面への文字の描画、及び実行ログとして出力されます。

    @section PageSampleAccountApplicationAuthorization_SectionDetail 解説
    このサンプルプログラムは次の順序で処理を行います。

    - ユーザーアカウントの選択
    - ユーザーアカウントの Open
    - ニンテンドーアカウントサーバーからの認可コードの取得
    - ユーザーアカウントの Close

    下記ソースコードの随所に補足説明を記載していますので、詳細はそちらを参照してください。

    AccountApplicationAuthorization.cpp
    @includelineno AccountApplicationAuthorization.cpp

    実行の結果、次のようなログが出力されます。

    AccountApplicationAuthorization_ExampleOutput.txt
    @verbinclude AccountApplicationAuthorization_ExampleOutput.txt
 */

#include "AccountApplicationAuthorization.h"

#include <cstdlib>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_NintendoAccountAuthorizationRequest.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_Selector.h>

#include <nn/nn_Assert.h>
#include <nn/err.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiNetworkConnection.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Tick.h>
#include <nn/util/util_ScopeExit.h>

namespace
{
NN_ALIGNAS(4096) char g_AuthorizationBuffer[nn::account::RequiredBufferSizeForNintendoAccountAuthorizationRequestContext];

} // 無名名前空間終わり

// アプリケーションのメイン
void Program::Execute() NN_NOEXCEPT
{
    // 依存するモジュールの初期化
    nn::nifm::Initialize();

    // アカウントライブラリの初期化
    nn::account::Initialize();

    // ユーザーアカウントの選択画面の表示
    // - ユーザー操作によるキャンセル時には nn::account::ResultCancelledByUser が返る。
    // - このサンプルでは、ネットワークサービスアカウントを要求してユーザーアカウント選択を行う。
    nn::account::Uid uid;
    nn::account::UserSelectionSettings cfg = nn::account::DefaultUserSelectionSettings;
    cfg.isNetworkServiceAccountRequired = true;
    auto r = nn::account::ShowUserSelector(&uid, cfg);
    if (nn::account::ResultCancelledByUser::Includes(r))
    {
        Printfln("- User selection cancelled");
        return;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(r);

    // アプリケーションのユーザー向けシーンを開始
    SceneSingleUser(uid);
}

// アプリケーションのユーザー向けシーン
void Program::SceneSingleUser(const nn::account::Uid& uid) NN_NOEXCEPT
{
    // 選択されたユーザーを Open
    nn::account::UserHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle, uid));
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::CloseUser(handle);
    };

    // シングルユーザーのオンラインモードを想定した処理を開始
    SceneSingleUserWithNetwork(handle);
}

// シングルユーザーのオンラインモードを想定した処理
void Program::SceneSingleUserWithNetwork(const nn::account::UserHandle& handle) NN_NOEXCEPT
{
    // ネットワーク接続要求
    while (NN_STATIC_CONDITION(true))
    {
        nn::nifm::SubmitNetworkRequestAndWait();
        if (!nn::nifm::IsNetworkAvailable())
        {
            auto r = nn::nifm::HandleNetworkRequestResult();
            if (!r.IsSuccess())
            {
                if (!nn::nifm::ResultErrorHandlingCompleted::Includes(r))
                {
                    // エラーかつ、それを解消できなかった場合はシーンを終了する。
                    return;
                }
                // エラーかつ、それを解消できた場合はリトライする。
                continue;
            }
            // break
        }
        // ネットワーク接続を利用できる。
        break;
    }

    // ニンテンドーアカウントの認可取得のためのパラメータ
    nn::account::NintendoAccountAuthorizationRequestParameters param =  {
        "user.basic openid",    // このパラメータは、アプリケーションがニンテンドーアカウントにどのような権限を要求するかを定める。
        "example_state",        // このパラメータは、認証,認可取得のセッションでの中間者攻撃を防止するために用いることができる。
        "example_nonce"         // このパラメータはIDトークンに含まれるため、通信の改ざんをサーバーサイドで検証できる。
    };

    nn::account::NintendoAccountAuthorizationRequestContext ctx;
    auto durationMsec = -1ll;
    while (NN_STATIC_CONDITION(true))
    {
        // このスコープの処理時間を測る
        {
            auto begin = nn::os::GetSystemTick();
            NN_UTIL_SCOPE_EXIT
            {
                auto end = nn::os::GetSystemTick();
            durationMsec = (end - begin).ToTimeSpan().GetMilliSeconds();
            };

            // ニンテンドーアカウントへの認可取得リクエストを行う
            // - ctx.Cancel() を呼ぶと、可能な限り早くイベントがシグナルされる。
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::CreateNintendoAccountAuthorizationRequest(
                &ctx, handle, param, g_AuthorizationBuffer, sizeof(g_AuthorizationBuffer)));
            nn::os::SystemEvent e;
            NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
            e.Wait();
        }

        // リクエストの結果を取得し、必要であれば本体システムのUIを表示してユーザーに操作を要求する。
        auto r = ctx.GetResultWithInteractionIfRequired();
        if (!r.IsSuccess())
        {
            if (nn::account::ResultCancelledByUser::Includes(r))
            {
                // 本体システムのUI上でユーザーがキャンセル操作を行った。
                Printfln("- NintendoAccountAuthorizationRequestContext cancelled");
                return;
            }
            else if (nn::account::ResultNetworkCommunicationError::Includes(r))
            {
                // ログイン時に通信関係のエラーが発生
                // - これにはアプリケーションの更新が必要な場合や、本体更新が必要な場合も含まれる。
                // - nn::error::ShowError() に渡し、詳細なエラー内容をユーザーに提示する必要がある。
                nn::err::ShowError(r);
                return;
            }
            // ctx.Cancel() が呼ばれていると、ここで nn::account::ResultCancelled() が返る場合がある。
            NN_ABORT_UNLESS_RESULT_SUCCESS(r);
        }

        // 成功した場合、 state パラメータが最初に指定したものと一致するかを念のために確認する。
        // - これは必要な操作ではなく、 nonce を指定していればサーバー側での検証のみでよい。
        size_t stateLength;
        char state[128];
        NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetState(&stateLength, state, sizeof(state)));
        NN_ABORT_UNLESS(strnlen(param.state, sizeof(param.state)) == stateLength, "Unexpected state used");
        NN_ABORT_UNLESS(std::strncmp(state, param.state, stateLength) == 0, "Unexpected state used");
        break;
    }
    Printfln("- Nintendo Account authorization turnaround: %lld msec", durationMsec);

    // ニンテンドーアカウントの認可コードとIDトークンを取得する
    auto* code = reinterpret_cast<char*>(std::malloc(nn::account::NintendoAccountAuthorizationCodeLengthMax));
    NN_UTIL_SCOPE_EXIT
    {
        std::free(code);
    };
    size_t codeLength;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetAuthorizationCode(&codeLength, code, nn::account::NintendoAccountAuthorizationCodeLengthMax));
    Printfln("- Nintendo Account authorization code: %.*s", codeLength, code);

    auto* idToken = reinterpret_cast<char*>(std::malloc(nn::account::NintendoAccountIdTokenLengthMax));
    NN_UTIL_SCOPE_EXIT
    {
        std::free(idToken);
    };
    size_t idTokenLength;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetIdToken(&idTokenLength, idToken, nn::account::NintendoAccountIdTokenLengthMax));
    Printfln("- Nintendo Account ID Token: %.*s", idTokenLength, idToken);

    // :
    // その他オンライン状態で実行可能なアプリケーション実装
    // :
}
