﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <common.glsl>

//--------------------------------------------------------------------------------------------------
// Uniform Blocks
//--------------------------------------------------------------------------------------------------

#define DISPLAY_VERTEX_NORMAL       (0)     // @@ id="display_vertex_normal" choice="bool" default="0" type="dynamic"
#define DISPLAY_BONE_WEIGHT         (0)     // @@ id="display_bone_weight" choice="bool" default="0" type="dynamic"

layout(std140) uniform Option   // @@ id="opt" type="option"
{
    //------------------------------------------------------------------------------------
    // NOTE:　ここに "display_vertex_normal" のオプション変数を定義すると、
    // バッチコンパイルが有効になるため、ジオメトリシェーダーの input がマクロで分岐できない。
    //------------------------------------------------------------------------------------
    int displayBoneWeight;      // @@ id="display_bone_weight"
};

layout(std140) uniform Skeleton // @@ id="skel" type="skeleton"
{
    vec4 mtxPalette[3 * 128]; // RADEON では配列のサイズを超えてアクセスできない。
};

layout(std140) uniform Shape // @@ id="shape" type="shape"
{
    vec4 shapeMtx[3];
    int vtxSkinCount;
    int padding[3];
};

layout(std140) uniform View // @@ id="view"
{
    vec4 projMtx[4];
    vec4 cameraMtx[3];
};

layout(std140) uniform SelectedInfo // @@ id="selectedInfo"
{
    ivec2 selectedBoneIndex;       // x: スムーススキン y: リジッドスキン
};

//--------------------------------------------------------------------------------------------------
// Input / Output
//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_VERTEX_SHADER
layout(location = 0)  in vec4   i_Position;     // @@ id="_p0"   hint="position0"
layout(location = 1)  in vec3   i_Normal;       // @@ id="_n0"   hint="normal0"
layout(location = 2)  in vec4   i_Tangent;      // @@ id="_t0"   hint="tangent0"
layout(location = 3)  in ivec4  i_Index0;       // @@ id="_i0"   hint="blendindex0"
layout(location = 4)  in ivec4  i_Index1;       // @@ id="_i1"   hint="blendindex1"
layout(location = 5)  in vec4   i_Weight0;      // @@ id="_w0"   hint="blendweight0"
layout(location = 6)  in vec4   i_Weight1;      // @@ id="_w1"   hint="blendweight1"
#endif // NN_G3D_VERTEX_SHADER

// VS Output / GS Input
#ifndef NN_G3D_FRAGMENT_SHADER
NN_G3D_PRIMITIVE
{
    vec3 worldPosition;
    vec3 worldNormal;
    vec3 worldTangent;
    vec4 color;
} nw_primitive;
#endif // !NN_G3D_FRAGMENT_SHADER

#ifdef NN_G3D_GEOMETRY_SHADER
#define INPUT_VERTEX_COUNT (3)
#define CREATE_VERTEX_COUNT (INPUT_VERTEX_COUNT * 4 * 3)    // 頂点数 * 四角形(4 頂点) * (N / T / B)

layout(triangles) in;
#if DISPLAY_VERTEX_NORMAL
layout(triangle_strip, max_vertices = INPUT_VERTEX_COUNT + CREATE_VERTEX_COUNT) out;
#else
// TODO: NV_geometry_shader_passthrough
layout(triangle_strip, max_vertices = INPUT_VERTEX_COUNT) out;
#endif // DISPLAY_VERTEX_NORMAL

#endif // NN_G3D_GEOMETRY_SHADER

// GS Output / FS Input
#ifndef NN_G3D_VERTEX_SHADER
NN_G3D_RASTER
{
    vec4 color;
} nn_g3d_raster;
#endif // !NN_G3D_VERTEX_SHADER

#ifdef NN_G3D_FRAGMENT_SHADER
layout(location = 0)    out vec4 o_Color;
#endif // NN_G3D_FRAGMENT_SHADER

//--------------------------------------------------------------------------------------------------
// Vertex Shader
//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_VERTEX_SHADER

void Skining(out vec4 worldPosition, out vec3 worldNormal, out vec3 worldTangent)
{
    worldPosition = vec4(0, 0, 0, 1);
    worldNormal = vec3(0, 0, 0);
    worldTangent = vec3(0, 0, 0);

    if (vtxSkinCount == 0)
    {
        // rigid body
        worldPosition.xyz   = NN_G3D_TRANSFORM_POS(shapeMtx, i_Position);
        worldNormal.xyz     = NN_G3D_TRANSFORM_VEC(shapeMtx, i_Normal);
        worldTangent.xyz    = NN_G3D_TRANSFORM_VEC(shapeMtx, i_Tangent);
    }
    else if (vtxSkinCount == 1)
    {
        // rigid skinning
        int offset = i_Index0.x * 3;
        worldPosition.xyz = NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
        worldNormal.xyz = NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
        worldTangent.xyz = NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
    }
    else if (vtxSkinCount <= 4)
    {
        // smooth skinning [1, 4]
        for (int i = 0; i < vtxSkinCount; ++i)
        {
            int offset = i_Index0[i] * 3;
            float weight = i_Weight0[i];
            worldPosition.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            worldNormal.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            worldTangent.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
        }
    }
    else if (vtxSkinCount <= 8)
    {
        // smooth skinning [5, 8]
        for (int i = 0; i < 4; ++i)
        {
            int offset = i_Index0[i] * 3;
            float weight = i_Weight0[i];
            worldPosition.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            worldNormal.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            worldTangent.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
        }
        for (int i = 4; i < vtxSkinCount; ++i)
        {
            int offset = i_Index1[i - 4] * 3;
            float weight = i_Weight1[i - 4];
            worldPosition.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            worldNormal.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            worldTangent.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
        }
    }
}

bool IsSelectedBoneIndex(int boneIndex)
{
    return selectedBoneIndex.x == boneIndex;
}

float GetSelectedBoneWeight()
{
    // rigid skinning
    if (vtxSkinCount == 1)
    {
        return (selectedBoneIndex.y == i_Index0.x) ? 1 : 0;
    }
    else if (vtxSkinCount <= 4)
    {
        // smooth skinning [1, 4]
        for (int i = 0; i < vtxSkinCount; ++i)
        {
            if(IsSelectedBoneIndex(i_Index0[i]))
            {
                return i_Weight0[i];
            }
        }
        return 0;
    }
    else if (vtxSkinCount <= 8)
    {
        // smooth skinning [5, 8]
        for (int i = 0; i < 4; ++i)
        {
            if(IsSelectedBoneIndex(i_Index0[i]))
            {
                return i_Weight0[i];
            }
        }
        for (int i = 4; i < vtxSkinCount; ++i)
        {
            if(IsSelectedBoneIndex(i_Index1[i - 4]))
            {
                return i_Weight1[i - 4];
            }
        }
        return 0;
    }
    return 0;
}

void main()
{
    vec4 worldPosition;

    OUT.color = vec4(.0);
    Skining(worldPosition, OUT.worldNormal, OUT.worldTangent);
    OUT.worldPosition = worldPosition.xyz;

    if(NN_G3D_TO_BOOL(DISPLAY_BONE_WEIGHT))
    {
        float weight = GetSelectedBoneWeight();
        OUT.color += vec4(vec3(weight), 1.0);
    }

    vec4 viewPosition = vec4(NN_G3D_TRANSFORM_POS(cameraMtx, worldPosition), 1);
    vec4 projectionPosition = NN_G3D_PROJECT(projMtx, viewPosition);
    gl_Position = projectionPosition;
}

#endif // NN_G3D_VERTEX_SHADER

//--------------------------------------------------------------------------------------------------
// Geometry Shader
//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_GEOMETRY_SHADER

void SetPositionAndEmitVertex(vec3 vertexWorldPosition)
{
    vec4 worldPosition = vec4(vertexWorldPosition, 1.0);
    vec4 viewPosition = vec4(NN_G3D_TRANSFORM_POS(cameraMtx, worldPosition), 1);
    gl_Position = NN_G3D_PROJECT(projMtx, viewPosition);
    EmitVertex();
}

void main()
{
    // そのままのポリゴンを生成
    for(int inputVertexIndex = 0; inputVertexIndex < INPUT_VERTEX_COUNT; ++inputVertexIndex)
    {
        OUT.color = IN[inputVertexIndex].color;
        SetPositionAndEmitVertex(IN[inputVertexIndex].worldPosition.xyz);
    }
    EndPrimitive();

    // 姿勢ベクトル板ポリゴンを作成
    if(NN_G3D_TO_BOOL(DISPLAY_VERTEX_NORMAL))
    {
        const float lineWidth = .03;
        const float lineLength = 1.0;
        const int vertexCount = 4;

        // 0___1
        // |  /|
        // | / |
        // 2/__3

        vec3 axisX = cameraMtx[0].xyz;
        vec3 axisY = cameraMtx[1].xyz;

        for(int inputVertexIndex = 0; inputVertexIndex < INPUT_VERTEX_COUNT; ++inputVertexIndex)
        {
            // 接線が存在するかを取得
            float lengthSq = dot(IN[inputVertexIndex].worldTangent,IN[inputVertexIndex].worldTangent);
            if(lengthSq > 1e-8)
            {
                // 接線
                {
                    OUT.color = vec4(1.0, .0, .0, 1.0);

                    float YoT = dot(axisY,IN[inputVertexIndex].worldTangent);
                    float YoT2 = YoT * YoT;

                    float XoT = dot(axisX,IN[inputVertexIndex].worldTangent);
                    float XoT2 = XoT * XoT;

                    if(XoT * YoT > .0)
                    {
                        XoT2 = -XoT2;
                    }

                    vec3 widthOffset = normalize(axisX * (YoT2) + axisY * (XoT2));
                    widthOffset *= lineWidth;
                    vec3 offsets[4] = {
                        -widthOffset,
                        widthOffset,
                        -widthOffset + IN[inputVertexIndex].worldTangent * lineLength,
                        widthOffset + IN[inputVertexIndex].worldTangent * lineLength,
                    };

                    for(int vertexIndex = 0; vertexIndex < vertexCount; ++vertexIndex)
                    {
                        vec3 worldPos = IN[inputVertexIndex].worldPosition.xyz + offsets[vertexIndex];
                        SetPositionAndEmitVertex(worldPos);
                    }
                    EndPrimitive();
                }

                // 従法線
                {
                    OUT.color = vec4(.0, 1.0, .0, 1.0);
                    vec3 worldBinormal = normalize(cross(IN[inputVertexIndex].worldNormal,IN[inputVertexIndex].worldTangent));

                    float YoB = dot(axisY, worldBinormal);
                    float YoB2 = YoB * YoB;

                    float XoB = dot(axisX, worldBinormal);
                    float XoB2 = XoB * XoB;

                    if(XoB * YoB > .0)
                    {
                        XoB2 = -XoB2;
                    }

                    vec3 widthOffset = normalize(axisX * (YoB2) + axisY * (XoB2));
                    widthOffset *= lineWidth;
                    vec3 offsets[4] = {
                        -widthOffset,
                        widthOffset,
                        -widthOffset + worldBinormal * lineLength,
                        widthOffset + worldBinormal * lineLength,
                    };

                    for(int vertexIndex = 0; vertexIndex < vertexCount; ++vertexIndex)
                    {
                        vec3 worldPos = IN[inputVertexIndex].worldPosition.xyz + offsets[vertexIndex];
                        SetPositionAndEmitVertex(worldPos);
                    }
                    EndPrimitive();
                }
            }

            // 法線
            {
                OUT.color = vec4(.0,.0,1.0,1.0);

                float YoN = dot(axisY,IN[inputVertexIndex].worldNormal);
                float YoN2 = YoN * YoN;

                float XoN = dot(axisX,IN[inputVertexIndex].worldNormal);
                float XoN2 = XoN * XoN;

                if(XoN * YoN > .0)
                {
                    XoN2 = -XoN2;
                }

                vec3 widthOffset = normalize(axisX * (YoN2) + axisY * (XoN2));
                widthOffset *= lineWidth;
                vec3 offsets[4] = {
                    -widthOffset,
                    widthOffset,
                    -widthOffset + IN[inputVertexIndex].worldNormal * lineLength,
                    widthOffset + IN[inputVertexIndex].worldNormal * lineLength,
                };

                for(int vertexIndex = 0; vertexIndex < vertexCount; ++vertexIndex)
                {
                    vec3 worldPos = IN[inputVertexIndex].worldPosition.xyz + offsets[vertexIndex];
                    SetPositionAndEmitVertex(worldPos);
                }
                EndPrimitive();
            }
        }
    }
}

#endif // NN_G3D_GEOMETRY_SHADER

//--------------------------------------------------------------------------------------------------
// Fragment Shader
//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_FRAGMENT_SHADER

void main()
{
    if(!NN_G3D_TO_BOOL(DISPLAY_BONE_WEIGHT))
    {
        if(length(IN.color.rgb) < .01)
        {
            discard;
        }
    }

    o_Color.rgba = IN.color;
}

#endif // NN_G3D_FRAGMENT_SHADER
