﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "3dSampleViewer_Viewer.h"
#include "3dSampleViewer.h"
#include "3dSampleViewer_Menu.h"

#include <nn/g3d/g3d_Viewer.h>

namespace
{
    const char* g_ShadingModelNames[] =
    {
        {"town"},
        {"basic"},
        {"visualizeInfo"}
    };
    NN_STATIC_ASSERT(NN_ARRAY_SIZE(g_ShadingModelNames) == ShaderArchiveType_Count);

    // モデルに適するシェーダーが存在しなかった場合に割り当てられるシェーダー
    nn::g3d::ResShadingModel* g_pFallbackShadingModel = nullptr;

    // デバッグ描画パスで利用するラスタライザーステート
    nn::gfx::RasterizerState g_VisializeRasterizerState;

    const nn::gfx::Buffer* g_pUniformBlocks[UniformBlockType_Count * BufferingCount];
    const char* g_UniformBlockIdArray[] = {
        {"env"},
        {"view"},
        {"context"},
        {"selectedInfo"}
    };
    NN_STATIC_ASSERT(NN_ARRAY_SIZE(g_UniformBlockIdArray) == UniformBlockType_Count);

    nns::g3d::ModelAnimObj* FindModelAnimObjFromHolder(g3ddemo::ResourceHolder* pHolder, const nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT
    {
        for (nns::g3d::ModelAnimObj* pModelAnimObj : pHolder->modelAnimObjs)
        {
            if (pModelAnimObj->GetModelObj() != pModelObj)
            {
                continue;
            }
            return pModelAnimObj;
        }
        return nullptr;
    }

    nns::g3d::RenderModelObj* FindRenderModelObjFromHolder(g3ddemo::ResourceHolder* pHolder, const nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT
    {
        for (nns::g3d::RenderModelObj* pRenderModelObj : pHolder->renderModelObjs)
        {
            if (pRenderModelObj->GetModelObj() != pModelObj)
            {
                continue;
            }
            return pRenderModelObj;
        }
        return nullptr;
    }

    // 指定のシェーディングモデルを持つシェーダーアーカイブを探します。
    nn::g3d::ResShaderArchive* FindShaderArchive(const g3ddemo::ResourceHolder* pHolder, const char* archiveName, const char* shadingModelName)
    {
        for (int shaderArchiveIndex = 0, shaderArchiveCount = pHolder->shaderFiles.GetCount(); shaderArchiveIndex < shaderArchiveCount; ++shaderArchiveIndex)
        {
            nn::g3d::ResShaderArchive* pShaderArchive = pHolder->shaderFiles[shaderArchiveIndex]->GetResShaderArchive();
            if (strcmp(pShaderArchive->GetName(), archiveName) != 0)
            {
                continue;
            }

            for (int shadingModelIndex = 0; shadingModelIndex < pShaderArchive->GetShadingModelCount(); ++shadingModelIndex)
            {
                nn::g3d::ResShadingModel* pShadingModel = pShaderArchive->GetShadingModel(shadingModelIndex);
                if (strcmp(pShadingModel->GetName(), shadingModelName) == 0)
                {
                    return pShaderArchive;
                }
            }
        }

        return nullptr;
    }

    int GetModelAnimObjIndex(const g3ddemo::ResourceHolder* pHolder, const nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT
    {
        const g3ddemo::Vector<nns::g3d::ModelAnimObj*>& modelAnimObjs = pHolder->modelAnimObjs;
        for (int modelIndex = 0, modelCount = modelAnimObjs.GetCount(); modelIndex < modelCount; ++modelIndex)
        {
            if (modelAnimObjs[modelIndex]->GetModelObj() != pModelObj)
            {
                continue;
            }
            return modelIndex;
        }

        return -1;
    }

    int GetModelAnimObjIndex(const g3ddemo::ResourceHolder* pHolder, const nns::g3d::ModelAnimObj* pModelAnimObj) NN_NOEXCEPT
    {
        return GetModelAnimObjIndex(pHolder, pModelAnimObj->GetModelObj());
    }

    void InitializeMaterialPass(
        nn::gfx::Device* pDevice,
        nns::g3d::RenderModel* pRenderModel,
        nn::g3d::ResShaderArchive** pPerMaterialResShaderArchiveArray) NN_NOEXCEPT
    {
        nn::g3d::ResModel* pResModel = pRenderModel->GetResModel();
        for (int shapeIndex = 0, shapeCount = pResModel->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            nn::g3d::ResShape* pResShape = pResModel->GetShape(shapeIndex);
            int materialIndex = pResShape->GetMaterialIndex();
            nns::g3d::RenderMaterial* pRenderMaterial = pRenderModel->GetRenderMaterial(DrawPassType_Material, materialIndex);

            // レンダーマテリアルの初期化
            if (!pRenderMaterial->IsInitialized())
            {
                nn::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(materialIndex);
                if (pPerMaterialResShaderArchiveArray[materialIndex] != nullptr)
                {
                    bool isInitialized = pRenderMaterial->Initialize(pDevice, pResMaterial, pPerMaterialResShaderArchiveArray[materialIndex]);
                    NN_UNUSED(isInitialized);
                    NN_ASSERT(isInitialized);
                }
                else
                {
                    // シェーダーアサインがないマテリアルがあった場合は強制的にフォールバックのシェーディングモデルを割り当てます。
                    // それ以外のケースでここに来る場合はバグの可能性があります。
                    NN_ASSERT_NOT_NULL(pResMaterial->GetShaderAssign());
                    NN_ASSERT_NOT_NULL(g_pFallbackShadingModel);
                    using ViewerServer = nn::g3d::viewer::ViewerServer;

                    char message[256];
                    snprintf(message, NN_ARRAY_SIZE(message), "Failed to bind shading model. %s shading model was bound forcibly.", g_pFallbackShadingModel->GetName());
                    ViewerServer::SendUserMessageArg userMessageArg(
                        message,
                        ViewerServer::SendUserMessageArg::MessageType_Error,
                        ViewerServer::SendUserMessageArg::MessageDestination_Log);
                    ViewerServer::GetInstance().QueueSendUserMessageCommand(userMessageArg);

                    pRenderMaterial->Initialize(pDevice, pResMaterial, g_pFallbackShadingModel);
                }
            }

            // レンダーシェイプの初期化
            nns::g3d::RenderShape* pRenderShape = pRenderModel->GetRenderShape(DrawPassType_Material, shapeIndex);
            pRenderShape->Initialize(pDevice, pResShape, pRenderMaterial);
        }
    }

    // 描画情報を取得し、モデルに反映させる
    void SetupRenderStateFromRenderInfo(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT
    {
        const char* renderStateInfoNames[] = {
            {"blend"},
            {"culling"}
        };

        using SetRenderStateFunc = void(*)(nns::g3d::RenderMaterial* pRenderMaterial, const nn::g3d::ResRenderInfo* pRenderInfo);
        SetRenderStateFunc SetRenderState[] = {
            [](nns::g3d::RenderMaterial* pRenderMaterial, const nn::g3d::ResRenderInfo* pRenderInfo) { pRenderMaterial->SetBlendState(g3ddemo::GetBlendState(*pRenderInfo->GetInt())); },
            [](nns::g3d::RenderMaterial* pRenderMaterial, const nn::g3d::ResRenderInfo* pRenderInfo) { pRenderMaterial->SetRasterizerState(g3ddemo::GetRasterizerState(*pRenderInfo->GetInt())); }
        };

        nns::g3d::RenderModel* pRenderModel = pRenderModelObj->GetRenderModel();
        int materialCount = pRenderModel->GetResModel()->GetMaterialCount();
        for (int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
        {
            nns::g3d::RenderMaterial* pRenderMaterial = pRenderModel->GetRenderMaterial(materialIndex);
            const nn::g3d::ResMaterial* pResMaterial = pRenderMaterial->GetResMaterial();
            for (int infoIndex = 0, infoCount = pResMaterial->GetRenderInfoCount(); infoIndex < infoCount; ++infoIndex)
            {
                const nn::g3d::ResRenderInfo* pResRenderInfo = pResMaterial->GetRenderInfo(infoIndex);
                for (int infoNameIndex = 0; infoNameIndex < NN_ARRAY_SIZE(renderStateInfoNames); ++infoNameIndex)
                {
                    if (strcmp(pResRenderInfo->GetName(), renderStateInfoNames[infoNameIndex]) != 0)
                    {
                        continue;
                    }
                    SetRenderState[infoNameIndex](pRenderMaterial, pResRenderInfo);
                }
            }
        }
    }

    // 全プログラムのシェーダーキーを表示します。デバッグしたいときに使います。
    void PrintShaderProgramKeys(const nn::g3d::ResShadingModel* pResShadingModel)
    {
        NN_LOG("shading model name: %s\n", pResShadingModel->GetName());
        for (int programIndex = 0; programIndex < pResShadingModel->GetShaderProgramCount(); ++programIndex)
        {
            const int stringCount = 1024;
            char pStaticOptionKey[stringCount];
            char pDynamicOptionKey[stringCount];

            pResShadingModel->PrintDynamicOptionTo(pDynamicOptionKey, sizeof(pDynamicOptionKey), pResShadingModel->GetDynamicKey(programIndex));
            pResShadingModel->PrintStaticOptionTo(pStaticOptionKey, sizeof(pStaticOptionKey), pResShadingModel->GetStaticKey(programIndex));
            NN_LOG("  [%d] \n  dynamic key: %s\n  static option key: %s\n", programIndex, pDynamicOptionKey, pStaticOptionKey);
        }
    }

    // シーンに適用するシーンアニメーション
    nn::g3d::SceneAnimObj* g_pBoundSceneAnimObj = nullptr;
    bool isUnbound = false;
}

void Viewer::ModelFileLoaded(nn::g3d::viewer::ModelFileLoadedOutArg& outArg, const nn::g3d::viewer::ModelFileLoadedArg& arg) NN_NOEXCEPT
{
    // 3DEditor が開いたモデルがリソースとして送られてきます。
    // リソースファイルはアプリケーションが管理する必要あるため、コピーします。
    nn::g3d::ResFile* pResFile = g3ddemo::CreateCopiedResFile(arg.pResFile, arg.fileSize, arg.alignment);
    if (pResFile == nullptr)
    {
        g3ddemo::SendErrorMessageTo3dEditor("Memory allocation failed.");
        return;
    }

    // コピーした ResFile は nn::g3d::viewer に渡す必要があります。
    outArg.pResFile = pResFile;
    g3ddemo::RegisterSamplerToDescriptorPool(pResFile);

    g3ddemo::ResourceHolder* pHolder = GetResourceHolder();
    pHolder->files.PushBack(pResFile);

    // インスタンスを生成
    nn::gfx::Device* pDevice = g3ddemo::GetGfxFramework()->GetDevice();
    for (int modelIndex = 0, modelCount = pResFile->GetModelCount(); modelIndex < modelCount; ++modelIndex)
    {
        nn::g3d::ResModel* pResModel = pResFile->GetModel(modelIndex);
        nns::g3d::RenderModel* pRenderModel = nns::g3d::CreateRenderModel(pDevice, pResModel, DrawPassType_Count);
        pHolder->renderModels.PushBack(pRenderModel);

        nn::g3d::ModelObj* pModelObj = nullptr;
        {
            nn::g3d::ModelObj::Builder builder(pResModel);
            {
                builder.SetBoundingEnabled();
                builder.ShapeBufferingCount(BufferingCount);
                builder.SkeletonBufferingCount(BufferingCount);
                builder.MaterialBufferingCount(BufferingCount);
                builder.ViewCount(ViewType_Count);
            }

            // シェイプアニメーション用の動的頂点バッファーを設定
            pResModel->ActivateDynamicVertexAttrForShapeAnim();

            nns::g3d::ModelAnimObj* pModelAnimObj = nns::g3d::CreateModelAnimObj(pDevice, builder);
            NN_ASSERT_NOT_NULL(pModelAnimObj);
            pHolder->modelAnimObjs.PushBack(pModelAnimObj);
            pModelObj = pModelAnimObj->GetModelObj();
            pModelObj->SetShapeAnimCalculationEnabled();
        }
        NN_ASSERT_NOT_NULL(pModelObj);

        // viewer に ModelObj を登録
        {
            nn::g3d::viewer::ViewerResult result = nn::g3d::viewer::ViewerServer::GetInstance().QueueAttachModelCommand(pModelObj);
            NN_ASSERT(result == nn::g3d::viewer::ViewerResult_Success);
        }

        // RenderModelObj の生成
        nns::g3d::RenderModelObj* pRenderModelObj = nns::g3d::CreateRenderModelObj(pModelObj, pRenderModel);
        pHolder->renderModelObjs.PushBack(pRenderModelObj);
    }
}

void Viewer::ModelFileUnloaded(const nn::g3d::viewer::ModelFileUnloadedArg& arg) NN_NOEXCEPT
{
    // 3DEditor でファイルが閉じられたので、関連するデータを破棄します。
    // GPU が参照中のデータを破棄してしまわないよう、アプリケーションの
    // フレームワークに応じて描画完了待ちを行ったり、描画をスキップするなど、
    // 必要な対策を講じてください。
    // このデモでは GPU による描画が完了してから呼ばれているため、
    // 特に対策は行っていません。

    // このコールバックの外でモデルの遅延破棄処理を行うこともできますが、
    // このサンプルではコールバック内ですぐに破棄を行います。

    nn::g3d::ResFile*           pResFile = arg.pResFile;
    nn::gfx::Device*            pDevice = g3ddemo::GetGfxFramework()->GetDevice();
    g3ddemo::ResourceHolder*    pHolder = GetResourceHolder();

    for (int resModelIndex = 0, resModelCount = pResFile->GetModelCount(); resModelIndex < resModelCount; ++resModelIndex)
    {
        nn::g3d::ResModel* pTargetResModel = pResFile->GetModel(resModelIndex);

        // RenderModelObj を破棄します。
        {
            int renderModelObjIndex = 0;
            while (renderModelObjIndex < pHolder->renderModelObjs.GetCount())
            {
                nns::g3d::RenderModelObj* pRenderModelObj = pHolder->renderModelObjs[renderModelObjIndex];
                const nn::g3d::ResModel*    pResModel = pRenderModelObj->GetModelObj()->GetResource();
                if (pResModel == pTargetResModel)
                {
                    nns::g3d::DestroyRenderModelObj(pRenderModelObj);
                    pHolder->renderModelObjs.Erase(renderModelObjIndex);
                    continue;
                }
                ++renderModelObjIndex;
            }
        }

        // modelAnimObj を破棄します。
        {
            int modelAnimObjIndex = 0;
            while (modelAnimObjIndex < pHolder->modelAnimObjs.GetCount())
            {
                nns::g3d::ModelAnimObj*     pModelAnimObj = pHolder->modelAnimObjs[modelAnimObjIndex];
                const nn::g3d::ResModel*    pResModel = pModelAnimObj->GetModelObj()->GetResource();
                if (pResModel == pTargetResModel)
                {
                    nns::g3d::DestroyModelAnimObj(pDevice, pModelAnimObj);
                    pHolder->modelAnimObjs.Erase(modelAnimObjIndex);
                    continue;
                }
                ++modelAnimObjIndex;
            }
        }

        // RenderModel を破棄します。
        {
            int renderModelIndex = 0;
            while (renderModelIndex < pHolder->renderModels.GetCount())
            {
                nns::g3d::RenderModel*    pRenderModel = pHolder->renderModels[renderModelIndex];
                const nn::g3d::ResModel*    pResModel = pRenderModel->GetResModel();
                if (pResModel == pTargetResModel)
                {
                    nns::g3d::DestroyRenderModel(pDevice, pRenderModel);
                    pHolder->renderModels.Erase(renderModelIndex);
                    continue;
                }
                ++renderModelIndex;
            }
        }

    }

    // リソースファイルを破棄します。
    for (int fileIndex = 0, fileCount = pHolder->files.GetCount(); fileIndex < fileCount; ++fileIndex)
    {
        if (pHolder->files[fileIndex] == pResFile)
        {
            g3ddemo::UnregisterSamplerFromDescriptorPool(pResFile);
            pResFile->Cleanup(pDevice);
            g3ddemo::FreeMemory(pResFile);
            pHolder->files.Erase(fileIndex);
            break;
        }
    }

    // メニューにモデルが破棄されたことを通知
    int selectedModelIndex = GetMenuContext()->GetSelectedIndex(MenuContext::SelectedInfo_Model);
    int modelCount = GetResourceHolder()->modelAnimObjs.GetCount();
    if (modelCount > 1)
    {
        const int envOffset = 1;
        const int count = modelCount - envOffset;
        selectedModelIndex = ((selectedModelIndex - envOffset) + (count - 1)) % count;
        selectedModelIndex += envOffset;
        SetSelectedIndex(MenuContext::SelectedInfo_Model, selectedModelIndex, true);
        return;
    }
    SetSelectedIndex(MenuContext::SelectedInfo_Model, MenuContext::InvalidIndex);
}

// 3DEditor でテクスチャーがロードされたときにコールされます。
void Viewer::TextureFileLoaded(const nn::g3d::viewer::TextureFileLoadedArg& arg) NN_NOEXCEPT
{
    g3ddemo::SetupTexture(g3ddemo::GetGfxFramework()->GetDevice(), arg.pResTextureFile);
}

// 3DEditor でテクスチャーがアンロードされたときにコールされます。
void Viewer::TextureFileUnloaded(const nn::g3d::viewer::TextureFileUnloadedArg& arg) NN_NOEXCEPT
{
    g3ddemo::CleanupTexture(g3ddemo::GetGfxFramework()->GetDevice(), arg.pResTextureFile);
}

void Viewer::ShaderAssignUpdated(const nn::g3d::viewer::ShaderAssignUpdatedArg& arg) NN_NOEXCEPT
{
    nn::gfx::Device*            pDevice = g3ddemo::GetGfxFramework()->GetDevice();
    g3ddemo::ResourceHolder*    pHolder = GetResourceHolder();

    int materialCount = arg.pNewResModel->GetMaterialCount();
    ScopedDynamicArray<nn::g3d::ResShaderArchive*> pResShaderArchiveArray(materialCount);
    for (int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
    {
        nn::g3d::ResMaterial* pOldResMaterial = arg.pOldResModel->GetMaterial(materialIndex);
        nn::g3d::ResMaterial* pNewResMaterial = arg.pNewResModel->GetMaterial(materialIndex);

        // サンプラーの再構築
        {
            for (int samplerIndex = 0, samplerCount = pOldResMaterial->GetSamplerCount(); samplerIndex < samplerCount; ++samplerIndex)
            {
                nn::gfx::Sampler* pSampler = pOldResMaterial->GetSampler(samplerIndex);
                if (!pSampler->GetUserPtr())
                {
                    continue;
                }

                g3ddemo::UnregisterSamplerFromDescriptorPool(pSampler);
            }

            for (int samplerIndex = 0, samplerCount = pNewResMaterial->GetSamplerCount(); samplerIndex < samplerCount; ++samplerIndex)
            {
                nn::gfx::Sampler* pSampler = pNewResMaterial->GetSampler(samplerIndex);
                if (pSampler->GetUserPtr())
                {
                    continue;
                }

                nn::gfx::DescriptorSlot descriptorSlot = g3ddemo::RegisterSamplerToDescriptorPool(pSampler);
                pNewResMaterial->SetSamplerDescriptorSlot(samplerIndex, descriptorSlot);
            }
        }

        // マテリアルに割り当てるシェーダーアーカイブを設定
        if (arg.pMaterialShaderInfos != nullptr &&
            arg.pMaterialShaderInfos[materialIndex].pResShaderArchive != nullptr)
        {
            // 最適化シェーダーが 3DEditor から送られてきた
            pResShaderArchiveArray[materialIndex] = arg.pMaterialShaderInfos[materialIndex].pResShaderArchive;
        }
        else if (pNewResMaterial->GetShaderAssign() != nullptr)
        {
            // 3DEditor でシェーダー最適化が有効になっていない、もしくは 3DEditor の設定により最適化シェーダーの送信がスキップされた
            pResShaderArchiveArray[materialIndex] = FindShaderArchive(
                pHolder,
                pNewResMaterial->GetShaderAssign()->GetShaderArchiveName(),
                pNewResMaterial->GetShaderAssign()->GetShadingModelName());
        }
    }

    // モデルリソースの構造の変化に伴い RenderModel の再構築を行います。
    for (nns::g3d::RenderModel* pRenderModel : pHolder->renderModels)
    {
        if (arg.pOldResModel != pRenderModel->GetResModel())
        {
            continue;
        }

        // RenderModel の再構築
        {
            pRenderModel->Finalize(pDevice);
            pRenderModel->Initialize(pDevice, arg.pNewResModel, DrawPassType_Count);

            // マテリアルパス
            InitializeMaterialPass(pDevice, pRenderModel, pResShaderArchiveArray);

            // デバッグ表示用パス
            {
                nn::g3d::ResShaderArchive* pResShaderArchive = pHolder->shaderFiles[ShaderArchiveType_VisualizeInfo]->GetResShaderArchive();
                nn::g3d::ResShadingModel* pResShadingModel = pResShaderArchive->FindShadingModel(g_ShadingModelNames[ShaderArchiveType_VisualizeInfo]);
                NN_ASSERT_NOT_NULL(pResShadingModel);
                pRenderModel->CreateDrawPass(pDevice, DrawPassType_Visualize, pResShadingModel);
                for (int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
                {
                    nns::g3d::RenderMaterial* pRenderMaterial = pRenderModel->GetRenderMaterial(DrawPassType_Visualize, materialIndex);
                    pRenderMaterial->SetRasterizerState(&g_VisializeRasterizerState);
                }
            }
        }
    }

    // ShaderScratchMemory のサイズを再計算します。
    {
        ScopedDynamicArray<const nn::g3d::ResShaderArchive*> pTempShaderArchiveArray(materialCount + pHolder->shaderFiles.GetCount());
        for (int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
        {
            pTempShaderArchiveArray[materialIndex] = pResShaderArchiveArray[materialIndex];
        }
        for (int shaderArchiveIndex = 0, shaderArchiveCount = pHolder->shaderFiles.GetCount(); shaderArchiveIndex < shaderArchiveCount; ++shaderArchiveIndex)
        {
            const int offset = materialCount;
            pTempShaderArchiveArray[shaderArchiveIndex + offset] = pHolder->shaderFiles[shaderArchiveIndex]->GetResShaderArchive();
        }
        g3ddemo::SetShaderScratchMemory(pTempShaderArchiveArray, pTempShaderArchiveArray.GetCount());
    }
}

void Viewer::MaterialUpdated(const nn::g3d::viewer::MaterialUpdatedArg& arg) NN_NOEXCEPT
{
    // ShaderAssignUpdated() で行われた変更に伴って ModelObj に変更が必要な場合は
    // このコールバックで処理を行います。
    // 編集のために使用するヒープを区別したい場合は arg.state で状態を判定します。
    // ShaderAssignUpdated() で初期化したリソースに従って MaterialObj や ShapeObj が
    // ビューアーライブラリーによって作り直されているので、必要な初期化を行います。
    nn::g3d::ModelObj* pModelObj = arg.pModelObj;
    g3ddemo::ResourceHolder* pHolder = GetResourceHolder();

    // RenderModel の変更に追従するため RenderModelObj を再構築します。
    nns::g3d::RenderModelObj* pRenderModelObj = FindRenderModelObjFromHolder(pHolder, pModelObj);
    NN_ASSERT_NOT_NULL(pRenderModelObj);

    nns::g3d::RenderModel* pRenderModel = pRenderModelObj->GetRenderModel();
    pRenderModelObj->Finalize();
    pRenderModelObj->Initialize(pModelObj, pRenderModel);

    // RenderInfo に設定されているパラメーターを設定 (とりあえず、半透明の描画順などは気にしません)
    SetupRenderStateFromRenderInfo(pRenderModelObj);

    // 各種割り当て
    {
        // シェーダーキーの書き込み
        {
            g3ddemo::WriteSkinningOption(pRenderModelObj);
            WriteVisializeInfoKey(pRenderModelObj);
        }

        // ダミーテクスチャ
        {
            nn::g3d::TextureRef dummyTextureRef = g3ddemo::GetDummyTextureRef();
            nn::gfx::DescriptorSlot dummySamplerDescriptorSlot = g3ddemo::GetDummySamplerDescriptorSlot();
            pRenderModelObj->BindDummySampler(dummyTextureRef.GetDescriptorSlot(), dummySamplerDescriptorSlot);
        }

        // town シェーダー用 環境テクスチャ
        {
            nn::g3d::ModelObj* pTownEnvModel = pHolder->modelAnimObjs[ResFileType_TownEnv]->GetModelObj();
            NN_ASSERT(strcmp(pTownEnvModel->GetName(), "env") == 0);

            nn::g3d::MaterialObj* pTownEnvMaterial = pTownEnvModel->GetMaterial(0);
            const nn::g3d::ResShaderAssign* pResShaderAssign = pTownEnvMaterial->GetResource()->GetShaderAssign();
            for (int samplerIndex = 0, samplerCount = pTownEnvMaterial->GetSamplerCount(); samplerIndex < samplerCount; ++samplerIndex)
            {
                const char* samplerId = pResShaderAssign->GetSamplerAssignId(samplerIndex);
                nn::g3d::SamplerRef samplerRef = pTownEnvMaterial->GetSampler(samplerIndex);
                nn::g3d::TextureRef textureRef = pTownEnvMaterial->GetTexture(samplerIndex);
                if (!samplerRef.IsValid() || !textureRef.IsValid())
                {
                    continue;
                }
                pRenderModelObj->BindSampler(samplerId, textureRef.GetDescriptorSlot(), samplerRef.GetDescriptorSlot());
            }
        }

        // ユニフォームブロック
        {
            for (int uniformIdIndex = 0; uniformIdIndex < NN_ARRAY_SIZE(g_UniformBlockIdArray); ++uniformIdIndex)
            {
                const nn::gfx::Buffer** uniformBlock = &g_pUniformBlocks[uniformIdIndex * BufferingCount];
                pRenderModelObj->BindUniformBlock(g_UniformBlockIdArray[uniformIdIndex], uniformBlock, BufferingCount);
            }
        }
    }
}

void Viewer::ShapeUpdated(const nn::g3d::viewer::ShapeUpdatedArg& arg) NN_NOEXCEPT
{
    nns::g3d::ModelAnimObj* pModelAnimObj = FindModelAnimObjFromHolder(GetResourceHolder(), arg.pModelObj);
    NN_ASSERT_NOT_NULL(pModelAnimObj);

    // シェイプが更新されているので計算します。
    pModelAnimObj->Calculate();
}

void Viewer::BoneBindUpdated(const nn::g3d::viewer::BoneBindUpdatedArg& arg) NN_NOEXCEPT
{
    // 3DEditor でプレビュー配置のバインド設定が行われた際にモデルに通知を行います。
    // 配置自体はここで行わずにアプリケーションのバインド機能によってバインドします。
    nns::g3d::ModelAnimObj* pModelAnimObj = FindModelAnimObjFromHolder(GetResourceHolder(), arg.pModelObj);
    pModelAnimObj->BindTo(arg.pParentModelObj, arg.parentBoneIndex);

    // バインド後 位置と回転の情報を初期化します。
    pModelAnimObj->SetTranslate(nn::util::Vector3f(.0f, .0f, .0f));
    pModelAnimObj->SetRotate(nn::util::Vector3f(.0f, .0f, .0f));

    using ViewerServer = nn::g3d::viewer::ViewerServer;
    ViewerServer::LayoutModelArg layoutModelArg(
        arg.pModelObj,
        pModelAnimObj->GetScale(),
        pModelAnimObj->GetRotate(),
        pModelAnimObj->GetTranslate()
    );
    ViewerServer::GetInstance().QueueLayoutModelCommand(layoutModelArg);
}

void Viewer::SamplerParamUpdated(const nn::g3d::viewer::SamplerParamUpdatedArg& arg) NN_NOEXCEPT
{
    // サンプラーが更新されたのでディスクリプタプールへ再登録します。
    g3ddemo::ReregisterSamplerToDescriptorPool(arg.pModelObj);
}

void Viewer::ModelLayoutUpdated(const nn::g3d::viewer::ModelLayoutUpdatedArg& arg) NN_NOEXCEPT
{
    nns::g3d::ModelAnimObj* pModelAnimObj = FindModelAnimObjFromHolder(GetResourceHolder(), arg.pModelObj);
    pModelAnimObj->SetScale(arg.scale);
    pModelAnimObj->SetRotate(arg.rotate);
    pModelAnimObj->SetTranslate(arg.translate);
}

void Viewer::SendModelLayoutRequested(
    nn::g3d::viewer::SendModelLayoutRequestedOutArg& outputData,
    const nn::g3d::viewer::SendModelLayoutRequestedArg& arg) NN_NOEXCEPT
{
    const nns::g3d::ModelAnimObj* pModelAnimObj = FindModelAnimObjFromHolder(GetResourceHolder(), arg.pModelObj);
    outputData.scale = pModelAnimObj->GetScale();
    outputData.rotate = pModelAnimObj->GetRotate();
    outputData.translate = pModelAnimObj->GetTranslate();
}

void Viewer::ModelAttached(const nn::g3d::viewer::ModelAttachedArg& arg) NN_NOEXCEPT
{
    int modelIndex = GetModelAnimObjIndex(GetResourceHolder(), arg.pModelObj);
    NN_ASSERT_NOT_EQUAL(modelIndex, -1);
    SetSelectedIndex(MenuContext::SelectedInfo_Model, modelIndex);
}

void Viewer::ModelDetached(const nn::g3d::viewer::ModelDetachedArg& arg) NN_NOEXCEPT
{
    // ボーンバインドを解除
    {
        for (nns::g3d::ModelAnimObj* pModelAnimObj : GetResourceHolder()->modelAnimObjs)
        {
            if (pModelAnimObj->GetBindModelObj() != arg.pModelObj)
            {
                continue;
            }
            pModelAnimObj->BindTo(nullptr, -1);
        }
    }
}

void Viewer::SceneAnimBound(const nn::g3d::viewer::SceneAnimBoundArg& arg) NN_NOEXCEPT
{
    // isUnbound が有効になっているケースは以下のどちらかの状態です。
    // - SceneAnimUnbound() が呼び出されたあとの呼び出し (Reload 状態)
    // - 初回ロード時やプレビューの設定を無効から有効に変更した
    //
    // 現在、バインド済みのアニメーションが存在する場合に、バインド済みのアニメーション以外がリロードされた場合には、
    // 再割り当てをスキップします。
    //
    bool isBound = arg.cameraAnimObjCount > 0 && !isUnbound || !g_pBoundSceneAnimObj;
    if (isBound)
    {
        g_pBoundSceneAnimObj = arg.pCameraAnimObjs[arg.cameraAnimObjCount - 1];
    }

    isUnbound = false;
}

void Viewer::SceneAnimUnbound(const nn::g3d::viewer::SceneAnimUnboundArg& arg) NN_NOEXCEPT
{
    for (int cameraAnimIndex = 0; cameraAnimIndex < arg.cameraAnimObjCount; ++cameraAnimIndex)
    {
        if (g_pBoundSceneAnimObj != arg.pCameraAnimObjs[cameraAnimIndex])
        {
            continue;
        }

        g_pBoundSceneAnimObj = nullptr;

        // プロジェクション行列をリセット
        SetDefaultProjection();
    }

    isUnbound = true;
}

void Viewer::ApplySceneAnimRequested(const nn::g3d::viewer::ApplySceneAnimRequestedArg& arg) NN_NOEXCEPT
{
    // カメラアニメーション
    {
        g3ddemo::CameraController* pCameraController = GetCameraController();

        // 更新を行う前にプロジェクション行列をリセット
        SetDefaultProjection();

        for (int cameraAnimIndex = 0, cameraAnimCount = arg.cameraAnimObjCount; cameraAnimIndex < cameraAnimCount; ++cameraAnimIndex)
        {
            const nn::g3d::CameraAnimObj* pCameraAnimObj = arg.pCameraAnimObjs[cameraAnimIndex];
            if (pCameraAnimObj != g_pBoundSceneAnimObj)
            {
                continue;
            }

            const nn::g3d::CameraAnimResult* pCameraAnimResult = pCameraAnimObj->GetResult();
            const nn::g3d::CameraAnimResult::View& view = pCameraAnimResult->view;
            pCameraController->SetPos(view.pos[0], view.pos[1], view.pos[2]);

            nn::Bit32 rotateMode = pCameraAnimObj->GetRotateMode();
            // カメラの視線を注視点から計算します。
            if (rotateMode == nn::g3d::ResCameraAnim::Flag_RotAim)
            {
                pCameraController->SetLookAtPos(view.aim[0], view.aim[1], view.aim[2]);
                pCameraController->SetTwist(view.twist);
            }
            // カメラの視線を回転角から計算します。
            else if (rotateMode == nn::g3d::ResCameraAnim::Flag_RotEulerZxy)
            {
                pCameraController->SetTwist(view.rotate[2]);
                pCameraController->SetAngle(view.rotate[0], view.rotate[1]);
            }

            nn::util::Matrix4x4fType projectionMatrix;
            const nn::g3d::CameraAnimResult::Projection& projection = pCameraAnimResult->proj;
            nn::Bit32 projectionMode = pCameraAnimObj->GetProjectionMode();
            if (projectionMode == nn::g3d::ResCameraAnim::Flag_ProjPersp)
            {
                nn::util::MatrixPerspectiveFieldOfViewRightHanded(
                    &projectionMatrix,
                    projection.fovy,
                    projection.aspect,
                    projection.nearZ,
                    projection.farZ);
            }
            else if (projectionMode == nn::g3d::ResCameraAnim::Flag_ProjOrtho)
            {
                float height = projection.height * 0.5f;
                float width = projection.height * projection.aspect * 0.5f;
                nn::util::MatrixOrthographicOffCenterRightHanded(
                    &projectionMatrix,
                    -width,
                    width,
                    -height,
                    height,
                    projection.nearZ,
                    projection.farZ
                );
            }
            SetProjectionMatrix(projectionMatrix);
        }
    }
}

void Viewer::TargetSelected(const nn::g3d::viewer::TargetSelectedArg& arg) NN_NOEXCEPT
{
    int selectedModelIndex = GetModelAnimObjIndex(GetResourceHolder(), arg.pModelObj);
    SetSelectedIndex(MenuContext::SelectedInfo_Model, selectedModelIndex);

    // 初回に選択されたもののみを選択します。
    using Kind = nn::g3d::viewer::TargetSelectedArg::TargetKind;
    switch (arg.targetKind)
    {
    case Kind::TargetKind_Shape:
        SetSelectedIndex(MenuContext::SelectedInfo_Shape, arg.index[0]);
        SelectInfo(MenuContext::SelectedInfo_Shape);
        break;
    case Kind::TargetKind_Bone:
        SetSelectedIndex(MenuContext::SelectedInfo_Bone, arg.index[0]);
        SelectInfo(MenuContext::SelectedInfo_Bone);
        break;
    case Kind::TargetKind_Model:
        SelectInfo(MenuContext::SelectedInfo_Model);
    default:
        break;
    }
}

void Viewer::Initialize(g3ddemo::ResourceHolder& resourceHolder) NN_NOEXCEPT
{
    g3ddemo::DemoViewerBase::Initialize(resourceHolder);
    StartPollThread();
    Open();
}

void Viewer::Finalize() NN_NOEXCEPT
{
    Close();
    StopPollThread();
    g3ddemo::DemoViewerBase::Finalize();
}

void Viewer::Update() NN_NOEXCEPT
{
    nn::g3d::viewer::ViewerServer::GetInstance().ExecuteCommands();
}

void SetFallBackShadingModel(nn::g3d::ResShadingModel* pResShadingModel) NN_NOEXCEPT
{
    g_pFallbackShadingModel = pResShadingModel;
}

void InitializeRenderState(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetDepthBias(-1);
    info.SetCullMode(nn::gfx::CullMode_None);
    g_VisializeRasterizerState.Initialize(pDevice, info);
}

void FinalizeRenderState(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    g_VisializeRasterizerState.Finalize(pDevice);
}

void SetUniformBlock(UniformBlockType type, const nn::gfx::Buffer** pBufferArray, int arraySize) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(type, UniformBlockType_Env, UniformBlockType_Count);
    NN_ASSERT_EQUAL(arraySize, BufferingCount);
    NN_UNUSED(arraySize);

    g_pUniformBlocks[static_cast<int>(type) * BufferingCount] = pBufferArray[0];
    g_pUniformBlocks[static_cast<int>(type) * BufferingCount + 1] = pBufferArray[1];
}
