﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "3dSampleViewer_DrawParam.h"

//---------------------------------------------------------------------------
//! @brief  基底クラスです。
//---------------------------------------------------------------------------
class Grid
{
public:
    //---------------------------------------------------------------------------
    //! @brief  描画メソッドです。
    //---------------------------------------------------------------------------
    virtual void ProcDraw( const DrawParam& param )
    {
        NN_UNUSED( param );
    }
};

//---------------------------------------------------------------------------
//! @brief  キューブグリッドクラスです。
//---------------------------------------------------------------------------
class CubeGrid : public Grid
{
public:
    //---------------------------------------------------------------------------
    //! @brief  標準背景描画メソッドです。
    //---------------------------------------------------------------------------
    virtual void ProcDraw( const DrawParam& drawParam ) NN_OVERRIDE
    {
        // Blend
        drawParam.m_pCommandBuffer->SetBlendState( drawParam.m_pPrimitiveRenderer->GetBlendState( nns::gfx::PrimitiveRenderer::BlendType::BlendType_Normal ) );

        // Depth Enable
        drawParam.m_pPrimitiveRenderer->SetDepthStencilState( drawParam.m_pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest );

        drawParam.m_pPrimitiveRenderer->SetProjectionMatrix( &drawParam.m_ProjMtx );
        drawParam.m_pPrimitiveRenderer->SetViewMatrix( &drawParam.m_ViewMtx );
        nn::util::Matrix4x3fType modelMatrix;
        nn::util::MatrixIdentity(&modelMatrix);
        drawParam.m_pPrimitiveRenderer->SetModelMatrix( &modelMatrix );

        const int gridCount = 5;
        const float gridWidth = 10.0f;
        int cnt = 0;
        for ( int i = -gridCount; i < gridCount; i++ )
        {
            for ( int j = -gridCount; j < gridCount; j++ )
            {
                nn::util::Vector3fType center = { i * gridWidth + gridWidth * 0.5f, 0.0f, j * gridWidth + gridWidth * 0.5f };
                nn::util::Vector3fType size   = { gridWidth, 0.1f, gridWidth };

                nn::util::Uint8x4 color = { { 32, 32, 32, 255 } };
                if ( cnt % 2 == 0 )
                {
                    color.v[0] = 64;
                    color.v[1] = 64;
                    color.v[2] = 64;
                }
                drawParam.m_pPrimitiveRenderer->SetColor(color);
                drawParam.m_pPrimitiveRenderer->DrawCube(drawParam.m_pCommandBuffer,
                    nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, size);
                cnt++;
            }
            cnt++;
        }
    }
};

//---------------------------------------------------------------------------
//! @brief  標準グリッド描画クラスです。
//---------------------------------------------------------------------------
class DefaultGrid : public Grid
{
public:
    //---------------------------------------------------------------------------
    //! @brief  標準グリッド描画メソッドです。
    //---------------------------------------------------------------------------
    virtual void ProcDraw( const DrawParam& drawParam ) NN_OVERRIDE
    {
        // Blend
        drawParam.m_pCommandBuffer->SetBlendState( drawParam.m_pPrimitiveRenderer->GetBlendState( nns::gfx::PrimitiveRenderer::BlendType::BlendType_Normal ) );

        // Depth Enable
        drawParam.m_pPrimitiveRenderer->SetDepthStencilState( drawParam.m_pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest );

        drawParam.m_pPrimitiveRenderer->SetProjectionMatrix( &drawParam.m_ProjMtx );
        drawParam.m_pPrimitiveRenderer->SetViewMatrix( &drawParam.m_ViewMtx );
        nn::util::Matrix4x3fType modelMatrix;
        nn::util::MatrixIdentity(&modelMatrix);

        drawParam.m_pPrimitiveRenderer->SetModelMatrix( &modelMatrix );
        {
            nn::util::Uint8x4 color = { { 255, 255, 255, 255 } };
            nn::util::Uint8x4 red = { { 255, 0, 0, 255 } };
            nn::util::Uint8x4 blue = { { 0, 0, 255, 255 } };
            nn::util::Uint8x4 green = { { 0, 255, 0, 255 } };

            drawParam.m_pPrimitiveRenderer->SetLineWidth(1.0f);
            drawParam.m_pPrimitiveRenderer->SetColor(color);

            const int gridCount = 101;
            const float gridWidth = 100.0f;
            const float maxGridWidth = 5000.0f;
            float interval = -maxGridWidth;
            for (int i = 0; i < gridCount; i++)
            {
                {
                    nn::util::Vector3fType from;
                    nn::util::Vector3fType to;
                    nn::util::VectorSet(&from, -maxGridWidth, 0.0f, interval);
                    nn::util::VectorSet(&to, maxGridWidth, 0.0f, interval);
                    drawParam.m_pPrimitiveRenderer->DrawLine(drawParam.m_pCommandBuffer, from, to);
                    nn::util::VectorSet(&from, interval, 0.0f, -maxGridWidth);
                    nn::util::VectorSet(&to, interval, 0.0f, maxGridWidth);
                    drawParam.m_pPrimitiveRenderer->DrawLine(drawParam.m_pCommandBuffer, from, to);
                }
                interval += gridWidth;
            }

            nn::util::Vector3fType zeroVector;
            nn::util::VectorSet(&zeroVector, 0.0f, 0.0f, 0.0f);
            nn::util::Vector3fType axisPos;
            nn::util::VectorSet(&axisPos, 10.0f, 0.0f, 0.0f);
            drawParam.m_pPrimitiveRenderer->SetLineWidth(4.0f);
            drawParam.m_pPrimitiveRenderer->SetColor(red);
            drawParam.m_pPrimitiveRenderer->DrawLine(drawParam.m_pCommandBuffer, zeroVector, axisPos );
            drawParam.m_pPrimitiveRenderer->SetLineWidth(4.0f);
            nn::util::VectorSet(&axisPos, 0.0f, 0.0f, 10.0f);
            drawParam.m_pPrimitiveRenderer->SetColor(blue);
            drawParam.m_pPrimitiveRenderer->DrawLine(drawParam.m_pCommandBuffer, zeroVector, axisPos );
            drawParam.m_pPrimitiveRenderer->SetLineWidth(4.0f);
            nn::util::VectorSet(&axisPos, 0.0f, 10.0f, 0.0f);
            drawParam.m_pPrimitiveRenderer->SetColor(green);
            drawParam.m_pPrimitiveRenderer->DrawLine(drawParam.m_pCommandBuffer, zeroVector, axisPos );
        }
    }
};

//---------------------------------------------------------------------------
//! @brief  高詳細グリッド描画クラスです。
//---------------------------------------------------------------------------
class HighGrid : public Grid
{
public:
    //---------------------------------------------------------------------------
    //! @brief  標準グリッド描画メソッドです。
    //---------------------------------------------------------------------------
    virtual void ProcDraw( const DrawParam& drawParam ) NN_OVERRIDE
    {
         // Blend
        drawParam.m_pCommandBuffer->SetBlendState( drawParam.m_pPrimitiveRenderer->GetBlendState( nns::gfx::PrimitiveRenderer::BlendType::BlendType_Normal ) );

        // Depth Enable
        drawParam.m_pPrimitiveRenderer->SetDepthStencilState( drawParam.m_pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest );

        drawParam.m_pPrimitiveRenderer->SetProjectionMatrix( &drawParam.m_ProjMtx );
        drawParam.m_pPrimitiveRenderer->SetViewMatrix( &drawParam.m_ViewMtx );
        nn::util::Matrix4x3fType modelMatrix;
        nn::util::MatrixIdentity(&modelMatrix);

        drawParam.m_pPrimitiveRenderer->SetModelMatrix( &modelMatrix );
        {
            nn::util::Uint8x4 color = { { 255, 255, 255, 255 } };
            nn::util::Uint8x4 red = { { 255, 0, 0, 255 } };
            nn::util::Uint8x4 blue = { { 0, 0, 255, 255 } };
            nn::util::Uint8x4 green = { { 0, 255, 0, 255 } };

            drawParam.m_pPrimitiveRenderer->SetLineWidth(1.0f);
            drawParam.m_pPrimitiveRenderer->SetColor(color);

            const int gridCount = 101;
            const float gridWidth = 1.0f;
            const float maxGridWidth = 50.0f;
            float interval = -maxGridWidth;
            for (int i = 0; i < gridCount; i++)
            {
                {
                    nn::util::Vector3fType from;
                    nn::util::Vector3fType to;
                    nn::util::VectorSet(&from, -maxGridWidth, 0.0f, interval);
                    nn::util::VectorSet(&to, maxGridWidth, 0.0f, interval);
                    drawParam.m_pPrimitiveRenderer->DrawLine(drawParam.m_pCommandBuffer, from, to);
                    nn::util::VectorSet(&from, interval, 0.0f, -maxGridWidth);
                    nn::util::VectorSet(&to, interval, 0.0f, maxGridWidth);
                    drawParam.m_pPrimitiveRenderer->DrawLine(drawParam.m_pCommandBuffer, from, to);
                }
                interval += gridWidth;
            }

            nn::util::Vector3fType zeroVector;
            nn::util::VectorSet(&zeroVector, 0.0f, 0.0f, 0.0f);
            nn::util::Vector3fType axisPos;
            nn::util::VectorSet(&axisPos, 10.0f, 0.0f, 0.0f);
            drawParam.m_pPrimitiveRenderer->SetLineWidth(4.0f);
            drawParam.m_pPrimitiveRenderer->SetColor(red);
            drawParam.m_pPrimitiveRenderer->DrawLine(drawParam.m_pCommandBuffer, zeroVector, axisPos);
            nn::util::VectorSet(&axisPos, 0.0f, 0.0f, 10.0f);
            drawParam.m_pPrimitiveRenderer->SetLineWidth(4.0f);
            drawParam.m_pPrimitiveRenderer->SetColor(blue);
            drawParam.m_pPrimitiveRenderer->DrawLine(drawParam.m_pCommandBuffer, zeroVector, axisPos);
            nn::util::VectorSet(&axisPos, 0.0f, 10.0f, 0.0f);
            drawParam.m_pPrimitiveRenderer->SetLineWidth(4.0f);
            drawParam.m_pPrimitiveRenderer->SetColor(green);
            drawParam.m_pPrimitiveRenderer->DrawLine(drawParam.m_pCommandBuffer, zeroVector, axisPos);
        }
    }
};
